package ru.yandex.travel.orders.services.cpa;

import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.protobuf.Timestamp;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.cpa.ECpaItemState;
import ru.yandex.travel.orders.cpa.ECpaOrderStatus;
import ru.yandex.travel.orders.cpa.TBusExtraData;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.cpa.TSuburbanExtraData;
import ru.yandex.travel.orders.cpa.TTrainExtraData;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderLabelParams;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.grpc.CpaProperties;
import ru.yandex.travel.orders.proto.TOrderRefund;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.cpa.CpaOrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.orders.services.buses.BusPartnersService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.train.model.TrainReservation;

import static java.util.stream.Collectors.toList;

@Component
public class GenericCpaSnapshotProvider extends AbstractCpaSnapshotProvider {
    private static final Map<EOrderItemState, ECpaItemState> GENERIC_ITEM_STATE_MAP = Map.of(
            EOrderItemState.IS_NEW, ECpaItemState.IS_NEW,
            EOrderItemState.IS_RESERVED, ECpaItemState.IS_RESERVED,
            EOrderItemState.IS_CANCELLED, ECpaItemState.IS_CANCELLED,
            EOrderItemState.IS_CONFIRMED, ECpaItemState.IS_CONFIRMED,
            EOrderItemState.IS_REFUNDED, ECpaItemState.IS_REFUNDED
    );

    private static final Map<EOrderState, ECpaOrderStatus> GENERIC_ORDER_STATE_MAP = Map.of(
            EOrderState.OS_NEW, ECpaOrderStatus.OS_UNPAID,
            EOrderState.OS_WAITING_RESERVATION, ECpaOrderStatus.OS_UNPAID,
            EOrderState.OS_RESERVED, ECpaOrderStatus.OS_UNPAID,
            EOrderState.OS_WAITING_PAYMENT, ECpaOrderStatus.OS_UNPAID,
            EOrderState.OS_WAITING_CONFIRMATION, ECpaOrderStatus.OS_PAID,
            EOrderState.OS_CONFIRMED, ECpaOrderStatus.OS_CONFIRMED,
            EOrderState.OS_REFUNDED, ECpaOrderStatus.OS_REFUNDED,
            EOrderState.OS_WAITING_CANCELLATION, ECpaOrderStatus.OS_CANCELLED,
            EOrderState.OS_CANCELLED, ECpaOrderStatus.OS_CANCELLED
    );

    private final OrderInfoMapper orderInfoMapper;
    private final AuthorizedUserRepository authorizedUserRepository;
    private final CpaOrderRepository cpaOrderRepository;
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;
    private final BusPartnersService busPartnersService;

    public GenericCpaSnapshotProvider(CpaProperties cpaProperties,
                                      OrderInfoMapper orderInfoMapper,
                                      CpaOrderRepository cpaOrderRepository,
                                      AuthorizedUserRepository authorizedUserRepository,
                                      SuburbanOrderItemEnvProvider suburbanEnvProvider,
                                      AuthorizationService authService,
                                      BusPartnersService busPartnersService
    ) {
        super(cpaProperties, authService);
        this.cpaOrderRepository = cpaOrderRepository;
        this.orderInfoMapper = orderInfoMapper;
        this.authorizedUserRepository = authorizedUserRepository;
        this.suburbanEnvProvider = suburbanEnvProvider;
        this.busPartnersService = busPartnersService;
    }

    @Override
    protected boolean supports(EOrderType orderType) {
        return orderType == EOrderType.OT_TRAIN || orderType == EOrderType.OT_GENERIC;
    }

    @Override
    public boolean supports(EOrderType orderType, EServiceType serviceType) {
        if (orderType == EOrderType.OT_TRAIN) {
            return true;
        }
        if (orderType == EOrderType.OT_GENERIC) {
            return serviceType == EServiceType.PT_SUBURBAN ||
                    serviceType == EServiceType.PT_BUS;
        }
        return false;
    }

    @Override
    public List<TOrderSnapshot> getSnapshotsV2(TListSnapshotsReqV2 request) {
        switch (request.getOrderType()) {
            case OT_TRAIN:
                return loadTrainOrderSnapshotsWithLimit(request);
            case OT_GENERIC:
                if (request.getServiceType() == EServiceType.PT_SUBURBAN) {
                    return loadSuburbanOrderSnapshots(request);
                } else if (request.getServiceType() == EServiceType.PT_BUS) {
                    return loadBusOrderSnapshots(request);
                }
            default:
                throw new IllegalArgumentException();
        }
    }

    private List<TOrderSnapshot> loadSuburbanOrderSnapshots(TListSnapshotsReqV2 request) {
        List<Order> orders = getOrdersBySnapshotsRequest(request, Set.of(EDisplayOrderType.DT_SUBURBAN));
        return orders.stream().map(order -> toSuburbanSnapshot(order, request.getAddPersonalData())).collect(toList());
    }

    private List<TOrderSnapshot> loadBusOrderSnapshots(TListSnapshotsReqV2 request) {
        List<Order> orders = getOrdersBySnapshotsRequest(request, Set.of(EDisplayOrderType.DT_BUS));
        List<TOrderSnapshot.Builder> snapshots = orders.stream()
                .flatMap(o -> OrderCompatibilityUtils.getBusOrderItems(o).stream().map(i -> toBusSnapshot(o, i, request.getAddPersonalData())))
                .collect(toList());
        return snapshots.stream().map(TOrderSnapshot.Builder::build).collect(toList());
    }

    private List<TOrderSnapshot> loadTrainOrderSnapshotsWithLimit(TListSnapshotsReqV2 request) {
        List<Order> orders = getOrdersBySnapshotsRequest(request, Set.of(EDisplayOrderType.DT_TRAIN));

        List<TOrderSnapshot.Builder> snapshots = orders.stream()
                .flatMap(o -> OrderCompatibilityUtils.getTrainOrderItems(o).stream().map(i -> toTrainSnapshot(o, i, request.getAddPersonalData())))
                .collect(toList());

        // TODO(mbobrov, ganintsev) remove this inefficiency
        Set<UUID> orderIds = snapshots.stream().map(x -> UUID.fromString(x.getTravelOrderId()))
                .collect(Collectors.toSet());

        List<AuthorizedUser> users = authorizedUserRepository.findByIdOrderIdInAndRole(orderIds,
                AuthorizedUser.OrderUserRole.OWNER);

        Map<UUID, AuthorizedUser> orderToUsers = new HashMap<>();
        users.forEach(u -> orderToUsers.put(u.getOrderId(), u));

        return snapshots.stream().map(s -> {
            TUserInfo.Builder userInfo = s.getTrainExtraData().getOwner().toBuilder();
            AuthorizedUser owner = orderToUsers.get(UUID.fromString(s.getTravelOrderId()));

            if (owner != null) {
                userInfo.setLogin(Strings.nullToEmpty(owner.getLogin()));
                userInfo.setYandexUid(Strings.nullToEmpty(owner.getYandexUid()));
                userInfo.setPassportId(Strings.nullToEmpty(owner.getPassportId()));
            }
            TTrainExtraData trainExtraData = s.getTrainExtraData().toBuilder().setOwner(userInfo).build();
            s.setTrainExtraData(trainExtraData);
            return s.build();
        }).collect(toList());
    }

    private TOrderSnapshot.Builder toBusSnapshot(Order order, BusOrderItem orderItem, boolean addPersonalData) {
        BusReservation payload = orderItem.getPayload();
        if (payload.getRide().getBusPartner() == null) {
            try {
                ObjectMapper objMapper = ProtoUtils.createObjectMapper();
                payload = objMapper.readerFor(payload.getClass()).readValue(objMapper.writeValueAsString(payload));
            } catch (JsonProcessingException e) {
                throw new RuntimeException("Can't copy bus payload", e);
            }
            payload.getRide().setBusPartner(busPartnersService.getPartnerById(payload.getRide().getSupplierId()));
        }
        ECpaOrderStatus status = getCpaOrderStatus(order);
        var extraBuilder = TBusExtraData.newBuilder();
        List<TOrderRefund> refunds = order.getOrderRefunds().stream()
                .flatMap(r -> orderInfoMapper.createBusOrderRefunds(r).stream()).collect(toList());
        extraBuilder
                .setOrderItemState(orderItem.getState())
                .setPayload(ProtoUtils.toTJson(payload))
                .addAllRefund(refunds)
                .setOwner(orderInfoMapper.buildUserInfoFromOrder(order));

        var invoice = order.getCurrentInvoice();
        if (invoice != null) {
            extraBuilder.setCurrentInvoice(orderInfoMapper.createInvoiceBuilder(invoice));
            if (invoice.getTrustPaymentTs() != null) {
                extraBuilder.setPaymentTs(ProtoUtils.fromInstant(invoice.getTrustPaymentTs()));
            }
        }
        if (orderItem.getConfirmedAt() != null) {
            extraBuilder.setConfirmedAt(ProtoUtils.fromInstant(orderItem.getConfirmedAt()));
        }

        return mapBasicOrderInfo(order, orderItem.getItemNumber(), addPersonalData)
                .setBusExtraData(extraBuilder.build())
                .setAmount(ProtoUtils.toTPrice(orderItem.totalCostAfterReservation()))
                .setItemState(getCpaItemState(orderItem))
                .setCpaOrderStatus(status);
    }

    private TOrderSnapshot.Builder toTrainSnapshot(Order order, TrainOrderItem orderItem, boolean addPersonalData) {
        TrainReservation payload = orderItem.getPayload();
        ECpaOrderStatus status = getCpaOrderStatus(order);
        var extraBuilder = TTrainExtraData.newBuilder();
        List<TOrderRefund> refunds = order.getOrderRefunds().stream()
                .flatMap(r -> orderInfoMapper.createTrainOrderRefunds(r).stream()).collect(toList());
        extraBuilder
                .setOrderItemState(orderItem.getState())
                .setOrderWorkflowState(order.getWorkflow().getState())
                .setPayload(ProtoUtils.toTJson(payload))
                .setInvoiceCount(order.getInvoices().size())
                .setOwner(orderInfoMapper.buildUserInfoFromOrder(order))
                .setRebookingEnabled(order.isTrainRebookingEnabled())
                .addAllRefund(refunds);
        var invoice = order.getCurrentInvoice();
        if (invoice != null) {
            extraBuilder.setCurrentInvoice(orderInfoMapper.createInvoiceBuilder(invoice));
            if (invoice.getTrustPaymentTs() != null) {
                extraBuilder.setPaymentTs(ProtoUtils.fromInstant(invoice.getTrustPaymentTs()));
            }
        }
        OrderLabelParams labelParams = order.getOrderLabelParams();
        if (labelParams != null && labelParams.getPayload() != null) {
            extraBuilder.setLabelParams(ProtoUtils.toTJson(labelParams.getPayload()));
        }
        if (orderItem.getConfirmedAt() != null) {
            extraBuilder.setConfirmedAt(ProtoUtils.fromInstant(orderItem.getConfirmedAt()));
        }

        return mapBasicOrderInfo(order, orderItem.getItemNumber(), addPersonalData)
                .setTrainExtraData(extraBuilder.build())
                .setAmount(ProtoUtils.toTPrice(orderItem.totalCostAfterReservation()))
                .setItemState(getCpaItemState(orderItem))
                .setCpaOrderStatus(status);
    }

    @Override
    public ECpaOrderStatus getCpaOrderStatus(Order order) {
        if (order.getPublicType() == EOrderType.OT_GENERIC) {
            EOrderState state = (EOrderState) order.getEntityState();
            return GENERIC_ORDER_STATE_MAP.getOrDefault(state, ECpaOrderStatus.OS_PENDING);
        }
        return super.getCpaOrderStatus(order);
    }

    private List<Order> getOrdersBySnapshotsRequest(TListSnapshotsReqV2 request,
                                                    Collection<EDisplayOrderType> displayTypes) {
        Instant updatedAtFrom = ProtoUtils.toInstant(request.getUpdatedAtFrom());
        Instant updatedAtTo = getUpdatedAtToOrNow(request);

        return cpaOrderRepository.findOrdersByUpdatedAt(
                updatedAtFrom,
                updatedAtTo,
                displayTypes,
                PageRequest.of(0, request.getMaxSnapshots() + 1)
        );
    }

    private ECpaItemState getCpaItemState(OrderItem item) {
        if (item.getItemState() instanceof EOrderItemState) {
            EOrderItemState state = (EOrderItemState) item.getItemState();
            return GENERIC_ITEM_STATE_MAP.getOrDefault(state, ECpaItemState.IS_PENDING);
        } else {
            throw new UnsupportedOperationException(String.format("Item %s with state type %s is unsupported",
                    item.getId(), item.getItemState().getDeclaringClass().getSimpleName()));
        }
    }

    private TOrderSnapshot toSuburbanSnapshot(Order order, boolean addPersonalData) {
        SuburbanOrderItem orderItem = OrderCompatibilityUtils.getSuburbanOrderItem(order);
        SuburbanReservation reservation = orderItem.getPayload();

        String providerOrderId = "";
        if (reservation.getCpaProviderOrderId() != null) {
            providerOrderId = reservation.getCpaProviderOrderId();
        }

        ZonedDateTime departureDatetime = reservation.getDepartureDateTime();
        Timestamp departure = departureDatetime != null ? ProtoUtils.fromInstant(departureDatetime.toInstant()) : null;

        var extraBuilder = TSuburbanExtraData.newBuilder()
                .setOrderItemState(orderItem.getState())
                .setProvider(reservation.getProvider().toString())
                .setCarrierPartner(reservation.getCarrier().toString())
                .setProviderOrderId(providerOrderId)
                .setStationFromId(reservation.getStationFrom().getId())
                .setStationToId(reservation.getStationTo().getId())
                .setStationFromTitle(reservation.getStationFrom().getTitleDefault())
                .setStationToTitle(reservation.getStationTo().getTitleDefault())
                .setDepartureDate(departure);

        Money price = orderItem.calculateOriginalCostWithPreliminaryFallback();
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(orderItem).createSuburbanProvider();
        Money rewardFee = provider.getRewardFee(price);

        return mapBasicOrderInfo(order, orderItem.getItemNumber(), addPersonalData)
                .setSuburbanExtraData(extraBuilder.build())
                .setAmount(ProtoUtils.toTPrice(price))
                .setProfit(ProtoUtils.toTPrice(rewardFee))
                .setItemState(getCpaItemState(orderItem))
                .setCpaOrderStatus(getCpaOrderStatus(order))
                .build();
    }
}
