package ru.yandex.travel.orders.services.cpa;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.WhiteLabelApplication;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.models.booking_flow.promo.WhiteLabelPromoCampaign;
import ru.yandex.travel.hotels.proto.EWhiteLabelEligibility;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.cpa.ECpaOrderStatus;
import ru.yandex.travel.orders.cpa.ECpaRefundReason;
import ru.yandex.travel.orders.cpa.THotelDolphinExtraData;
import ru.yandex.travel.orders.cpa.THotelExpediaExtraData;
import ru.yandex.travel.orders.cpa.THotelExtraData;
import ru.yandex.travel.orders.cpa.THotelTravellineExtraData;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.cpa.TWhiteLabelCpaInfo;
import ru.yandex.travel.orders.cpa.TYandexPlusCpaInfo;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.entities.promo.mir2020.Mir2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.mir2020.MirPromoOrderEligibility;
import ru.yandex.travel.orders.grpc.CpaProperties;
import ru.yandex.travel.orders.repository.cpa.CpaHotelOrderRepository;
import ru.yandex.travel.orders.repository.promo.mir2020.Mir2020PromoOrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.hotels.HotelOrderDetailsHelpers;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;

import static java.util.stream.Collectors.toList;

@Component
@Slf4j
public class HotelCpaSnapshotProvider extends AbstractCpaSnapshotProvider {

    private static final Set<EHotelOrderState> CPA_HOTEL_BLACKLIST_STATES =
            Set.of(EHotelOrderState.OS_UNKNOWN,
                    EHotelOrderState.OS_MANUAL_PROCESSING,
                    EHotelOrderState.UNRECOGNIZED);
    public static final List<EHotelOrderState> CPA_HOTEL_ORDER_STATES_EXTENDED =
            Arrays.stream(EHotelOrderState.values())
                    .filter(e -> !CPA_HOTEL_BLACKLIST_STATES.contains(e))
                    .collect(Collectors.toUnmodifiableList());
    private static final Map<EServiceType, WellKnownWorkflowEntityType> HOTEL_SERVICE_TYPE_TO_WORKFLOW_ENTITY_TYPE_MAP = Map.of(
            EServiceType.PT_BNOVO_HOTEL, WellKnownWorkflowEntityType.BNOVO_ORDER_ITEM,
            EServiceType.PT_DOLPHIN_HOTEL, WellKnownWorkflowEntityType.DOLPHIN_ORDER_ITEM,
            EServiceType.PT_EXPEDIA_HOTEL, WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM,
            EServiceType.PT_TRAVELLINE_HOTEL, WellKnownWorkflowEntityType.TRAVELLINE_ORDER_ITEM,
            EServiceType.PT_BRONEVIK_HOTEL, WellKnownWorkflowEntityType.BRONEVIK_ORDER_ITEM
    );

    private static final BigDecimal DOLPHIN_RATE = new BigDecimal("0.13");
    private static final BigDecimal EXPEDIA_RATE = new BigDecimal("0.085");
    private static final BigDecimal BRONEVIK_RATE = new BigDecimal("0.125");

    private final CpaHotelOrderRepository cpaHotelOrderRepository;
    private final Mir2020PromoOrderRepository mir2020PromoOrderRepository;
    private final YandexPlusPromoService yandexPlusPromoService;

    public HotelCpaSnapshotProvider(CpaProperties cpaProperties,
                                    CpaHotelOrderRepository cpaHotelOrderRepository,
                                    Mir2020PromoOrderRepository mir2020PromoOrderRepository,
                                    YandexPlusPromoService yandexPlusPromoService,
                                    AuthorizationService authService) {
        super(cpaProperties, authService);
        this.cpaHotelOrderRepository = cpaHotelOrderRepository;
        this.mir2020PromoOrderRepository = mir2020PromoOrderRepository;
        this.yandexPlusPromoService = yandexPlusPromoService;
    }

    @Override
    public boolean supports(EOrderType orderType) {
        return orderType == EOrderType.OT_HOTEL_EXPEDIA;
    }

    @Override
    public List<TOrderSnapshot> getSnapshotsV2(TListSnapshotsReqV2 request) {
        List<HotelOrder> orders;

        Instant updatedAtFrom = ProtoUtils.toInstant(request.getUpdatedAtFrom());
        Instant updatedAtTo = getUpdatedAtToOrNow(request);
        // TODO(tlg-13): use cpaOrderRepository to support generic orders
        orders = cpaHotelOrderRepository.findOrdersByItemTypeAndUpdatedAtAndStateIn(
                updatedAtFrom,
                updatedAtTo,
                CPA_HOTEL_ORDER_STATES_EXTENDED,
                HOTEL_SERVICE_TYPE_TO_WORKFLOW_ENTITY_TYPE_MAP.get(request.getServiceType()).getDiscriminatorValue(),
                PageRequest.of(0, request.getMaxSnapshots() + 1)
        );
        List<UUID> ids = orders.stream().map(Order::getId).collect(toList());
        Set<UUID> promoOrderIds = mir2020PromoOrderRepository
                .findAllById(ids)
                .stream()
                .filter(p -> p.getEligibility() == MirPromoOrderEligibility.ELIGIBLE && p.getPaidWithMir() != null && p.getPaidWithMir())
                .map(Mir2020PromoOrder::getOrderId)
                .collect(Collectors.toSet());
        return orders.stream()
                .map(order -> mapHotelOrder(order, promoOrderIds, request.getAddPersonalData()))
                .collect(toList());
    }

    private TOrderSnapshot mapHotelOrder(HotelOrder hotelOrder,
                                         Set<UUID> mir2020PromoSet,
                                         boolean addPersonalData) {
        // TODO (mbobrov): think of a better way for getting hotel order item
        HotelOrderItem hotelOrderItem = (HotelOrderItem) hotelOrder.getOrderItems().get(0);
        TOrderSnapshot.Builder builder = mapBasicOrderInfo(hotelOrder, hotelOrderItem.getItemNumber(), addPersonalData);
        builder.setCpaOrderStatus(fromEHotelOrderStateExtended(hotelOrder.getState()));
        THotelExtraData.Builder hotelExtraBuilder = THotelExtraData.newBuilder();
        hotelExtraBuilder.setCheckInDate(ProtoUtils.toTDate(hotelOrderItem.getHotelItinerary().getOrderDetails().getCheckinDate()));
        hotelExtraBuilder.setCheckOutDate(ProtoUtils.toTDate(hotelOrderItem.getHotelItinerary().getOrderDetails().getCheckoutDate()));
        Optional.ofNullable(HotelOrderDetailsHelpers.getHotelCountry(hotelOrderItem.getHotelItinerary().getOrderDetails()))
                .filter(x -> x.getName() != null)
                .ifPresent(geo -> hotelExtraBuilder.setHotelCountryName(geo.getName()));
        Optional.ofNullable(HotelOrderDetailsHelpers.getHotelCity(hotelOrderItem.getHotelItinerary().getOrderDetails()))
                .filter(x -> x.getName() != null)
                .ifPresent(geo -> hotelExtraBuilder.setHotelCityName(geo.getName()));
        hotelExtraBuilder.setHotelName(hotelOrderItem.getHotelItinerary().getOrderDetails().getHotelName());
        hotelExtraBuilder.setPermalink(hotelOrderItem.getHotelItinerary().getOrderDetails().getPermalink());
        hotelExtraBuilder.setMir2020Eligible(mir2020PromoSet.contains(hotelOrder.getId()));

        DirectHotelBillingPartnerAgreement agreement = null;

        EServiceType serviceType = hotelOrderItem.getPublicType();
        if (serviceType == EServiceType.PT_BNOVO_HOTEL || serviceType == EServiceType.PT_TRAVELLINE_HOTEL) {
            agreement = ((DirectHotelBillingPartnerAgreementProvider) hotelOrderItem).getAgreement();
            if (agreement == null) {
                log.warn("Order with id {} doesn't have agreement and will be returned with 0 profit",
                        hotelOrder.getId());
                builder.setProfit(TPrice.newBuilder().setCurrency(ECurrency.C_RUB).build());
            }
        }
        //TODO [HOTELS-4751] duplicated commission calculation in TravellineFinancialDataProvider
        Money discountAmount = hotelOrder.calculateDiscountAmount();
        switch (hotelOrder.getState()) {
            case OS_CONFIRMED:
                builder.setAmountPayable(builder.getAmount());
                Supplier<BigDecimal> rateSupplier = agreement == null ? null : agreement::getOrderConfirmedRate;
                Money profit = ProtoUtils.fromTPrice(builder.getAmount())
                        .multiply(getRateByType(serviceType, rateSupplier))
                        .subtract(discountAmount);
                builder.setProfit(ProtoUtils.toTPrice(profit));
                break;
            case OS_REFUNDED:
                RefundInfo refundInfo = hotelOrderItem.getHotelItinerary().getRefundInfo();
                Money penalty = refundInfo.getPenalty().asMoney();
                builder.setAmountPayable(ProtoUtils.toTPrice(penalty));
                rateSupplier = agreement == null ? null : agreement::getOrderRefundedRate;
                profit = penalty.multiply(getRateByType(serviceType, rateSupplier));
                builder.setProfit(ProtoUtils.toTPrice(profit));
                if (refundInfo.getReason() != null) {
                    switch (refundInfo.getReason()) {
                        case SCHEDULE:
                            builder.setRefundReason(ECpaRefundReason.RR_SCHEDULE);
                            break;
                        case USER:
                            builder.setRefundReason(ECpaRefundReason.RR_USER);
                            break;
                        case OPERATOR:
                            builder.setRefundReason(ECpaRefundReason.RR_OPERATOR);
                            break;
                    }
                }
                break;
            default:
                // TODO (mbobrov): move profit to a persistent field. Assume that there's no profit
                //  for orders in states other than confirmed and refunded
                builder.setAmountPayable(TPrice.newBuilder().setCurrency(ECurrency.C_RUB).build());
                builder.setProfit(TPrice.newBuilder().setCurrency(ECurrency.C_RUB).build());
                break;
        }

        switch (hotelOrderItem.getPublicType()) {
            case PT_EXPEDIA_HOTEL:
                ExpediaHotelItinerary expediaHotelItinerary =
                        (ExpediaHotelItinerary) hotelOrderItem.getHotelItinerary();
                THotelExpediaExtraData.Builder extraData = THotelExpediaExtraData.newBuilder()
                        .setItineraryId(Strings.nullToEmpty(expediaHotelItinerary.getExpediaItineraryId()));
                if (expediaHotelItinerary.getConfirmation() != null) {
                    extraData.setConfirmationId(Strings.nullToEmpty(expediaHotelItinerary.getConfirmation().getPartnerConfirmationId()));
                }
                hotelExtraBuilder.setExpediaExtraData(extraData);
                break;
            case PT_DOLPHIN_HOTEL:
                if (hotelOrderItem.getHotelItinerary().getConfirmation() != null) {
                    hotelExtraBuilder.setDolphinExtraData(
                            THotelDolphinExtraData.newBuilder().setCode(
                                    Strings.nullToEmpty(hotelOrderItem.getHotelItinerary().getConfirmation().getPartnerConfirmationId())));
                }
                break;
            case PT_TRAVELLINE_HOTEL:
            case PT_BNOVO_HOTEL:
                if (hotelOrderItem.getHotelItinerary().getConfirmation() != null) {
                    hotelExtraBuilder.setTravellineExtraData(THotelTravellineExtraData.newBuilder()
                            .setConfirmationId(Strings.nullToEmpty(hotelOrderItem.getHotelItinerary().getConfirmation().getPartnerConfirmationId())));
                }
                break;
        }

        builder.setHotelExtraData(hotelExtraBuilder);
        return builder.build();
    }

    private BigDecimal getRateByType(EServiceType serviceType, Supplier<BigDecimal> rateSupplier) {
        BigDecimal rate;

        switch (serviceType) {
            case PT_BNOVO_HOTEL:
            case PT_TRAVELLINE_HOTEL:
                rate = rateSupplier.get();
                break;
            case PT_DOLPHIN_HOTEL:
                rate = DOLPHIN_RATE;
                break;
            case PT_EXPEDIA_HOTEL:
                rate = EXPEDIA_RATE;
                break;
            case PT_BRONEVIK_HOTEL:
                rate = BRONEVIK_RATE;
                break;
            default:
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Unexpected service type %s", serviceType).toEx();
        }
        return rate;
    }

    private ECpaOrderStatus fromEHotelOrderStateExtended(EHotelOrderState orderState) {
        switch (orderState) {
            case OS_NEW:
            case OS_WAITING_RESERVATION:
            case OS_WAITING_CANCELLATION:
            case OS_WAITING_REFUND_AFTER_CANCELLATION:
            case OS_WAITING_PAYMENT:
            case OS_WAITING_EXTRA_PAYMENT:
            case OS_WAITING_MONEY_ONLY_REFUND:
                return ECpaOrderStatus.OS_UNPAID;
            case OS_WAITING_CONFIRMATION:
            case OS_WAITING_SERVICE_REFUND:
            case OS_WAITING_INVOICE_REFUND:
                return ECpaOrderStatus.OS_PAID;
            case OS_CONFIRMED:
                return ECpaOrderStatus.OS_CONFIRMED;
            case OS_CANCELLED:
                return ECpaOrderStatus.OS_CANCELLED;
            case OS_REFUNDED:
                return ECpaOrderStatus.OS_REFUNDED;
            default:
                throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Encountered unexpected order state %s",
                        orderState).toEx();
        }
    }

    @Override
    public ECpaOrderStatus getCpaOrderStatus(Order order) {
        if (order.getPublicType() == EOrderType.OT_HOTEL_EXPEDIA) {
            return fromEHotelOrderStateExtended((EHotelOrderState) order.getEntityState());
        }
        return super.getCpaOrderStatus(order);
    }

    protected void enhanceBasicOrderInfo(TOrderSnapshot.Builder dto, Order order, int itemNumber) {
        HotelOrder hotelOrder = (HotelOrder) order;

        TYandexPlusCpaInfo yandexPlusCpaInfo = yandexPlusPromoService.getYandexPlusCpaInfoForOrder(hotelOrder);
        if (yandexPlusCpaInfo != null) {
            dto.setYandexPlusCpaInfo(yandexPlusCpaInfo);
        }
        TWhiteLabelCpaInfo whiteLabelCpaInfo = getWhiteLabelCpaInfoForOrder(hotelOrder);
        if (whiteLabelCpaInfo != null) {
            dto.setWhiteLabelCpaInfo(whiteLabelCpaInfo);
        }
    }

    private TWhiteLabelCpaInfo getWhiteLabelCpaInfoForOrder(HotelOrder hotelOrder) {
        HotelItinerary itinerary = OrderCompatibilityUtils.getOnlyHotelOrderItem(hotelOrder).getHotelItinerary();
        if (Optional.ofNullable(itinerary.getActivePromoCampaigns())
                .map(PromoCampaignsInfo::getWhiteLabel)
                .map(WhiteLabelPromoCampaign::getEligible)
                .filter(e -> e == EWhiteLabelEligibility.WLE_ELIGIBLE).isEmpty()) {
            return null;
        }

        WhiteLabelPromoCampaign whiteLabel = itinerary.getActivePromoCampaigns().getWhiteLabel();
        TWhiteLabelCpaInfo.Builder infoBuilder = TWhiteLabelCpaInfo.newBuilder()
                .setPointsAmount(whiteLabel.getPoints().getAmount())
                .setPointsType(whiteLabel.getPoints().getPointsType());

        Optional.ofNullable(itinerary.getAppliedPromoCampaigns())
                .map(AppliedPromoCampaigns::getWhiteLabel)
                .map(WhiteLabelApplication::getCustomerNumber)
                .ifPresent(infoBuilder::setCustomerNumber);

        return infoBuilder.build();
    }
}
