package ru.yandex.travel.orders.services.finances;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;

import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.services.finances.providers.DirectHotelBillingFinancialDataProvider;
import ru.yandex.travel.orders.services.finances.providers.ExpediaVirtualFinancialDataProvider;
import ru.yandex.travel.orders.services.finances.providers.FinancialDataProvider;
import ru.yandex.travel.orders.services.finances.providers.FinancialDataProviderRegistry;
import ru.yandex.travel.orders.services.finances.providers.ServiceBalance;
import ru.yandex.travel.orders.services.finances.providers.YandexPlusFinancialDataProvider;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.tx.utils.TransactionMandatory;

import static ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode.MRM_PROMO_MONEY_FIRST;

@Service
@RequiredArgsConstructor
@Slf4j
public class FinancialEventService {
    public static final EMoneyRefundMode DEFAULT_MONEY_REFUND_MODE = MRM_PROMO_MONEY_FIRST;

    private static final EServiceType ANY_UNSUPPORTED_SERVICE_TYPE = null;

    private final FinancialEventServiceProperties properties;
    private final FinancialEventRepository repository;
    private final FinancialDataProviderRegistry providerRegistry;
    // todo(tlg-13): rm this direct reference, use the registry instead
    private final DirectHotelBillingFinancialDataProvider directHotelBillingFinancialDataProvider;
    private final ExpediaVirtualFinancialDataProvider expediaVirtualProvider;
    private final YandexPlusFinancialDataProvider yandexPlusProvider;

    private final Map<EServiceType, Counter> confirmCounters = new HashMap<>();
    private final Map<EServiceType, Counter> fullyPaidCounters = new HashMap<>();
    private final Map<EServiceType, Counter> refundCounters = new HashMap<>();
    private final Map<EServiceType, Counter> extraCounters = new HashMap<>();
    private final Counter topupCounter = Counter.builder("finances.events")
            .tag("type", "topup").register(Metrics.globalRegistry);

    @PostConstruct
    public void init() {
        if (properties.getEnabled()) {
            log.info("FinancialEventService initialized; supported service types: {}",
                    providerRegistry.getSupportedServiceTypes());
            for (EServiceType serviceType : providerRegistry.getSupportedServiceTypes()) {
                confirmCounters.put(serviceType, Counter.builder("finances.events")
                        .tag("provider", serviceType.name()).tag("type", "confirm").register(Metrics.globalRegistry));
                refundCounters.put(serviceType, Counter.builder("finances.events")
                        .tag("provider", serviceType.name()).tag("type", "refund").register(Metrics.globalRegistry));
                extraCounters.put(serviceType, Counter.builder("finances.events")
                        .tag("provider", serviceType.name()).tag("type", "extra").register(Metrics.globalRegistry));
                fullyPaidCounters.put(serviceType, Counter.builder("finances.events")
                        .tag("provider", serviceType.name()).tag("type", "fullyPaid").register(Metrics.globalRegistry));
            }
        } else {
            log.warn("FinancialEventService is disabled");
        }
        confirmCounters.put(ANY_UNSUPPORTED_SERVICE_TYPE, Counter.builder("finances.events")
                .tag("provider", "UNSUPPORTED").tag("type", "confirm").register(Metrics.globalRegistry));
        refundCounters.put(ANY_UNSUPPORTED_SERVICE_TYPE, Counter.builder("finances.events")
                .tag("provider", "UNSUPPORTED").tag("type", "refund").register(Metrics.globalRegistry));
        extraCounters.put(ANY_UNSUPPORTED_SERVICE_TYPE, Counter.builder("finances.events")
                .tag("provider", "UNSUPPORTED").tag("type", "extra").register(Metrics.globalRegistry));
        fullyPaidCounters.put(ANY_UNSUPPORTED_SERVICE_TYPE, Counter.builder("finances.events")
                .tag("provider", "UNSUPPORTED").tag("type", "fullyPaid").register(Metrics.globalRegistry));
    }

    public boolean isEnabledFor(OrderItem orderItem) {
        if (OrderCompatibilityUtils.isTrainOrder(orderItem.getOrder())) {
            TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(orderItem.getOrder());
            return properties.getEnabled()
                    && providerRegistry.supportsProvider(orderItem.getPublicType())
                    && invoice.getProcessThroughYt();
        } else {
            return properties.getEnabled()
                    && providerRegistry.supportsProvider(orderItem.getPublicType());
        }

    }

    @TransactionMandatory
    public void registerConfirmedService(OrderItem orderItem) {
        registerConfirmedService(orderItem, null);
    }

    @TransactionMandatory
    public void registerConfirmedService(OrderItem orderItem, Instant payoutAt) {
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring service confirmation");
            confirmCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());
        for (FinancialEvent event : provider.onConfirmation(orderItem, properties.getEnablePromoFee())) {
            if (payoutAt != null) {
                event.setPayoutAt(payoutAt);
                if (event.getAccountingActAt().isBefore(payoutAt)) {
                    event.setAccountingActAt(payoutAt);
                }
            }
            event = repository.save(event);
            confirmCounters.get(orderItem.getPublicType()).increment();
            log.info("Registered a new confirmation event; event id {}", event.getId());
        }
    }

    @TransactionMandatory
    public boolean updateEventsWithoutBalanceChanges(OrderItem orderItem, boolean inferBillingIdsFromEvents) {
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring service confirmation");
            confirmCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return false;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());
        var events = provider.onUpdateFinancialEventsWithoutBalanceChanges(orderItem, properties.getEnablePromoFee(),
                inferBillingIdsFromEvents);
        if (events.isEmpty()) {
            return false;
        }
        setOriginalEventsIfNecessary(events);
        for (FinancialEvent event : events) {
            event = repository.save(event);
            log.info("Registered a new correction event; event id {}", event.getId());
        }
        return true;
    }

    @TransactionMandatory
    public void registerFullyPaidPaymentSchedule(OrderItem orderItem, PaymentSchedule schedule) {
        Preconditions.checkArgument(schedule.getState() == EPaymentState.PS_FULLY_PAID,
                "Schedule is not fully paid");
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring service confirmation");
            fullyPaidCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());
        for (FinancialEvent event : provider.onPaymentScheduleFullyPaid(orderItem, schedule, properties.getEnablePromoFee())) {
            event = repository.save(event);
            fullyPaidCounters.get(orderItem.getPublicType()).increment();
            log.info("Registered a new fully paid event; event id {}", event.getId());
        }
    }

    @TransactionMandatory
    public <T extends HotelOrderItem & DirectHotelBillingPartnerAgreementProvider> void registerServiceFullCorrection(
            T orderItem, Instant payoutAt) {
        if (payoutAt == null) {
            payoutAt = repository.findOriginalPaymentEvent(orderItem).getPayoutAt();
        }
        List<FinancialEvent> events = directHotelBillingFinancialDataProvider
                .issueFullCorrectionEvent(orderItem, payoutAt, properties.getEnablePromoFee());
        setOriginalEventsIfNecessary(events);
        repository.saveAll(events);
        refundCounters.get(orderItem.getPublicType()).increment();
        for (FinancialEvent event : events) {
            String type = event.isRefund() ? "refund" : "correction";
            log.info("Registered a new correction {} event; event id {}", type, event.getId());
        }
    }

    private void setOriginalEventsIfNecessary(List<FinancialEvent> events) {
        for (FinancialEvent event : events) {
            if (event.isRefund() && event.getOriginalEvent() == null) {
                // the rollback operation may produce additional payment events (corrections)
                // we need to link the refund event until the correction is stored
                FinancialEvent originalEvent = repository.findOriginalPaymentEvent(event.getOrderItem());
                event.setOriginalEvent(originalEvent);
            }
        }
    }

    @TransactionMandatory
    public void registerRefundedService(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Money penalty,
                                        Money refund, Instant refundedAt) {
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring service refund");
            refundCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());

        List<FinancialEvent> events = penalty == null ?
                provider.onRefund(orderItem, moneyRefundMode, properties.getEnablePromoFee()) :
                provider.onRefund(orderItem, moneyRefundMode, properties.getEnablePromoFee(), penalty, refund, refundedAt);
        setOriginalEventsIfNecessary(events);

        repository.saveAll(events);
        refundCounters.get(orderItem.getPublicType()).increment();
        for (FinancialEvent event : events) {
            String type = event.isRefund() ? "refund" : "correction";
            log.info("Registered a new {} event; event id {}", type, event.getId());
        }
    }

    @TransactionMandatory
    public void registerRefundedService(OrderItem orderItem, EMoneyRefundMode moneyRefundMode) {
        registerRefundedService(orderItem, moneyRefundMode, null, null, null);
    }

    @TransactionMandatory
    public void registerExtraPaymentForService(OrderItem orderItem, Payment payment) {
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring extra payment for service");
            extraCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());

        for (FinancialEvent event : provider.onExtraPayment(orderItem, payment, properties.getEnablePromoFee())) {
            event = repository.save(event);
            extraCounters.get(orderItem.getPublicType()).increment();
            log.info("Registered an extra payment event; event id {}", event.getId());
        }
    }

    @TransactionMandatory
    public void registerPartialRefundForService(OrderItem orderItem, OrderRefund orderRefund) {
        if (!isEnabledFor(orderItem)) {
            log.info("Ignoring partial refund for service");
            extraCounters.get(ANY_UNSUPPORTED_SERVICE_TYPE).increment();
            return;
        }
        FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());

        List<FinancialEvent> events = provider.onPartialRefund(orderItem, orderRefund, properties.getEnablePromoFee());
        setOriginalEventsIfNecessary(events);
        for (FinancialEvent event : events) {
            event = repository.save(event);
            refundCounters.get(orderItem.getPublicType()).increment();
            log.info("Registered a partial refund event; event id {}", event.getId());
        }
    }

    public OverallServiceBalance getOverallServiceBalance(OrderItem orderItem) {
        if (isEnabledFor(orderItem)) {
            List<FinancialEvent> events = repository.findAllByOrderItem(orderItem);
            Money total = orderItem.calculateOriginalCostWithPreliminaryFallback();
            ServiceBalance balance = new ServiceBalance(events, total.getCurrency());
            return balance.getOverallBalance().convertToOverall();
        } else if (orderItem.getPublicType() == EServiceType.PT_EXPEDIA_HOTEL) {
            return expediaVirtualProvider.getApproximateOverallServiceBalance(orderItem);
        } else {
            throw new IllegalArgumentException("Unsupported service type: " + orderItem.getPublicType());
        }
    }

    @TransactionMandatory
    public void registerPlusPointsTopup(YandexPlusTopup topup) {
        FinancialEvent event = yandexPlusProvider.onPlusTopup(topup);
        event = repository.save(event);
        topupCounter.increment();
        log.info("Registered a new topup event; event id {}", event.getId());
    }

    public VatType getYandexFeeVat(OrderItem orderItem) {
        if (isEnabledFor(orderItem)) {
            FinancialDataProvider provider = providerRegistry.getProvider(orderItem.getPublicType());
            return provider.getYandexFeeVat(orderItem);
        } else if (orderItem.getPublicType() == EServiceType.PT_EXPEDIA_HOTEL) {
            return expediaVirtualProvider.getYandexFeeVat(orderItem);
        } else {
            throw new IllegalArgumentException("Unsupported service type: " + orderItem.getPublicType());
        }
    }
}
