package ru.yandex.travel.orders.services.finances.billing;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.Month;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import ru.yandex.travel.integration.balance.BillingCsvApiClient;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.finances.BankOrderPayment;
import ru.yandex.travel.orders.entities.finances.BankOrderPaymentDetailsStatus;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.OebsPaymentStatus;
import ru.yandex.travel.orders.repository.finances.BankOrderPaymentRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.report.HotelPartnerPaymentOrderReportSender;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

import static ru.yandex.travel.orders.entities.finances.BankOrderPaymentDetailsStatus.FETCHED;
import static ru.yandex.travel.orders.entities.finances.BankOrderPaymentDetailsStatus.STATUSES_FOR_UPDATE;
import static ru.yandex.travel.orders.repository.finances.BankOrderPaymentRepository.NO_EXCLUDE_IDS;

@RequiredArgsConstructor
@Slf4j
public class BillingBankOrderSyncService {

    public static final String TASK_KEY = "BILLING_BANK_ORDER_SYNC";
    private final BillingCsvApiClient billingCsvApiClient;
    private final BankOrderRepository bankOrderRepository;
    private final BankOrderPaymentRepository bankOrderPaymentRepository;
    private final BillingBankOrderSyncProperties billingBankOrderSyncProperties;
    private final SingleOperationService singleOperationService;
    private final Clock clock;
    private final Set<BillingBankOrderSyncService.DateInterval> intervalsForFetching = Collections.synchronizedSet(new HashSet<>());

    private static List<DateInterval> splitToIntervals(LocalDate from, LocalDate to, int intervalMaxDays) {
        Preconditions.checkArgument(0 < intervalMaxDays, "intervalMaxDays must be greater than zero");

        final List<DateInterval> result = new ArrayList<>();

        LocalDate currentDateFrom = from;
        LocalDate currentDateTo = from.plusDays(intervalMaxDays - 1);
        while (currentDateTo.compareTo(to) < 0) {
            log.info("new interval for loading billing bank order {} - {}", currentDateFrom, currentDateTo);
            result.add(new DateInterval(currentDateFrom, currentDateTo));
            currentDateFrom = currentDateTo.plusDays(1);
            currentDateTo = currentDateFrom.plusDays(intervalMaxDays - 1);
        }
        log.info("new interval for loading billing bank order {} - {}", currentDateFrom, to);
        result.add(new DateInterval(currentDateFrom, to));

        return result;
    }

    @TransactionMandatory
    public void syncNewBankOrders(String taskKeyIgnored) {

        if (!intervalsForFetching.isEmpty()) {
            return;
        }
        // earliestNotFullyProcessedPaymentDate || latestFullyProcessedPaymentDate || START_DATE
        final LocalDate from = Optional.ofNullable(billingBankOrderSyncProperties.getStartDate())
                .orElse(Optional.ofNullable(bankOrderRepository.earliestNotFullyProcessedPaymentDate())
                                .orElse(Optional.ofNullable(bankOrderRepository.latestFullyProcessedPaymentDate())
                                        .orElse(LocalDate.of(2019, Month.JANUARY, 1))));
        final LocalDate to = LocalDate.ofInstant(Instant.now(clock), BillingCsvApiClient.BILLING_TIME_ZONE_ID);

        log.info("Load billing bank orders in date range {} - {}", from, to);
        log.info("max inteval for loading {}", billingBankOrderSyncProperties.getMaxIntervalDays());

        var  newIntervals = splitToIntervals(from, to, billingBankOrderSyncProperties.getMaxIntervalDays());
        log.info("add {} new intervals for loading billing bank orders ", newIntervals.size());
        intervalsForFetching.addAll(newIntervals);
    }

    @TransactionMandatory
    public void fetchBatchBankOrders(DateInterval interval) {

        log.info("Loading billing bank orders for interval {}", interval);
        final List<BankOrder> bankOrders =
                billingCsvApiClient.getBankOrders(FinancialEventPaymentScheme.HOTELS.getServiceId(),
                                interval.getFrom(), interval.getTo().plusDays(1))
                        .stream()
                        .map(BillingObjectMapper::map)
                        .filter(b -> Objects.nonNull(b.getBankOrderId()))
                        .collect(Collectors.toList());
        log.info("Loaded {} bank orders form billing API", bankOrders.size());
        for (int i = 0, bankOrdersSize = bankOrders.size(); i < bankOrdersSize; i++) {
            BankOrder bankOrder = bankOrders.get(i);
            final BankOrder orderInDatabase = findAndEnrichDbOrder(bankOrder);
            sendReportIfNeeded(orderInDatabase, i);
            bankOrderRepository.save(orderInDatabase);
            log.info("Loading bank order to db with id: {}", orderInDatabase.getBankOrderId());
        }
        log.info("Done loading billing bank orders in date range {} - {}", interval.getFrom(), interval.getTo());
        intervalsForFetching.remove(interval);
    }

    /**
     * Looks for an existing order in db. If none is present, creates one.
     */
    @NotNull
    private BankOrder findAndEnrichDbOrder(BankOrder bankOrder) {
        final Optional<BankOrder> foundOrder =
                Optional.ofNullable(bankOrderRepository.lookForExistingBankOrders(
                        bankOrder.getBankOrderPayment().getPaymentBatchId(),
                        bankOrder.getBankOrderId()
                ));
        final BankOrder orderInDatabase;
        if (foundOrder.isPresent()) {
            orderInDatabase = foundOrder.get();
            orderInDatabase.setTrantime(bankOrder.getTrantime());
            orderInDatabase.setEventtime(bankOrder.getEventtime());
            orderInDatabase.setStatus(bankOrder.getStatus());
            orderInDatabase.setOebsPaymentStatus(bankOrder.getOebsPaymentStatus());
            if (orderInDatabase.getBankOrderId() == null) {
                orderInDatabase.setBankOrderId(bankOrder.getBankOrderId());
            }
            log.info("Found bank order: paymentBatchId = {}", bankOrder.getBankOrderPayment().getPaymentBatchId());
        } else {
            final Optional<BankOrderPayment> foundPayment =
                    bankOrderPaymentRepository.findById(bankOrder.getBankOrderPayment().getPaymentBatchId());
            if (foundPayment.isPresent()) {
                bankOrder.setBankOrderPayment(foundPayment.get());
            } else {
                bankOrder.getBankOrderPayment().setStatus(BankOrderPaymentDetailsStatus.NEW);
            }
            orderInDatabase = bankOrder;
            log.info("Creating a new bank order: paymentBatchId = {}, found order payment = {}",
                    bankOrder.getBankOrderPayment().getPaymentBatchId(),
                    foundPayment.isPresent()
            );
        }
        return orderInDatabase;
    }


    /**
     * Sends a report to the partner for the bank order. Gets sent
     */
    private void sendReportIfNeeded(BankOrder order) {
        sendReportIfNeeded(order, 0);
    }

    private void sendReportIfNeeded(BankOrder order, int idx) {
        if (order.getReportSent() || !EnumSet.of(
                OebsPaymentStatus.RECONCILED,
                OebsPaymentStatus.RETURNED,
                OebsPaymentStatus.CONFIRMED
        ).contains(order.getOebsPaymentStatus())) {
            log.info("Dont send report: already sent, paymentBatchId = {}, bankOrderId = {}",
                    order.getBankOrderPayment().getPaymentBatchId(), order.getBankOrderId());
            return;
        }
        if (order.getBankOrderPayment().getStatus() != FETCHED) {
            // in case if the report is not consistent, but old, for now we'll send it anyway to keep backward
            // compatibility
            // TODO TRAVELBACK-3122 need to clarify the cases for empty reports and not consistent
            if (order.getCreatedAt() == null || order.getCreatedAt().isAfter(Instant.now(clock).minus(1,
                    ChronoUnit.DAYS))) {
                log.info("Dont send report: order payment is fetched, paymentBatchId = {}, bankOrderId = {}",
                        order.getBankOrderPayment().getPaymentBatchId(), order.getBankOrderId());
                return;
            }
        }
        sendReport(order, idx);
        order.setReportSent(true);
        log.info("Report is sent, paymentBatchId = {}, bankOrderId = {}",
                order.getBankOrderPayment().getPaymentBatchId(), order.getBankOrderId());
    }

    private void sendReport(BankOrder bankOrder, int idx) {
        var opName = String.format("PartnerPaymentOrderReport_%s_%s",
                bankOrder.getBankOrderId(),
                bankOrder.getBankOrderPayment().getPaymentBatchId());
        Instant scheduleAt = Instant.now(clock).plus(billingBankOrderSyncProperties.getEmailsSendingDelay()
                .multipliedBy(idx));
        log.info("Scheduling to send partner payment order reports {} at {}", opName, scheduleAt);
        singleOperationService.scheduleUniqueOperation(
                opName,
                OperationTypes.HOTELS_SEND_PARTNER_PAYMENT_ORDERS_REPORT.getValue(),
                HotelPartnerPaymentOrderReportSender.Params.fromBankOrder(bankOrder),
                scheduleAt);
    }

    @TransactionMandatory
    public void syncBankOrderPaymentDetails(String paymentBatchId) {
        final BankOrderPayment foundPayment = bankOrderPaymentRepository.getOne(paymentBatchId);
        final List<BankOrderDetail> details =
                billingCsvApiClient.getBankOrderDetails(foundPayment.getPaymentBatchId())
                        .stream()
                        .map(BillingObjectMapper::map)
                        .collect(Collectors.toList());

        if (details.isEmpty()) {
            if (foundPayment.safeGetCheckAttempt() < billingBankOrderSyncProperties.getBankPaymentCheckAttempts()) {
                log.error("No payment details for payment with batch id {}. Rescheduling fetch",
                        paymentBatchId);
                foundPayment.rescheduleNextCheckAtAndRegisterAttempt(
                        Instant.now(clock).plus(billingBankOrderSyncProperties.getBankPaymentAttemptDelay())
                );
            } else {
                log.error("No payment details for payment with batch id {}. Skipping", paymentBatchId);
                foundPayment.setStatus(BankOrderPaymentDetailsStatus.SKIPPED);
            }
        } else {
            for (BankOrderDetail detail : details) {
                foundPayment.addBankOrderDetail(detail);
            }
            if (foundPayment.isConsistent()) {
                foundPayment.setStatus(BankOrderPaymentDetailsStatus.FETCHED);
                foundPayment.getOrders().forEach(this::sendReportIfNeeded);
            } else {
                foundPayment.setStatus(BankOrderPaymentDetailsStatus.NOT_FULLY_FETCHED);
                foundPayment.rescheduleNextCheckAtAndRegisterAttempt(
                        Instant.now(clock).plus(billingBankOrderSyncProperties.getBankPaymentAttemptDelay())
                );
            }
        }
    }

    public Collection<DateInterval> getIntervalsForFetching() {
        return intervalsForFetching;
    }

    public int countIntervalsForFetching(){
        return intervalsForFetching.size();
    }


    @TransactionMandatory
    public Collection<String> getPendingBankOrderPayments(Set<String> excludeIds, int limit) {
        Pageable paging = PageRequest.of(0, limit);
        Set<String> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return bankOrderPaymentRepository.findIdsForDetailsSynchronization(STATUSES_FOR_UPDATE,
                Instant.now(clock), safeExcludeIds, paging);
    }

    @TransactionMandatory
    public long countPendingBankOrderPayments(Set<String> excludeIds) {
        Set<String> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return bankOrderPaymentRepository.countIdsForDetailsSynchronization(STATUSES_FOR_UPDATE,
                Instant.now(clock), safeExcludeIds);
    }

    @Data
    @AllArgsConstructor
    public static class DateInterval implements Comparable<DateInterval> {
        private final LocalDate from;
        private final LocalDate to;

        @Override
        public int compareTo(@NotNull DateInterval o) {
            return o.from.compareTo(from);
        }
    }

}
