package ru.yandex.travel.orders.services.finances.billing;

import java.time.Clock;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.integration.balance.BillingCsvApiClient;
import ru.yandex.travel.orders.repository.finances.BankOrderPaymentRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.RegularTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorHelper;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Configuration
@ConditionalOnProperty("billing-bank-order-sync.enabled")
@RequiredArgsConstructor
public class BillingBankOrderSyncServiceConfiguration {

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final BillingCsvApiClient billingCsvApiClient;
    private final BankOrderPaymentRepository bankOrderPaymentRepository;
    private final BankOrderRepository bankOrderRepository;
    private final SingleOperationService singleOperationService;
    private final Clock clock;

    @Bean
    @ConfigurationProperties(prefix = "billing-bank-order-sync", ignoreUnknownFields = false)
    public BillingBankOrderSyncProperties billingBankOrderSyncProperties() {
        return BillingBankOrderSyncProperties.builder()
                .bankPaymentProcessor(
                        TaskProcessorProperties.builder()
                                .name("BillingBankOrderPaymentSync")
                                .daemonPoolThreads(true)
                                .gracefulPoolShutdown(true)
                                .build()
                ).build();
    }

    @Bean
    public BillingBankOrderSyncService billingBankOrderSyncService(BillingBankOrderSyncProperties billingBankOrderSyncProperties) {
        return new BillingBankOrderSyncService(
                billingCsvApiClient, bankOrderRepository, bankOrderPaymentRepository,
                billingBankOrderSyncProperties, singleOperationService, clock
        );
    }

    @Bean
    public TaskProcessor<String> billingBankOrderPaymentSyncTaskProcessor(BillingBankOrderSyncService billingBankOrderSyncService,
                                                                          BillingBankOrderSyncProperties billingBankOrderSyncProperties) {

        TaskKeyProvider<String> taskKeyProvider =
                TaskProcessorHelper.createTaskKeyProvider(billingBankOrderSyncService::getPendingBankOrderPayments,
                        billingBankOrderSyncService::countPendingBankOrderPayments);

        TransactionDefinition txDefinition = TaskProcessorHelper.createDefaultTxDefinition(
                "BillingBankOrderPaymentSync");

        return new TaskProcessor<>(
                taskKeyProvider,
                billingBankOrderSyncService::syncBankOrderPaymentDetails,
                forcedRollbackTxManagerWrapper,
                txDefinition,
                billingBankOrderSyncProperties.getBankPaymentProcessor()
        );
    }

    @Bean
    public TaskProcessor<String> billingBankOrderSyncTaskProcessor(BillingBankOrderSyncService billingBankOrderSyncService,
                                                                   BillingBankOrderSyncProperties billingBankOrderSyncProperties) {
        TaskKeyProvider<String> taskKeyProvider = new RegularTaskKeyProvider<>(BillingBankOrderSyncService.TASK_KEY);

        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("BillingBankOrderSync");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(
                taskKeyProvider,
                billingBankOrderSyncService::syncNewBankOrders,
                forcedRollbackTxManagerWrapper,
                txDefinition,
                TaskProcessorProperties.builder()
                        .name("BillingBankOrderSync")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(1)  // need only one worker
                        .initialStartDelay(billingBankOrderSyncProperties.getStartDelay())
                        .scheduleRate(billingBankOrderSyncProperties.getRefreshInterval())
                        .build()
        );
    }

    @Bean
    public BillingFetchingBankOrderSyncTask billingFetchingBankOrderSyncTask(
            BillingBankOrderSyncService billingBankOrderSyncService
    ) {
        return new BillingFetchingBankOrderSyncTask(billingBankOrderSyncService);
    }

    @Bean
    public TaskProcessor<BillingBankOrderSyncService.DateInterval> billingFetchingBankOrderTaskProcessor(
            BillingFetchingBankOrderSyncTask billingFetchingBankOrderSyncTask
    ) {
        return TaskProcessorHelper.simpleTaskProcessor(
                billingFetchingBankOrderSyncTask,
                forcedRollbackTxManagerWrapper,
                billingBankOrderSyncProperties().getBillingFetchingBankOrdersProcessor()
        );
    }
}
