package ru.yandex.travel.orders.services.finances.billing;

import com.google.common.base.Strings;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.integration.balance.model.csv.BillingCurrency;
import ru.yandex.travel.integration.balance.model.csv.PaymentType;
import ru.yandex.travel.integration.balance.model.csv.TransactionType;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.finances.BankOrderPayment;
import ru.yandex.travel.orders.entities.finances.BankOrderStatus;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionType;
import ru.yandex.travel.orders.entities.finances.OebsPaymentStatus;

public final class BillingObjectMapper {

    public static BankOrder map(ru.yandex.travel.integration.balance.model.csv.BankOrder source) {
        final BankOrder result = new BankOrder();

        /*
           Often Billing returns fresh records with empty bank order id, later
           the same records are returned with defined bank order id
         */
        if (!StringUtils.isBlank(source.getBankOrderId())) {
            result.setBankOrderId(source.getBankOrderId());
        }
        result.setServiceId(source.getServiceId());
        result.setTrantime(source.getTrantime());
        result.setEventtime(source.getEventtime());
        result.setSum(source.getSum());
        result.setOebsPaymentStatus(map(source.getOebsPaymentStatus()));
        result.setStatus(map(source.getStatus()));
        result.setDescription(source.getDescription());

        final BankOrderPayment bankOrderPayment = new BankOrderPayment();
        bankOrderPayment.setPaymentBatchId(source.getPaymentBatchId());
        result.setBankOrderPayment(bankOrderPayment);

        return result;
    }

    public static OebsPaymentStatus map(ru.yandex.travel.integration.balance.model.csv.OebsPaymentStatus source) {
        switch (source) {
            case VOID:
                return OebsPaymentStatus.VOID;
            case CREATED:
                return OebsPaymentStatus.CREATED;
            case CONFIRMED:
                return OebsPaymentStatus.CONFIRMED;
            case TRANSMITTED:
                return OebsPaymentStatus.TRANSMITTED;
            case RECONCILED:
                return OebsPaymentStatus.RECONCILED;
            case RETURNED:
                return OebsPaymentStatus.RETURNED;
            default:
                throw new RuntimeException(String.format("Unsupported status %s", source));
        }
    }

    public static BankOrderStatus map(ru.yandex.travel.integration.balance.model.csv.BankOrderStatus source) {
        switch (source) {
            case NEW:
                return BankOrderStatus.NEW;
            case STARTED:
                return BankOrderStatus.STARTED;
            case DONE:
                return BankOrderStatus.DONE;
            case CANCELLED:
                return BankOrderStatus.CANCELLED;
            default:
                throw new RuntimeException(String.format("Unsupported status %s", source));
        }
    }

    public static BankOrderDetail map(ru.yandex.travel.integration.balance.model.csv.BankOrderDetail source) {
        final BankOrderDetail result = new BankOrderDetail();

        result.setTrustPaymentId(source.getTrustPaymentId());
        result.setTrustRefundId(source.getTrustRefundId());
        if (!Strings.isNullOrEmpty(source.getTrustPaymentId())) {
            result.setYtId(getLongFromHex(source.getTrustPaymentId()));
        } else if (!Strings.isNullOrEmpty(source.getTrustRefundId())) {
            result.setYtId(getLongFromHex(source.getTrustRefundId()));
        }
        result.setPaymentTime(source.getPaymentTime());
        result.setHandlingTime(source.getHandlingTime());
        result.setPaymentType(map(source.getPaymentType()));
        result.setTransactionType(map(source.getTransactionType()));
        result.setSum(source.getSum());
        result.setCurrency(map(source.getCurrency()));
        result.setAgencyCommission(source.getAgencyCommission());
        result.setContractId(source.getContractId());
        result.setServiceOrderId(source.getServiceOrderId());

        return result;
    }

    private static long getLongFromHex(String hexString) {
        return Long.parseLong(hexString, 16);
    }

    public static BillingTransactionPaymentType map(PaymentType source) {
        switch (source) {
            case COST:
                return BillingTransactionPaymentType.COST;
            case REWARD:
                return BillingTransactionPaymentType.REWARD;
            case YANDEX_ACCOUNT_WITHDRAW:
                return BillingTransactionPaymentType.YANDEX_ACCOUNT_COST_WITHDRAW;
            default:
                throw new RuntimeException(String.format("Unsupported payment type %s", source));
        }
    }

    public static BillingTransactionType map(TransactionType source) {
        switch (source) {
            case PAYMENT:
                return BillingTransactionType.PAYMENT;
            case REFUND:
                return BillingTransactionType.REFUND;
            default:
                throw new RuntimeException(String.format("Unsupported transaction type %s", source));
        }
    }

    public static ProtoCurrencyUnit map(BillingCurrency source) {
        switch (source) {
            case RUR:
                return ProtoCurrencyUnit.RUB;
            case USD:
                return ProtoCurrencyUnit.USD;
            case EUR:
                return ProtoCurrencyUnit.EUR;
            default:
                throw new RuntimeException(String.format("Unsupported currency %s", source));
        }
    }

    BillingObjectMapper() {
        /* static class */
    }
}
