package ru.yandex.travel.orders.services.finances.billing;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.integration.balance.BillingClientContract;
import ru.yandex.travel.orders.cache.HotelAgreementDictionary;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.repository.BillingPartnerConfigRepository.NO_EXCLUDE_IDS;

@RequiredArgsConstructor
@Slf4j
public class BillingPartnerAgreementSynchronizer implements FinancialEventProcessor {
    private final BillingPartnerAgreementSynchronizerProperties properties;
    private final BillingPartnerConfigRepository partnerConfigRepository;
    private final BillingApiClient billingApiClient;
    private final ClockService clockService;
    private final HotelAgreementDictionary hotelAgreementDictionary;

    @TransactionMandatory
    public List<Long> getReadyTasks(Set<Long> excludeIds, int limit) {
        Pageable paging = PageRequest.of(0, limit);
        Set<Long> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return partnerConfigRepository.findIdsForAgreementSynchronization(
                getMaxSynchronizedAt(), safeExcludeIds, paging);
    }

    @TransactionMandatory
    public long countCountTasks(Set<Long> excludeIds) {
        Set<Long> safeExcludeIds = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return partnerConfigRepository.countIdsForAgreementSynchronization(getMaxSynchronizedAt(), safeExcludeIds);
    }

    @TransactionMandatory
    public void processTask(Long billingClientId) {
        log.debug("Synchronizing agreement status; billing client id {}", billingClientId);
        BillingPartnerConfig config = partnerConfigRepository.getOne(billingClientId);
        int activeContracts = 0;
        List<BillingClientContract> contracts = billingApiClient.getClientContracts(billingClientId);
        for (BillingClientContract contract : contracts) {
            if (contract.isActive()) {
                activeContracts += 1;
            }
        }
        Preconditions.checkState(activeContracts <= 1, "Multiple active contracts detected; " +
                "billing client id %s, contacts: %s", billingClientId, contracts);
        boolean active = activeContracts > 0;
        log.info("Agreement status for billing client id {}: {}",
                billingClientId, active ? "active" : "inactive");
        if (!active && config.isAgreementActive()) {
            reportDeactivated(config);
        }
        config.setAgreementActive(active);
        config.setSynchronizedAt(Instant.now(clockService.getUtc()));
    }

    private void reportDeactivated(BillingPartnerConfig config) {
        log.warn("BillingPartnerConfig {} is being deactivated.", config);
        Counter deactivatedAgreementCounter = hotelAgreementDictionary.hasClientId(config.getBillingClientId()) ?
                BillingTransactionMeters.billingAgreementDeactivatedHotels :
                BillingTransactionMeters.billingAgreementDeactivatedCommon;
        deactivatedAgreementCounter.increment();
    }

    @Override
    public String getName() {
        return "BillingAgreementSync";
    }

    @Override
    public Duration getCurrentProcessingDelay() {
        Instant synchronizeSinceTs = getMaxSynchronizedAt();
        ProcessingTasksInfo tasksInfo = partnerConfigRepository
                .findOldestTimestampForAgreementSynchronization(synchronizeSinceTs);
        return ProcessingDelaysHelper.getDelay(tasksInfo, getName(), synchronizeSinceTs);
    }

    private Instant getMaxSynchronizedAt() {
        return Instant.now(clockService.getUtc()).minus(properties.getSynchronizationInterval());
    }
}
