package ru.yandex.travel.orders.services.finances.billing;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.cache.HotelAgreementDictionary;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@Service
public class BillingPartnerConfigSynchronizer extends AbstractTaskKeyProvider<String> {

    private static final String SINGLETON_BILLING_PARTNER_CONFIG_SYNC_TASK = "singletonBillingPartnerConfigSyncTask";

    private final BillingPartnerConfigRepository billingPartnerConfigRepository;
    private final HotelAgreementDictionary hotelAgreementDictionary;
    private final AtomicLong billingPartnerSyncTimestamp = new AtomicLong(0);

    public BillingPartnerConfigSynchronizer(BillingPartnerConfigRepository billingPartnerConfigRepository,
                                            HotelAgreementDictionary hotelAgreementDictionary) {
        this.billingPartnerConfigRepository = billingPartnerConfigRepository;
        this.hotelAgreementDictionary = hotelAgreementDictionary;
    }

    @TransactionMandatory
    public void insertMissingClientIdsIntoBillingPartnerConfig(String taskId) {
        long tableCacheUpdateTimestamp = hotelAgreementDictionary.getTableCacheUpdateTimestamp();
        if (hotelAgreementDictionary.isReady() && tableCacheUpdateTimestamp > billingPartnerSyncTimestamp.get()) {
            Set<Long> billingClientIdSet = hotelAgreementDictionary.getAllClientIds();
            List<Long> existingClientIds = billingPartnerConfigRepository.findAllClientIds();
            billingClientIdSet.removeAll(existingClientIds);
            if (!billingClientIdSet.isEmpty()) {
                log.info("{} new billingClientIds are going to be inserted", billingClientIdSet.size());
                List<BillingPartnerConfig> newPartnerConfigs = billingClientIdSet.stream().map(billingClientId -> {
                    log.info("BillingPartnerConfig for id={} will be inserted", billingClientId);
                    BillingPartnerConfig partnerConfig = new BillingPartnerConfig();
                    partnerConfig.setBillingClientId(billingClientId);
                    partnerConfig.setAgreementActive(false);
                    partnerConfig.setGenerateTransactions(true);
                    partnerConfig.setExportToYt(true);
                    partnerConfig.setSynchronizeAgreement(true);
                    partnerConfig.setCreatedAt(Instant.now());
                    return partnerConfig;
                }).collect(Collectors.toList());
                billingPartnerConfigRepository.saveAll(newPartnerConfigs);
            }
            billingPartnerSyncTimestamp.set(tableCacheUpdateTimestamp);
        }
    }

    @Override
    @TransactionMandatory
    public Collection<String> getPendingTaskKeys(int maxResultSize) {
        Set<String> excludedIds = getLockedTaskKeys();
        if (excludedIds.contains(SINGLETON_BILLING_PARTNER_CONFIG_SYNC_TASK)) {
            // shouldn't happen in case of a single threaded task processor
            return List.of();
        }
        return List.of(SINGLETON_BILLING_PARTNER_CONFIG_SYNC_TASK);
    }

    @Override
    @TransactionMandatory
    public long getPendingTasksCount() {
        return 1;
    }
}
