package ru.yandex.travel.orders.services.finances.billing;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.repository.BillingTransactionRepository.NO_EXCLUDE_IDS;

@RequiredArgsConstructor
@Slf4j
public class BillingTransactionActCommitter implements FinancialEventProcessor {
    static final Duration ACCEPTABLE_TX_ACT_COMMIT_LAG = Duration.ofHours(24);

    private final BillingTransactionRepository billingTransactionRepository;
    private final BillingApiClient billingApiClient;
    private final ClockService clockService;
    private final Duration actCommitDelay;

    @TransactionMandatory
    public Collection<Long> getTransactionIdsWaitingForCommit(Set<Long> excludeIds, Integer maxResultSize) {
        Pageable pageable = PageRequest.of(0, maxResultSize);
        Collection<Long> excludeFilter = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return billingTransactionRepository.findIdsReadyForActCommit(getMaxPayoutAt(), getMaxActAt(), excludeFilter,
                pageable);
    }

    @TransactionMandatory
    public long countTransactionsWaitingForCommit(Set<Long> excludeIds) {
        Collection<Long> excludeFilter = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return billingTransactionRepository.countReadyForActCommit(getMaxPayoutAt(), getMaxActAt(), excludeFilter);
    }

    @TransactionMandatory
    public void processTransactionWaitingForCommit(Long txId) {
        BillingTransaction tx = billingTransactionRepository.getOne(txId);
        if (tx.isActCommitted()) {
            log.warn("Billing transaction {} has already been committed", txId);
            return;
        }
        try (NestedMdc ignored = entityMdcOrEmpty(tx)) {
            log.debug("Submitting {} to act", tx.getDescription());
            // it's impossible to fix already generated acts so we should be careful not to submit transactions
            // into old acts (it's not clear what happens in this case);
            // for simplicity this committer always commits only into acts that includes today or yesterday payments
            // (the fix is important for paused payments/partners)
            BillingTransactionGenerator.fixPastEventActDate(tx, clockService.getUtc(), ACCEPTABLE_TX_ACT_COMMIT_LAG);

            billingApiClient.updatePayment(tx.getServiceId(), tx.getYtId(), tx.getAccountingActAt());
            tx.setActCommitted(true);
            tx.setActCommittedAt(Instant.now(clockService.getUtc()));
            BillingTransactionMeters.billingTransactionsActCommitted.increment();
        }
    }

    private NestedMdc entityMdcOrEmpty(BillingTransaction tx) {
        if (tx.getSourceFinancialEvent() != null) {
            return NestedMdc.forOptionalEntity(tx.getSourceFinancialEvent().getOrderItem());
        } else {
            return NestedMdc.empty();
        }
    }

    @Override
    public String getName() {
        return "BillingTxActCommitter";
    }

    @TransactionMandatory
    @Override
    public Duration getCurrentProcessingDelay() {
        // the returned value isn't an actual timestamp but a period based on the query params and the oldest record
        ProcessingTasksInfo tasksInfo = billingTransactionRepository
                .findMaxDelaySecondsOfTxReadyForActCommit(getMaxPayoutAt(), getMaxActAt());
        if (tasksInfo.getOldestProcessAt() == null) {
            boolean isEmptyQueue = tasksInfo.getCount() == 0;
            return ProcessingDelaysHelper.getDefaultDelay(isEmptyQueue, getName());
        }
        return Duration.ofSeconds(tasksInfo.getOldestProcessAt().getEpochSecond());
    }

    private Instant getMaxPayoutAt() {
        return Instant.now(clockService.getUtc()).minus(actCommitDelay);
    }

    private Instant getMaxActAt() {
        return Instant.now(clockService.getUtc());
    }
}
