package ru.yandex.travel.orders.services.finances.billing;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;

import ru.yandex.travel.commons.lang.MoneyUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.BillingTransactionKind;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.INSURANCE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.POSTPAY;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.PROMO_CODE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.SBERBANK;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.YANDEX;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.YANDEX_MONEY;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.COST;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.COST_INSURANCE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.FEE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.REWARD;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.REWARD_INSURANCE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.YANDEX_ACCOUNT_COST_WITHDRAW;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.YANDEX_ACCOUNT_REWARD_WITHDRAW;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.YANDEX_ACCOUNT_TOPUP;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionType.PAYMENT;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionType.REFUND;
import static ru.yandex.travel.orders.repository.FinancialEventRepository.NO_EXCLUDE_IDS;

@Slf4j
@RequiredArgsConstructor
public class BillingTransactionGenerator implements FinancialEventProcessor {
    private final FinancialEventRepository financialEventRepository;
    private final BillingTransactionRepository billingTransactionRepository;
    private final ClockService clockService;

    @TransactionMandatory
    public Collection<Long> fetchEventIdsWaitingProcessing(Set<Long> excludeIds, int maxResultSize) {
        Pageable pageable = PageRequest.of(0, maxResultSize, Sort.by("id").ascending());
        // empty id list doesn't work in PG
        Collection<Long> excludeFilter = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return financialEventRepository.findUnprocessedIds(excludeFilter, pageable);
    }

    @TransactionMandatory
    public long countEventsWaitingProcessing(Set<Long> excludeIds) {
        // empty id list doesn't work in PG
        Collection<Long> excludeFilter = excludeIds != null && !excludeIds.isEmpty() ? excludeIds : NO_EXCLUDE_IDS;
        return financialEventRepository.countUnprocessedIds(excludeFilter);
    }

    @TransactionMandatory
    public void processPendingEvent(Long eventId) {
        FinancialEvent event = financialEventRepository.getOne(eventId);
        try (NestedMdc ignored = NestedMdc.forOptionalEntity(event.getOrderItem())) {
            if (event.isProcessed()) {
                log.warn("The event has already been processed, skipping it; {}", event.getDescription());
                return;
            }
            log.info("Processing {}", event.getDescription());
            ensureOriginalEvent(event);
            for (BillingTransaction tx : createBillingTransactions(event)) {
                log.info("Generated {}", tx.getDescription());
                billingTransactionRepository.save(tx);
            }
            event.setProcessed(true);

            // Extranet gets information based on updated at and it needs info about billing transactions
            Optional.ofNullable(event.getOrder()).ifPresent(o -> o.setUpdatedAt(Instant.now()));

            BillingTransactionMeters.billingTransactionsGenerated.increment();
        } catch (Exception e) {
            log.warn("Got an exception while processing {}; e.msg={}", event.getDescription(), e.getMessage());
            throw e;
        }
    }

    @Override
    public String getName() {
        return "BillingTxGenerator";
    }

    @TransactionMandatory
    @Override
    public Duration getCurrentProcessingDelay() {
        ProcessingTasksInfo tasksInfo = financialEventRepository.findOldestUnprocessedTimestamp();
        return ProcessingDelaysHelper.getDelay(tasksInfo, getName(), Instant.now(clockService.getUtc()));
    }

    void ensureOriginalEvent(FinancialEvent event) {
        if (!event.isRefund() || event.getOriginalEvent() != null) {
            // nothing to link
            return;
        }
        var financialEventType = event.isYandexAccountTopup() ? FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT :
                FinancialEventType.PAYMENT;
        // refund events will require payment->refund transactions matching, linking them here
        FinancialEvent sourcePayment;
        if (event.isYandexAccountTopup() && event.getOrderItem() == null) {
            // external plus topup
            sourcePayment =
                    financialEventRepository.findFirstByOrderPrettyIdAndTypeOrderByIdAsc(event.getOrderPrettyId(),
                    financialEventType);
        } else {
            sourcePayment = financialEventRepository.findFirstByOrderItemAndTypeOrderByIdAsc(event.getOrderItem(),
                    financialEventType);
        }
        log.info("Linking refund {} with source payout {}", event.getDescription(), sourcePayment.getDescription());
        event.setOriginalEvent(sourcePayment);
    }

    private String getLastTrustPaymentId(Order order) {
        return order.getPayments().stream()
                .map(Payment::getLastPaidAttempt)
                .filter(Objects::nonNull)
                .max(Comparator.comparing(Invoice::getCreatedAt))
                .orElse(order.getCurrentInvoice())
                .getTrustPaymentId();
    }

    List<BillingTransaction> createBillingTransactions(FinancialEvent fe) {
        switch (fe.getType()) {
            case PAYMENT:
            case MANUAL_PAYMENT: {
                String trustPaymentId = getLastTrustPaymentId(fe.getOrder());
                // todo(tlg-13): it's better to sum to the result transactions and compare the result to the event total
                ensureOnlyValues(fe, fe.getPartnerAmount(), fe.getFeeAmount(), fe.getPromoCodePartnerAmount(),
                        fe.getPromoCodeFeeAmount(), fe.getPartnerFeeAmount(),
                        fe.getPlusPartnerAmount(), fe.getPlusFeeAmount(),
                        fe.getPostPayUserAmount(), fe.getPostPayPartnerPayback()
                );
                Preconditions.checkArgument(fe.getPartnerAmount() != null,
                        "Partner amount shouldn't be empty for confirmation events");
                Preconditions.checkArgument(fe.getFeeAmount() != null,
                        "Fee amount shouldn't be empty for confirmation events");
                List<BillingTransaction> transactions = new ArrayList<>();
                BillingTransactionPaymentSystemType paySystemType = getProperBillingPaymentSystemType(fe);
                if (!isNullOrZero(fe.getPartnerAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            COST, paySystemType, fe.getPartnerAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getFeeAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            REWARD, paySystemType, fe.getFeeAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPromoCodePartnerAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            COST, PROMO_CODE, fe.getPromoCodePartnerAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPromoCodeFeeAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            REWARD, PROMO_CODE, fe.getPromoCodeFeeAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPartnerFeeAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            FEE, YANDEX, fe.getPartnerFeeAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPlusPartnerAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            YANDEX_ACCOUNT_COST_WITHDRAW, PROMO_CODE, fe.getPlusPartnerAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPlusFeeAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            YANDEX_ACCOUNT_REWARD_WITHDRAW, PROMO_CODE, fe.getPlusFeeAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPostPayPartnerPayback())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            REWARD, POSTPAY, fe.getPostPayPartnerPayback(), null, null));
                }
                return transactions;
            }
            case INSURANCE_PAYMENT: {
                String trustPaymentId = getLastTrustPaymentId(fe.getOrder());
                ensureOnlyValues(fe, fe.getPartnerAmount(), fe.getFeeAmount());
                Preconditions.checkArgument(fe.getPartnerAmount() != null,
                        "Partner amount shouldn't be empty for insurance confirmation events");
                Preconditions.checkArgument(fe.getFeeAmount() != null,
                        "Fee amount shouldn't be empty for insurance confirmation events");
                List<BillingTransaction> transactions = new ArrayList<>();
                transactions.add(createTransaction(fe, PAYMENT, COST_INSURANCE, INSURANCE, fe.getPartnerAmount(),
                        trustPaymentId, null));
                transactions.add(createTransaction(
                        fe, PAYMENT, REWARD_INSURANCE, INSURANCE, fe.getFeeAmount(), trustPaymentId, null));
                // todo(tlg-13): it seems like promo money are lost here (aren't actually supported right now; be aware)
                return transactions;
            }
            case REFUND:
            case MANUAL_REFUND: {
                String trustPaymentId = getLastTrustPaymentId(fe.getOrder());
                ensureOnlyValues(fe, fe.getPartnerRefundAmount(), fe.getFeeRefundAmount(),
                        fe.getPromoCodePartnerRefundAmount(), fe.getPromoCodeFeeRefundAmount(),
                        fe.getPartnerFeeAmount(), fe.getPartnerFeeRefundAmount(),
                        fe.getPlusPartnerRefundAmount(), fe.getPlusFeeRefundAmount(),
                        fe.getPostPayUserRefund(), fe.getPostPayPartnerRefund()
                );
                Preconditions.checkArgument(fe.getPartnerRefundAmount() != null,
                        "Partner refund amount shouldn't be empty for refund events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getFeeRefundAmount() != null,
                        "Fee refund amount shouldn't be empty for refund events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getOriginalEvent() != null,
                        "Refund financial event must have an associated payment event; event %s", fe.getId());
                List<BillingTransaction> transactions = new ArrayList<>();
                BillingTransactionPaymentSystemType paySystemType = getProperBillingPaymentSystemType(fe);
                if (!isNullOrZero(fe.getPartnerRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            COST, paySystemType, fe.getPartnerRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), COST, paySystemType)));
                }
                if (!isNullOrZero(fe.getFeeRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            REWARD, paySystemType, fe.getFeeRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), REWARD, paySystemType)));
                }
                if (!isNullOrZero(fe.getPromoCodePartnerRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            COST, PROMO_CODE, fe.getPromoCodePartnerRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), COST, PROMO_CODE)));
                }
                if (!isNullOrZero(fe.getPromoCodeFeeRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            REWARD, PROMO_CODE, fe.getPromoCodeFeeRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), REWARD, PROMO_CODE)));
                }
                if (!isNullOrZero(fe.getPartnerFeeAmount())) {
                    transactions.add(createTransaction(fe, PAYMENT,
                            FEE, YANDEX, fe.getPartnerFeeAmount(), trustPaymentId, null));
                }
                if (!isNullOrZero(fe.getPartnerFeeRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            FEE, YANDEX, fe.getPartnerFeeRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), FEE, YANDEX)));
                }
                if (!isNullOrZero(fe.getPlusPartnerRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            YANDEX_ACCOUNT_COST_WITHDRAW, PROMO_CODE, fe.getPlusPartnerRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), YANDEX_ACCOUNT_COST_WITHDRAW, PROMO_CODE)));
                }
                if (!isNullOrZero(fe.getPlusFeeRefundAmount())) {
                    transactions.add(createTransaction(fe, REFUND,
                            YANDEX_ACCOUNT_REWARD_WITHDRAW, PROMO_CODE, fe.getPlusFeeRefundAmount(), trustPaymentId,
                            findTransactionForEvent(fe.getOriginalEvent(), YANDEX_ACCOUNT_REWARD_WITHDRAW, PROMO_CODE)));
                }
                if (!isNullOrZero(fe.getPostPayPartnerRefund())) {
                    transactions.add(createTransaction(fe, REFUND,
                            REWARD, POSTPAY, fe.getPostPayPartnerRefund(), null,
                            findTransactionForEvent(fe.getOriginalEvent(), REWARD, POSTPAY)));
                }
                return transactions;
            }
            case INSURANCE_REFUND: {
                String trustPaymentId = getLastTrustPaymentId(fe.getOrder());
                ensureOnlyValues(fe, fe.getPartnerRefundAmount(), fe.getFeeRefundAmount());
                Preconditions.checkArgument(fe.getPartnerRefundAmount() != null,
                        "Partner refund amount shouldn't be empty for refund events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getFeeRefundAmount() != null,
                        "Fee refund amount shouldn't be empty for refund events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getOriginalEvent() != null,
                        "Refund financial event must have an associated payment event; event %s", fe.getId());
                List<BillingTransaction> transactions = new ArrayList<>();
                transactions.add(createTransaction(fe, REFUND, COST_INSURANCE, INSURANCE, fe.getPartnerRefundAmount(),
                        trustPaymentId, findTransactionForEvent(fe.getOriginalEvent(), COST_INSURANCE, INSURANCE)));
                transactions.add(createTransaction(fe, REFUND, REWARD_INSURANCE, INSURANCE, fe.getFeeRefundAmount(),
                        trustPaymentId, findTransactionForEvent(fe.getOriginalEvent(), REWARD_INSURANCE, INSURANCE)));
                return transactions;
            }
            case YANDEX_ACCOUNT_TOPUP_PAYMENT: {
                ensureOnlyValues(fe, fe.getPlusTopupAmount());
                Preconditions.checkArgument(fe.getPlusTopupAmount() != null,
                        "Topup amount shouldn't be empty for topup events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getPaymentScheme() == FinancialEventPaymentScheme.HOTELS,
                        "Only hotels service supports yandex account topups; event %s, scheme %s",
                        fe.getId(), fe.getPaymentScheme());
                var trustPaymentId = Objects.requireNonNull(fe.getTrustPaymentId(), "explicit trust payment id");
                return List.of(createTransaction(fe, PAYMENT, YANDEX_ACCOUNT_TOPUP, PROMO_CODE, fe.getPlusTopupAmount(),
                        trustPaymentId, null));
            }
            case YANDEX_ACCOUNT_TOPUP_REFUND: {
                ensureOnlyValues(fe, fe.getPlusTopupRefundAmount());
                Preconditions.checkArgument(fe.getPlusTopupRefundAmount() != null,
                        "Topup amount shouldn't be empty for topup refund events; event %s", fe.getId());
                Preconditions.checkArgument(fe.getPaymentScheme() == FinancialEventPaymentScheme.HOTELS,
                        "Only hotels service supports yandex account topups; event %s, scheme %s",
                        fe.getId(), fe.getPaymentScheme());
                var trustPaymentId = Objects.requireNonNull(fe.getTrustPaymentId(), "explicit trust payment id");
                return List.of(createTransaction(fe, REFUND, YANDEX_ACCOUNT_TOPUP, PROMO_CODE,
                        fe.getPlusTopupRefundAmount(), trustPaymentId,
                        findTransactionForEvent(fe.getOriginalEvent(), YANDEX_ACCOUNT_TOPUP, PROMO_CODE)));
            }
            default:
                throw new RuntimeException("Unsupported financial event type: " + fe.getType());
        }
    }

    void ensureOnlyValues(FinancialEvent event, Money... values) {
        Money valuesSum = null;
        for (Money value : values) {
            valuesSum = MoneyUtils.safeAdd(valuesSum, value);
        }
        Money eventSum = event.getTotalAmount();

        // negative values aren't allowed and can break the total sum comparison
        event.ensureNoNegativeValues();
        Preconditions.checkArgument(Objects.equals(eventSum, valuesSum),
                "Illegal cost breakdown. Expected event values should sum up to %s but got %s; event %s",
                eventSum, valuesSum, event);
    }

    private BillingTransactionPaymentSystemType getProperBillingPaymentSystemType(FinancialEvent fe) {
        BillingTransactionPaymentSystemType paySystemType = null;
        switch (fe.getPaymentScheme()) {
            case HOTELS_POSTPAY:
                paySystemType = POSTPAY;
                break;
            case HOTELS:
            case SUBURBAN:
                paySystemType = YANDEX_MONEY;
                break;
            case TRAINS:
                paySystemType = SBERBANK;
                break;
            default:
                break;
        }
        Preconditions.checkArgument(paySystemType != null, "Payment system type is undefined for event; event=%s", fe);
        return paySystemType;
    }

    private BillingTransaction findTransactionForEvent(FinancialEvent event,
                                                       BillingTransactionPaymentType paymentType,
                                                       BillingTransactionPaymentSystemType paymentSystemType) {
        BillingTransaction tx = billingTransactionRepository
                .findBySourceFinancialEventAndPaymentTypeAndPaymentSystemType(event, paymentType, paymentSystemType);
        Preconditions.checkState(tx != null, "No billing transaction for a previous %s payment of event %s, " +
                        "payment system type - %s",
                paymentType, event.getId(), paymentSystemType);
        return tx;
    }

    BillingTransaction createTransaction(
            FinancialEvent fe,
            BillingTransactionType transactionType,
            BillingTransactionPaymentType paymentType,
            BillingTransactionPaymentSystemType paymentSystemType,
            Money value,
            String trustPaymentId,
            BillingTransaction originalTransaction
    ) {
        // we apply strict validation rules here to early detect and prevent data import errors on the later stages
        // (YT -> Billing)
        Preconditions.checkArgument(fe != null, "No financialEvent");
        String eventDesc = String.format("event %s, type %s", fe.getId(), transactionType);
        FinancialEventPaymentScheme paymentScheme = fe.getPaymentScheme();
        Preconditions.checkArgument(transactionType != null, "No transactionType; %s", eventDesc);
        Preconditions.checkArgument(paymentType != null, "No paymentType; %s", eventDesc);
        Preconditions.checkArgument(paymentSystemType != null, "No paymentSystemType; %s", eventDesc);
        Preconditions.checkArgument(value != null, "No value; %s", eventDesc);
        Preconditions.checkArgument(paymentScheme != null, "No paymentScheme; %s", eventDesc);
        Preconditions.checkArgument(value.isPositiveOrZero(), "Operation value can't be negative; %s, value %s",
                eventDesc, value);
        if (fe.isRefund() && paymentType != FEE) {
            Preconditions.checkArgument(originalTransaction != null,
                    "No originalTransaction for a refund transaction; %s", eventDesc);
            Preconditions.checkArgument(REFUND == transactionType,
                    "Event transactionType mismatch; expected %s, got %s, %s",
                    REFUND, transactionType, eventDesc);
            Preconditions.checkArgument(PAYMENT == originalTransaction.getTransactionType(),
                    "Original event transactionType mismatch; expected %s, got %s, %s",
                    PAYMENT, originalTransaction.getTransactionType(), eventDesc);
            Preconditions.checkArgument(paymentType == originalTransaction.getPaymentType(),
                    "Original event paymentType mismatch; expected %s, got %s, %s",
                    paymentType, originalTransaction.getPaymentType(), eventDesc);
            Preconditions.checkArgument(paymentSystemType == originalTransaction.getPaymentSystemType(),
                    "Original event paymentSystemType mismatch; expected %s, got %s, %s",
                    paymentSystemType, originalTransaction.getPaymentSystemType(), eventDesc);
            Preconditions.checkArgument(originalTransaction.getValue() != null,
                    "Original transaction without money value: %s", originalTransaction);
            Preconditions.checkArgument(value.isLessThanOrEqualTo(originalTransaction.getValue()),
                    "Refund value %s can't exceed original tx value %s", value, originalTransaction.getValue());
            Preconditions.checkArgument(Objects.equals(originalTransaction.getServiceId(),
                            paymentScheme.getServiceId()),
                    "Original event serviceId mismatch; expected %s, got %s, %s",
                    paymentScheme.getServiceId(), originalTransaction.getServiceId(), eventDesc);
        } else if (fe.isPayment() || paymentType == FEE) {
            // todo(tlg-13): this code looks buggy, won't work for fee refund; they're unsupported atm though.
            Preconditions.checkArgument(PAYMENT == transactionType,
                    "transactionType mismatch; expected %s, got %s, %s",
                    PAYMENT, transactionType, eventDesc);
            Preconditions.checkArgument(originalTransaction == null,
                    "No originalTransaction expected for payments; %s, original tx %s",
                    fe.getDescription(), originalTransaction);
        } else {
            throw new IllegalArgumentException("Either PAYMENT or REFUND events are expected here but got " + fe);
        }
        Preconditions.checkArgument(fe.getBillingClientId() != null, "No billingClientId; %s", eventDesc);
        Preconditions.checkArgument(fe.getOrderPrettyId() != null, "No orderPrettyId; %s", eventDesc);
        Preconditions.checkArgument(fe.getPayoutAt() != null, "No payoutAt; %s", eventDesc);
        Preconditions.checkArgument(fe.getAccountingActAt() != null, "No accountingActAt; %s", eventDesc);

        Instant now = Instant.now(clockService.getUtc());

        BillingTransactionKind kind = BillingTransactionKind.PAYMENT;
        if (!isNullOrZero(fe.getPostPayUserAmount())
                || !isNullOrZero(fe.getPostPayPartnerPayback())
                || !isNullOrZero(fe.getPostPayUserRefund())
                || !isNullOrZero(fe.getPostPayPartnerRefund())) {
            Preconditions.checkArgument(paymentScheme.equals(FinancialEventPaymentScheme.HOTELS_POSTPAY),
                    "Postpay event should have postpay scheme, has %s", paymentScheme);
            Preconditions.checkArgument(paymentSystemType.equals(POSTPAY),
                    "Postpay event should have postpay system type, has %s", paymentSystemType);
            // for now postpay cannot be mixed with prepay
            Preconditions.checkArgument(isNullOrZero(fe.getPartnerAmount()), "Postpay event should have no partner amount");
            Preconditions.checkArgument(isNullOrZero(fe.getFeeAmount()), "Postpay event should have no fee amount");
            Preconditions.checkArgument(isNullOrZero(fe.getPartnerRefundAmount()), "Postpay event should have no partner refund");
            Preconditions.checkArgument(isNullOrZero(fe.getFeeRefundAmount()), "Postpay event should have no fee refund");
            kind = BillingTransactionKind.INCOME;
        } else {
            Preconditions.checkArgument(!Strings.isNullOrEmpty(trustPaymentId), "No trustPaymentId; %s", eventDesc);
        }

        BillingTransaction tx = new BillingTransaction();
        tx.setKind(kind);
        tx.setSourceFinancialEvent(fe);
        tx.setOriginalTransaction(originalTransaction);
        tx.setServiceId(paymentScheme.getServiceId());
        tx.setTransactionType(transactionType);
        tx.setPaymentType(paymentType);
        tx.setPaymentSystemType(paymentSystemType);
        tx.setPartnerId(fe.getBillingClientId());
        tx.setValue(value);
        tx.setTrustPaymentId(trustPaymentId);
        tx.setClientId(0L); // unused billing field
        tx.setServiceOrderId(fe.getOrderPrettyId());
        tx.setCreatedAt(now);
        tx.setPayoutAt(fe.getPayoutAt());
        tx.setAccountingActAt(fe.getAccountingActAt());
        tx.setExportedToYt(false);
        tx.setActCommitted(false);

        fixPastEventSchedule(tx);

        return tx;
    }

    private boolean isNullOrZero(Money money) {
        return money == null || money.isZero();
    }

    /**
     * We don't want to export records into tables from the past
     * as Billing won't read them and the transactions will get lost.
     */
    private void fixPastEventSchedule(BillingTransaction tx) {
        Duration acceptableLag = Duration.ofMinutes(5);
        fixPastEventPayoutDate(tx, clockService.getUtc(), acceptableLag);
        fixPastEventActDate(tx, clockService.getUtc(), acceptableLag);
    }

    static void fixPastEventPayoutDate(BillingTransaction tx, Clock clock, Duration acceptableLag) {
        Instant now = Instant.now(clock);
        // some small mismatches are still ok
        if (tx.getPayoutAt().isBefore(now.minus(acceptableLag))) {
            log.warn("{} payoutAt is in the past - {}, setting it to {}",
                    tx.getDescription(), tx.getPayoutAt(), now);
            tx.setPayoutAt(now);
        }
    }

    static void fixPastEventActDate(BillingTransaction tx, Clock clock, Duration acceptableLag) {
        Instant now = Instant.now(clock);
        // some small mismatches are still ok
        if (tx.getAccountingActAt().isBefore(now.minus(acceptableLag))) {
            log.warn("{} accountingActAt is in the past - {}, setting it to {}",
                    tx.getDescription(), tx.getAccountingActAt(), now);
            tx.setAccountingActAt(now);
        }
    }
}
