package ru.yandex.travel.orders.services.finances.billing;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

import static java.util.stream.Collectors.toList;

@RequiredArgsConstructor
@Slf4j
public class BillingTransactionYtExporter implements FinancialEventProcessor {
    static final String SINGLETON_BILLING_TX_EXPORT_TASK = "singletonBillingTxExportTask";
    static final Duration ACCEPTABLE_TX_EXPORT_LAG = Duration.ofHours(24);

    private final BillingTransactionYtTableClientProperties properties;
    private final BillingTransactionRepository billingTransactionRepository;
    private final BillingTransactionYtTableClient billingTransactionYtTableClient;
    private final ClockService clockService;

    @TransactionMandatory
    public Collection<String> getExportTasks(Set<String> excludedIds) {
        if (excludedIds.contains(SINGLETON_BILLING_TX_EXPORT_TASK)) {
            // shouldn't happen in case of a single threaded task processor
            return List.of();
        }
        if (countNotExportedTransactions() == 0) {
            return List.of();
        }
        return List.of(SINGLETON_BILLING_TX_EXPORT_TASK);
    }

    @TransactionMandatory
    public long countNotExportedTransactions() {
        return billingTransactionRepository.countReadyForExport();
    }

    @TransactionMandatory
    public void startExportTask(String taskId) {
        log.info("Starting a new billing transactions batch export to yt");
        Preconditions.checkArgument(SINGLETON_BILLING_TX_EXPORT_TASK.equals(taskId));
        List<BillingTransaction> transactions = getBatchToUpload(properties.getBatchSize());
        if (transactions.isEmpty()) {
            log.warn("No billing transactions to export");
            return;
        }
        for (BillingTransaction tx : transactions) {
            try (NestedMdc ignored = entityMdcOrEmpty(tx)) {
                log.info("Submitting {} to YT", tx.getDescription());
                // exporting transactions into old tables may result in hanged partner payouts;
                // here we export transactions only into YT transaction tables for today or yesterday payments
                // (there are no payouts from the future right now; the fix is important for paused payments/partners)
                BillingTransactionGenerator.fixPastEventPayoutDate(tx, clockService.getUtc(), ACCEPTABLE_TX_EXPORT_LAG);
            }
        }
        LocalDate transactionsDate = BillingHelper.toBillingDate(transactions.get(0).getPayoutAt()).toLocalDate();
        billingTransactionYtTableClient.exportTransactions(transactionsDate, transactions);
        Instant exportedToYtAt = Instant.now(clockService.getUtc());
        transactions.forEach(tx -> {
            tx.setExportedToYt(true);
            tx.setExportedToYtAt(exportedToYtAt);
            // processes that follow by updated at need to know ytId
            Optional.ofNullable(tx.getSourceFinancialEvent()).map(FinancialEvent::getOrder)
                    .ifPresent(o -> o.setUpdatedAt(Instant.now()));
        });
        BillingTransactionMeters.billingTransactionsExportedToYt.increment(transactions.size());
    }

    private NestedMdc entityMdcOrEmpty(BillingTransaction tx) {
        if (tx.getSourceFinancialEvent() != null) {
            return NestedMdc.forOptionalEntity(tx.getSourceFinancialEvent().getOrderItem());
        } else {
            return NestedMdc.empty();
        }
    }

    @Override
    public String getName() {
        return "BillingTxYtExporter";
    }

    @Override
    @TransactionMandatory
    public Duration getCurrentProcessingDelay() {
        ProcessingTasksInfo tasksInfo = billingTransactionRepository.findOldestTimestampReadyForYtExport();
        return ProcessingDelaysHelper.getDelay(tasksInfo, getName(), Instant.now(clockService.getUtc()));
    }

    List<BillingTransaction> getBatchToUpload(int batchSize) {
        List<BillingTransaction> transactions = billingTransactionRepository.findReadyForExport(
                PageRequest.of(0, batchSize, Sort.by("ytId").ascending()));
        if (transactions.isEmpty()) {
            return transactions;
        }
        LocalDate dayToExport = BillingHelper.toBillingDate(transactions.get(0).getPayoutAt()).toLocalDate();
        return transactions.stream()
                // using 'takeWhile' instead of simple 'filter' because of tricky manual support issues, see the tests
                .takeWhile(tx -> BillingHelper.toBillingDate(tx.getPayoutAt()).toLocalDate().equals(dayToExport))
                .collect(toList());
    }
}
