package ru.yandex.travel.orders.services.finances.billing;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.services.finances.tasks.FinancialEventProcessor;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.utils.ClockService;

@RequiredArgsConstructor
@Slf4j
public class BillingTransactionYtIdGenerator implements FinancialEventProcessor {
    static final String SINGLETON_BILLING_TX_YT_ID_GENERATOR_TASK = "singletonBillingTxYtIdGeneratorTask";

    private final BillingTransactionRepository billingTransactionRepository;
    private final ClockService clockService;

    @TransactionMandatory
    public Collection<String> getYtIdBulkGenerationTaskIds(Set<String> excludeIds) {
        if (excludeIds.contains(SINGLETON_BILLING_TX_YT_ID_GENERATOR_TASK)) {
            // shouldn't happen in case of a single threaded task processor
            return List.of();
        }
        if (countYdIdGenerationTasks() == 0) {
            return List.of();
        }
        return List.of(SINGLETON_BILLING_TX_YT_ID_GENERATOR_TASK);
    }

    @TransactionMandatory
    public long countYdIdGenerationTasks() {
        return billingTransactionRepository.countReadyTransactionsWithoutYtId(getMaxPayoutAtForExport());
    }

    @TransactionMandatory
    public void bulkGenerateNewYtIds(String taskId) {
        log.info("Starting a new batch generation of billing transaction YT ids");
        Preconditions.checkArgument(SINGLETON_BILLING_TX_YT_ID_GENERATOR_TASK.equals(taskId));
        int total = billingTransactionRepository.generateNewYtIdsForReadyTransactions(getMaxPayoutAtForExport());
        log.info("{} new YT ids generated for ready for export billing transactions", total);
        if (total == 0) {
            log.warn("No ready for export billing transactions without YT id found");
        }
        BillingTransactionMeters.billingTransactionsYtIdGenerated.increment(total);
    }

    @Override
    public String getName() {
        return "BillingTxYtIdGenerator";
    }

    @Override
    @TransactionMandatory
    public Duration getCurrentProcessingDelay() {
        Instant maxPayoutAt = getMaxPayoutAtForExport();
        ProcessingTasksInfo stats = billingTransactionRepository.findOldestTimestampReadyForYtIdGeneration(maxPayoutAt);
        return ProcessingDelaysHelper.getDelay(stats, getName(), maxPayoutAt);
    }

    private Instant getMaxPayoutAtForExport() {
        return Instant.now(clockService.getUtc());
    }
}
