package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;

import static ru.yandex.travel.orders.services.finances.providers.LegacySplitHelper.calculateLegacyPenaltySplit;

/**
 * {@link DirectHotelBillingFinancialDataProvider} and {@link DolphinFinancialDataProvider}
 * share a lot of common functionality, while the code was apparently created independently. This class can be a base
 * for extracting those common parts in future refactorings.
 */
@Slf4j
public abstract class AbstractHotelFinancialDataProvider implements FinancialDataProvider {

    @Value
    @AllArgsConstructor
    protected static class PenaltyAndRefund {
        Money penalty;
        Money refund;
    }

    private MoneySourceSplit getSplitForPostPaidItem(HotelOrderItem orderItem, Money price) {
        Preconditions.checkArgument(orderItem.getBillingPartnerAgreement() instanceof DirectHotelBillingPartnerAgreement,
                "Postpay is supported only for DirectHotelBillingPartnerAgreement");
        BigDecimal rate = ((DirectHotelBillingPartnerAgreement) orderItem.getBillingPartnerAgreement())
                .getOrderConfirmedRate();
        Money reverseFee = price.multiply(rate);
        return MoneySourceSplit.fromPostPayOnly(price, reverseFee);
    }

    protected MoneySourceSplit getTargetSplit(HotelOrderItem orderItem) {
        MoneyMarkup itemMarkup = orderItem.getTotalMoneyMarkup().map(ProviderHelper::ensureMoneyScale);
        if (orderItem.isPostPaid()) {
            Preconditions.checkArgument(itemMarkup.getTotal().isZero(),
                    "Total markup for post pay item should be zero");
            Money price = orderItem.getHotelItinerary().getRealHotelPrice();
            return getSplitForPostPaidItem(orderItem, price);
        }
        Money discount = getDiscount(orderItem);
        MoneyMarkup finalBalanceMarkup = fitMarkupToActualPrice(itemMarkup,
                ProviderHelper.ensureMoneyScale(orderItem.getHotelItinerary().getRealHotelPrice())
                        .subtract(discount));
        return MoneySourceSplit.fromMarkupAndPromo(finalBalanceMarkup, discount);
    }

    protected MoneySourceSplit getTargetRefundSplit(HotelOrderItem orderItem, ServiceBalance serviceBalance,
                                                    Money penalty, EMoneyRefundMode moneyRefundMode) {
        if (orderItem.isPostPaid()) {
            Preconditions.checkArgument(penalty.isPositiveOrZero(), "Negative penalty: %s", penalty);
            return getSplitForPostPaidItem(orderItem, penalty);
        }
        return calculateLegacyPenaltySplit(serviceBalance, penalty, moneyRefundMode);
    }

    /**
     * Returns the discount of the order item. If the actual price is less, returns the actual price.
     * <p>
     * actual price (especially for dolphin) may slightly differ,
     * we need to adjust the applied promo code money in case it exceeds the actual price (e.g. by a few
     * roubles)
     */
    protected Money getDiscount(HotelOrderItem orderItem) {
        Money discount = orderItem.getTotalDiscount();

        Money totalUserPrice = orderItem.getHotelItinerary().getFiscalPrice();
        Preconditions.checkArgument(discount.isLessThanOrEqualTo(totalUserPrice),
                "Discount cannot exceed the total order item price: discount %s, price %s", discount, totalUserPrice);

        Money totalActualPrice = orderItem.getHotelItinerary().getRealHotelPrice();
        return ComparatorUtils.min(discount, totalActualPrice);
    }

    /**
     * When actual price doesn't match the fiscal price (with kopecks or dolphin managed to changed it
     * during booking), we need to restore repay it somehow to the hotel. Those adjustments are miniscule,
     * but we need to work with money correctly.
     *
     * @implNote this is the most important method. ATM it's marked as paid from withdrawn from yandex account
     * (though, it's not actually charged from user's account).
     */
    @VisibleForTesting
    static MoneyMarkup fitMarkupToActualPrice(MoneyMarkup markup, Money actualPrice) {
        markup.ensureNoNegativeValues();
        Money userTotal = markup.getTotal();
        Money diff = actualPrice.subtract(userTotal);
        // trying to balance the money out with 'pluses' first
        Money newCard = markup.getCard();
        Money newYandexAccount = markup.getYandexAccount().add(diff);
        if (newYandexAccount.isNegative()) {
            newCard = newCard.subtract(newYandexAccount.negate());
            newYandexAccount = Money.zero(newYandexAccount.getCurrency());
        }
        MoneyMarkup newMarkup = MoneyMarkup.builder()
                .card(newCard)
                .yandexAccount(newYandexAccount)
                .build();
        if (!diff.isZero()) {
            log.info("Actual price ({}) difference of {} results in a changed markup: original {}, corrected {}",
                    actualPrice, diff, markup, newMarkup);
        }
        return newMarkup;
    }

    protected Money maybePatchPenaltyForManualRefund(ServiceBalance balance, Money penalty, Money refund,
                                                     EMoneyRefundMode moneyRefundMode) {
        if (balance.getOverallBalance().getTotalPromo().isPositive() && moneyRefundMode == EMoneyRefundMode.MRM_PROPORTIONAL) {
            var totalInvoice = penalty.add(refund);
            var newPenalty = ProviderHelper.ensureMoneyScale(penalty.add(penalty.multiply(balance.getOverallBalance().getTotalPromo().getNumberStripped()).divide(totalInvoice.getNumberStripped())));
            log.info("Patching penalty from {} to {} for proportional refund", penalty, newPenalty);
            return newPenalty;
        }
        return penalty;
    }
}
