package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Collections;
import java.util.List;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.BronevikBillingPartnerAgreement;
import ru.yandex.travel.orders.grpc.helpers.OrderProtoUtils;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.orderitem.bronevik.BronevikProperties;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.services.finances.providers.LegacySplitHelper.calculateLegacyPenaltySplit;

// TODO(mokosha): Пока что не генерируем фин ивенты, так как нет айдишников биллинга
//@Service
@RequiredArgsConstructor
@Slf4j
public class BronevikFinancialDataProvider extends AbstractHotelFinancialDataProvider {

    private final FinancialEventRepository financialEventRepository;
    private final FullMoneySplitCalculator fullSplitCalculator;
    private final ClockService clockService;
    private final BronevikProperties bronevikProperties;

    @Override
    public List<EServiceType> getServiceTypes() {
        return List.of(EServiceType.PT_BRONEVIK_HOTEL);
    }

    @Override
    public List<FinancialEvent> onConfirmation(OrderItem orderItem, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() == null || orderItem.getOrder().getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
            return onPaymentImpl(orderItem, enablePromoFee);
        }

        return Collections.emptyList();
    }

    @Override
    public List<FinancialEvent> onExtraPayment(OrderItem orderItem, Payment extraPayment, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() != null && orderItem.getOrder().getPaymentSchedule().getState() != EPaymentState.PS_FULLY_PAID) {
            throw new IllegalStateException("Extra payments are not allowed for incomplete orders");
        }

        return onPaymentImpl(orderItem, enablePromoFee);
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee) {
        var penaltyAndRefund = getPenaltyAndRefund((BronevikOrderItem) orderItem);

        return onRefund(orderItem, moneyRefundMode, enablePromoFee, penaltyAndRefund.getPenalty(), penaltyAndRefund.getRefund(), orderItem.getRefundedAt());
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee,
                                         Money penalty, Money refund, Instant refundedAt) {
        BronevikOrderItem bronevikOrderItem = (BronevikOrderItem) orderItem;
        BronevikBillingPartnerAgreement agreement = bronevikOrderItem.getBillingPartnerAgreement();

        List<FinancialEvent> existingEvents = financialEventRepository.findAllByOrderItem(orderItem);
        ServiceBalance balance = new ServiceBalance(existingEvents,
                bronevikOrderItem.getHotelItinerary().getFiscalPrice().getCurrency());
        MoneySourceSplit finalSourceView = calculateLegacyPenaltySplit(balance, penalty, moneyRefundMode);

        return balance.setBalanceTo(
                finalSourceView,
                agreement.getConfirmRate(),
                () -> createEvent(bronevikOrderItem, FinancialEventType.PAYMENT),
                () -> createEvent(bronevikOrderItem, FinancialEventType.REFUND),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    @Override
    public List<FinancialEvent> onPartialRefund(OrderItem orderItem, OrderRefund partialRefund, Boolean enablePromoFee) {
        throw new UnsupportedOperationException("partial refund for hotel order items is not supported");
    }

    @Override
    public List<FinancialEvent> onPaymentScheduleFullyPaid(OrderItem orderItem, PaymentSchedule schedule, Boolean enablePromoFee) {
        return onPaymentImpl(orderItem, enablePromoFee);
    }

    @Override
    public VatType getYandexFeeVat(OrderItem orderItem) {
        return OrderProtoUtils.fromEVat(bronevikProperties.getBilling().getFiscalVat());
    }

    private List<FinancialEvent> onPaymentImpl(OrderItem orderItem, Boolean enablePromoFee) {
        BronevikOrderItem bronevikOrderItem = (BronevikOrderItem) orderItem;
        BronevikBillingPartnerAgreement agreement = bronevikOrderItem.getBillingPartnerAgreement();

        Preconditions.checkArgument(!orderItem.getFiscalItems().isEmpty(),
                "No fiscal items for proper money calculation");

        MoneySourceSplit targetSplit = getTargetSplit(bronevikOrderItem);

        ServiceBalance balance = new ServiceBalance(financialEventRepository.findAllByOrderItem(bronevikOrderItem),
                bronevikOrderItem.getHotelItinerary().getFiscalPrice().getCurrency());

        return balance.increaseBalanceTo(
                targetSplit,
                agreement.getConfirmRate(),
                () -> createEvent(bronevikOrderItem, FinancialEventType.PAYMENT),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    private FinancialEvent createEvent(BronevikOrderItem orderItem, FinancialEventType eventType) {
        FinancialEvent event = FinancialEvent.create(orderItem, eventType, FinancialEventPaymentScheme.HOTELS,
                Instant.now(clockService.getUtc()));
        BronevikBillingPartnerAgreement agreement = orderItem.getBillingPartnerAgreement();
        event.setBillingClientId(agreement.getBillingClientId());
        event.setBillingContractId(agreement.getBillingContractId());
        Instant lastDayOfMonth = BillingHelper.getNextBillingMondayAfterCheckout(orderItem);
        event.setPayoutAt(lastDayOfMonth);
        event.setAccountingActAt(lastDayOfMonth);

        return event;
    }

    private Money getPenaltyMoney(BronevikOrderItem orderItem) {
        RefundInfo refundInfo = orderItem.getHotelItinerary().getRefundInfo();
        BigDecimal penalty = ProviderHelper.ensureMoneyScale(new BigDecimal(refundInfo.getPenalty().getAmount()));

        return Money.of(penalty, refundInfo.getPenalty().getCurrency());
    }

    private PenaltyAndRefund getPenaltyAndRefund(BronevikOrderItem orderItem) {
        RefundInfo refundInfo = orderItem.getHotelItinerary().getRefundInfo();
        BigDecimal penalty = ProviderHelper.ensureMoneyScale(new BigDecimal(refundInfo.getPenalty().getAmount()));
        BigDecimal refund = ProviderHelper.ensureMoneyScale(new BigDecimal(refundInfo.getRefund().getAmount()));
        return new PenaltyAndRefund(
                Money.of(penalty, refundInfo.getPenalty().getCurrency()),
                Money.of(refund, refundInfo.getRefund().getCurrency()));
    }
}
