package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.money.CurrencyUnit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.services.finances.FinancialEventService.DEFAULT_MONEY_REFUND_MODE;

@Service
@RequiredArgsConstructor
@Slf4j
public class DirectHotelBillingFinancialDataProvider extends AbstractHotelFinancialDataProvider {
    private static final Duration PAYOUT_DELAY = Duration.ofDays(3);
    private static final long POSTPAY_PAYOUT_DELAY = 1;

    private final ClockService clockService;
    private final FullMoneySplitCalculator fullSplitCalculator;
    private final FinancialEventRepository financialEventRepository;

    @Override
    public List<EServiceType> getServiceTypes() {
        return List.of(EServiceType.PT_TRAVELLINE_HOTEL, EServiceType.PT_BNOVO_HOTEL);
    }

    @Override
    public List<FinancialEvent> onConfirmation(OrderItem orderItem, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() == null || orderItem.getOrder().getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
            HotelOrderItem hotelOrderItem = (HotelOrderItem) orderItem;
            return onPaymentImpl(hotelOrderItem, getPayoutAtForOrderItem(hotelOrderItem), enablePromoFee);
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public List<FinancialEvent> onUpdateFinancialEventsWithoutBalanceChanges(OrderItem orderItem,
                                                                             Boolean enablePromoFee,
                                                                             Boolean inferBillingIdsFromEvents) {
        HotelOrderItem hotelOrderItem = (HotelOrderItem) orderItem;
        var payoutInstant = getPayoutAtForOrderItem(hotelOrderItem);
        Preconditions.checkArgument(hotelOrderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                "Unexpected order item type");
        DirectHotelBillingPartnerAgreement agreement =
                ((DirectHotelBillingPartnerAgreementProvider) hotelOrderItem).getAgreement();

        var currency = hotelOrderItem.getHotelItinerary().getFiscalPrice().getCurrency();
        var billingClientId = agreement.getFinancialClientId();
        var billingContractId = agreement.getFinancialContractId();

        if (inferBillingIdsFromEvents) {
            var billingClientIdsWithPositiveBalance = financialEventRepository.findAllByOrderItem(hotelOrderItem)
                    .stream()
                    .collect(Collectors.groupingBy(FinancialEvent::getBillingClientId))
                    .entrySet()
                    .stream()
                    .filter(x -> new ServiceBalance(x.getValue(), currency).getOverallBalance().getTotal().isPositive())
                    .map(Map.Entry::getKey)
                    .collect(Collectors.toUnmodifiableList());
            if (!billingClientIdsWithPositiveBalance.isEmpty()) {
                Preconditions.checkState(billingClientIdsWithPositiveBalance.size() == 1,
                        "Expected single billing client id with positive balance");
                var referenceFinancialEvent = financialEventRepository.findAllByOrderItem(hotelOrderItem)
                        .stream()
                        .filter(x -> Objects.equals(x.getBillingClientId(), billingClientIdsWithPositiveBalance.get(0)))
                        .findFirst();
                Preconditions.checkState(referenceFinancialEvent.isPresent());
                billingClientId = referenceFinancialEvent.get().getBillingClientId();
                billingContractId = referenceFinancialEvent.get().getBillingContractId();
            }
        }

        ServiceBalance balance = new ServiceBalance(financialEventRepository.findAllByOrderItem(hotelOrderItem), currency);

        Long finalBillingClientId = billingClientId;
        Long finalBillingContractId = billingContractId;
        return balance.recalculateBalance(
                agreement.getOrderConfirmedRate(),
                () -> createRefundEvent(hotelOrderItem, finalBillingClientId, finalBillingContractId, payoutInstant),
                fullSplitCalculator,
                enablePromoFee);
    }

    private List<FinancialEvent> onPaymentImpl(OrderItem orderItem, Instant payoutInstant, Boolean enablePromoFee) {
        HotelOrderItem hotelOrderItem = (HotelOrderItem) orderItem;
        Preconditions.checkArgument(hotelOrderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                "Unexpected order item type");
        DirectHotelBillingPartnerAgreement agreement =
                ((DirectHotelBillingPartnerAgreementProvider) hotelOrderItem).getAgreement();

        MoneySourceSplit targetSplit = getTargetSplit(hotelOrderItem);

        ServiceBalance balance = new ServiceBalance(financialEventRepository.findAllByOrderItem(hotelOrderItem),
                hotelOrderItem.getHotelItinerary().getFiscalPrice().getCurrency());
        return balance.increaseBalanceTo(
                targetSplit,
                agreement.getOrderConfirmedRate(),
                () -> createPaymentEvent(hotelOrderItem, agreement, payoutInstant),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    @Override
    public List<FinancialEvent> onExtraPayment(OrderItem orderItem, Payment extraPayment, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() == null || orderItem.getOrder().getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
            Instant payoutInstant = orderItem.isPostPaid()
                    ? getPayoutAtForOrderItem((HotelOrderItem) orderItem)
                    : getPayoutAtForPayment(extraPayment);
            return onPaymentImpl(orderItem, payoutInstant, enablePromoFee);
        } else {
            throw new IllegalStateException("Extra payments are not allowed for incomplete orders");
        }
    }

    @VisibleForTesting
    List<FinancialEvent> onRefund(OrderItem orderItem, Boolean enablePromoFee) {
        return onRefund(orderItem, DEFAULT_MONEY_REFUND_MODE, enablePromoFee);
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee) {
        var penaltyAndRefund = getPenaltyAndRefund((HotelOrderItem) orderItem);
        return onRefund(orderItem, moneyRefundMode, enablePromoFee, penaltyAndRefund.getPenalty(), penaltyAndRefund.getRefund(), orderItem.getRefundedAt());
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode,
                                         Boolean enablePromoFee, Money penalty, Money refund, Instant refundedAt) {
        HotelOrderItem hotelOrderItem = (HotelOrderItem) orderItem;
        Preconditions.checkArgument(orderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                "Unexpected order item type");
        DirectHotelBillingPartnerAgreement agreement =
                ((DirectHotelBillingPartnerAgreementProvider) orderItem).getAgreement();

        List<FinancialEvent> existingEvents = financialEventRepository.findAllByOrderItem(orderItem);
        ServiceBalance balance = new ServiceBalance(existingEvents,
                hotelOrderItem.getHotelItinerary().getFiscalPrice().getCurrency());

        penalty = maybePatchPenaltyForManualRefund(balance, penalty, refund, moneyRefundMode);
        MoneySourceSplit finalSourceView = getTargetRefundSplit(hotelOrderItem, balance, penalty, moneyRefundMode);
        return balance.setBalanceTo(
                finalSourceView,
                agreement.getOrderRefundedRate(),
                () -> createPaymentEvent(hotelOrderItem, agreement, getPayoutAtForOrderItem(hotelOrderItem)),
                () -> createRefundEvent(hotelOrderItem, agreement, refundedAt),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    @Override
    public List<FinancialEvent> onPartialRefund(OrderItem orderItem, OrderRefund partialRefund, Boolean enablePromoFee) {
        throw new UnsupportedOperationException("partial refund for hotel order items is not supported");
    }

    @Override
    public List<FinancialEvent> onPaymentScheduleFullyPaid(OrderItem orderItem, PaymentSchedule schedule, Boolean enablePromoFee) {
        return onPaymentImpl((HotelOrderItem) orderItem, getPayoutAtForPayment(schedule), enablePromoFee);
    }

    @Override
    public VatType getYandexFeeVat(OrderItem orderItem) {
        Preconditions.checkArgument(orderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                "Unexpected order item type");
        DirectHotelBillingPartnerAgreement agreement =
                ((DirectHotelBillingPartnerAgreementProvider) orderItem).getAgreement();
        return agreement.getVatType();
    }

    public <T extends HotelOrderItem & DirectHotelBillingPartnerAgreementProvider> List<FinancialEvent> issueFullCorrectionEvent(
            T orderItem, Instant payoutAt, Boolean enablePromoFee) {
        DirectHotelBillingPartnerAgreement agreement = orderItem.getAgreement();
        List<FinancialEvent> existingEvents = financialEventRepository.findAllByOrderItem(orderItem);
        CurrencyUnit currency = orderItem.getHotelItinerary().getFiscalPrice().getCurrency();
        ServiceBalance balance = new ServiceBalance(existingEvents, currency);
        MoneySourceSplit zeroSourceSplit = MoneySourceSplit.zero(currency);

        return balance.decreaseBalanceTo(zeroSourceSplit,
                agreement.getOrderRefundedRate(),
                () -> createRollbackEvent(orderItem, agreement, payoutAt),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    private FinancialEvent createRollbackEvent(HotelOrderItem orderItem, DirectHotelBillingPartnerAgreement agreement,
                                               Instant payoutAt) {
        FinancialEvent event = FinancialEvent.create(orderItem, FinancialEventType.REFUND,
                FinancialEventPaymentScheme.HOTELS,
                Instant.now(clockService.getUtc()));
        event.setBillingClientId(agreement.getFinancialClientId());
        event.setBillingContractId(agreement.getFinancialContractId());
        setFinEventDates(event, payoutAt, orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate());
        return event;
    }

    private FinancialEvent createPaymentEvent(HotelOrderItem orderItem, DirectHotelBillingPartnerAgreement agreement,
                                              Instant payoutInstant) {
        FinancialEvent event = FinancialEvent.create(orderItem,
                FinancialEventType.PAYMENT,
                orderItem.isPostPaid() ? FinancialEventPaymentScheme.HOTELS_POSTPAY : FinancialEventPaymentScheme.HOTELS,
                Instant.now(clockService.getUtc()));
        event.setBillingClientId(agreement.getFinancialClientId());
        event.setBillingContractId(agreement.getFinancialContractId());
        setFinEventDates(event, payoutInstant, orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate());
        return event;
    }

    private FinancialEvent createRefundEvent(HotelOrderItem orderItem, DirectHotelBillingPartnerAgreement agreement, Instant refundedAt) {
        return createRefundEvent(orderItem, agreement.getFinancialClientId(), agreement.getFinancialContractId(), refundedAt);
    }

    private FinancialEvent createRefundEvent(HotelOrderItem orderItem, Long billingClientId, Long billingContractId, Instant refundedAt) {
        FinancialEvent event = FinancialEvent.create(orderItem,
                FinancialEventType.REFUND,
                orderItem.isPostPaid() ? FinancialEventPaymentScheme.HOTELS_POSTPAY : FinancialEventPaymentScheme.HOTELS,
                Instant.now(clockService.getUtc()));
        event.setBillingClientId(billingClientId);
        event.setBillingContractId(billingContractId);

        Instant payoutAt = getPayoutAtForOrderItem(orderItem);
        if (refundedAt.isAfter(payoutAt)) {
            payoutAt = refundedAt;
        }
        setFinEventDates(event, payoutAt, orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate());
        return event;
    }

    private void setFinEventDates(FinancialEvent event, Instant payoutAt, LocalDate checkOut) {
        event.setPayoutAt(payoutAt);

        // accounting act is set to checkout date or to payout date
        Instant accountingActAt = BillingHelper.getBillingDayStart(checkOut);
        if (accountingActAt.isBefore(event.getPayoutAt())) {
            accountingActAt = event.getPayoutAt();
        }
        event.setAccountingActAt(accountingActAt);
    }

    private PenaltyAndRefund getPenaltyAndRefund(HotelOrderItem orderItem) {
        RefundInfo refundInfo = orderItem.getHotelItinerary().getRefundInfo();
        BigDecimal penalty = ProviderHelper.ensureMoneyScale(new BigDecimal(refundInfo.getPenalty().getAmount()));
        BigDecimal refund = ProviderHelper.ensureMoneyScale(new BigDecimal(refundInfo.getRefund().getAmount()));
        return new PenaltyAndRefund(
                Money.of(penalty, refundInfo.getPenalty().getCurrency()),
                Money.of(refund, refundInfo.getRefund().getCurrency()));
    }

    private Instant getPayoutAtForOrderItem(HotelOrderItem orderItem) {
        Instant payoutAt = orderItem.getConfirmedAt().plus(PAYOUT_DELAY);
        if (orderItem.isPostPaid()) {
            var orderDetails = orderItem.getHotelItinerary().getOrderDetails();
            var zoneId = orderDetails.getHotelTimeZoneId();
            if (zoneId == null) {
                zoneId = ZoneId.systemDefault();
            }
            payoutAt = orderDetails.getCheckoutDate().plusDays(POSTPAY_PAYOUT_DELAY).atStartOfDay(zoneId).toInstant();
        }
        return BillingHelper.getBillingDayStart(payoutAt);
    }

    private Instant getPayoutAtForPayment(Payment payment) {
        return BillingHelper.getBillingDayStart(payment.getClosedAt().plus(PAYOUT_DELAY));
    }
}
