package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.util.Collections;
import java.util.List;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.hotels.common.orders.RefundReason;
import ru.yandex.travel.hotels.common.partners.dolphin.model.DolphinRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.model.FixedPercentFeeRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.DolphinRefundRulesBuilder;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.DolphinBillingPartnerAgreement;
import ru.yandex.travel.orders.grpc.helpers.OrderProtoUtils;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.orderitem.dolphin.DolphinProperties;
import ru.yandex.travel.utils.ClockService;

import static ru.yandex.travel.orders.services.finances.FinancialEventService.DEFAULT_MONEY_REFUND_MODE;
import static ru.yandex.travel.orders.services.finances.providers.LegacySplitHelper.calculateLegacyPenaltySplit;

@Service
@RequiredArgsConstructor
@Slf4j
public class DolphinFinancialDataProvider extends AbstractHotelFinancialDataProvider {

    private final DolphinFinancialDataProviderProperties dolphinBillingProperties;
    private final ClockService clockService;
    private final FullMoneySplitCalculator fullSplitCalculator;
    private final FinancialEventRepository financialEventRepository;
    private final DolphinProperties dolphinProperties;

    @Override
    public List<EServiceType> getServiceTypes() {
        return List.of(EServiceType.PT_DOLPHIN_HOTEL);
    }

    @Override
    public List<FinancialEvent> onConfirmation(OrderItem orderItem, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() == null || orderItem.getOrder().getPaymentSchedule().getState() == EPaymentState.PS_FULLY_PAID) {
            return onConfirmationOrExtra(orderItem, enablePromoFee);
        }

        return Collections.emptyList();
    }

    @Override
    public List<FinancialEvent> onExtraPayment(OrderItem orderItem, Payment extraPayment, Boolean enablePromoFee) {
        if (orderItem.getOrder().getPaymentSchedule() != null && orderItem.getOrder().getPaymentSchedule().getState() != EPaymentState.PS_FULLY_PAID) {
            throw new IllegalStateException("Extra payments are not allowed for incomplete orders");
        }

        return onConfirmationOrExtra(orderItem, enablePromoFee);
    }

    @VisibleForTesting
    List<FinancialEvent> onRefund(OrderItem orderItem) {
        return onRefund(orderItem, DEFAULT_MONEY_REFUND_MODE, false);
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee) {
        var penaltyAndRefund = getPenaltyAndRefund((DolphinOrderItem) orderItem);
        return onRefund(orderItem, moneyRefundMode, enablePromoFee, penaltyAndRefund.getPenalty(), penaltyAndRefund.getRefund(), orderItem.getRefundedAt());
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee, Money penalty, Money refund, Instant refundedAt) {
        DolphinOrderItem dolphinOrderItem = (DolphinOrderItem) orderItem;
        DolphinBillingPartnerAgreement agreement = dolphinOrderItem.getBillingPartnerAgreement();
        ServiceBalance balance = new ServiceBalance(financialEventRepository.findAllByOrderItem(orderItem),
                orderItem.getOrder().getCurrency());

        penalty = maybePatchPenaltyForManualRefund(balance, penalty, refund, moneyRefundMode);
        MoneySourceSplit penaltySourceSplit = calculateLegacyPenaltySplit(balance, penalty, moneyRefundMode);
        return balance.setBalanceTo(
                penaltySourceSplit,
                agreement.getConfirmRate(),
                () -> createEvent(dolphinOrderItem, FinancialEventType.PAYMENT),
                () -> createEvent(dolphinOrderItem, FinancialEventType.REFUND),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    @Override
    public List<FinancialEvent> onPartialRefund(OrderItem orderItem, OrderRefund partialRefund, Boolean enablePromoFee) {
        throw new UnsupportedOperationException("partial refund for hotel order items is not supported");
    }

    @Override
    public List<FinancialEvent> onPaymentScheduleFullyPaid(OrderItem orderItem, PaymentSchedule schedule, Boolean enablePromoFee) {
        return onConfirmationOrExtra(orderItem, enablePromoFee);
    }

    @Override
    public VatType getYandexFeeVat(OrderItem orderItem) {
        return OrderProtoUtils.fromEVat(dolphinProperties.getBilling().getFiscalVat());
    }

    private List<FinancialEvent> onConfirmationOrExtra(OrderItem orderItem, Boolean enablePromoFee) {
        DolphinOrderItem dolphinOrderItem = (DolphinOrderItem) orderItem;

        DolphinBillingPartnerAgreement agreement = dolphinOrderItem.getBillingPartnerAgreement();

        Preconditions.checkArgument(!orderItem.getFiscalItems().isEmpty(),
                "No fiscal items for proper money calculation");

        MoneySourceSplit targetSplit = getTargetSplit(dolphinOrderItem);

        ServiceBalance balance = new ServiceBalance(financialEventRepository.findAllByOrderItem(orderItem),
                dolphinOrderItem.getHotelItinerary().getRealHotelPrice().getCurrency());
        return balance.increaseBalanceTo(
                targetSplit,
                agreement.getConfirmRate(),
                () -> createEvent(dolphinOrderItem, FinancialEventType.PAYMENT),
                fullSplitCalculator,
                enablePromoFee
        );
    }

    private FinancialEvent createEvent(DolphinOrderItem orderItem, FinancialEventType eventType) {
        FinancialEvent event = FinancialEvent.create(orderItem, eventType, FinancialEventPaymentScheme.HOTELS,
                Instant.now(clockService.getUtc()));
        DolphinBillingPartnerAgreement agreement = orderItem.getBillingPartnerAgreement();
        event.setBillingClientId(agreement.getBillingClientId());
        event.setBillingContractId(agreement.getBillingContractId());
        Instant lastDayOfMonth = BillingHelper.getLastBillingDayOfCheckoutMonth(orderItem);
        event.setPayoutAt(lastDayOfMonth);
        event.setAccountingActAt(lastDayOfMonth);
        return event;
    }

    private PenaltyAndRefund getPenaltyAndRefund(DolphinOrderItem dolphinOrderItem) {
        if (dolphinOrderItem.getHotelItinerary().getRefundInfo() != null &&
                dolphinOrderItem.getHotelItinerary().getRefundInfo().getReason() == RefundReason.OPERATOR) {
            log.info("Order is being refunded (may be partially) by operator. This overrides refund rules," +
                    "so we use refund info to calculate refund penalty instead of recalculating it from scratch");
            BigDecimal penalty = ProviderHelper.ensureMoneyScale(
                    new BigDecimal(dolphinOrderItem.getHotelItinerary().getRefundInfo().getPenalty().getAmount()));
            BigDecimal refund = ProviderHelper.ensureMoneyScale(
                    new BigDecimal(dolphinOrderItem.getHotelItinerary().getRefundInfo().getRefund().getAmount()));
            return new PenaltyAndRefund(
                    Money.of(penalty, dolphinOrderItem.getHotelItinerary().getRefundInfo().getPenalty().getCurrency()),
                    Money.of(refund, dolphinOrderItem.getHotelItinerary().getRefundInfo().getRefund().getCurrency()));
        }
        Instant checkinMoment = dolphinOrderItem.getItinerary().getCheckInMoment();
        DolphinRefundParams refundParams = dolphinOrderItem.getHotelItinerary().getRefundParams();
        if (refundParams == null) {
            refundParams = new FixedPercentFeeRefundParams(dolphinBillingProperties.getRefundPenalties());
        }
        Money actualPrice = ProviderHelper.ensureMoneyScale(dolphinOrderItem.getHotelItinerary().getRealHotelPrice());
        // re-calculating the penalties because of possible actual price mismatch
        BigDecimal totalPrice = actualPrice.getNumberStripped().setScale(2, RoundingMode.HALF_UP);
        RefundRules refundRules = DolphinRefundRulesBuilder.build(
                Instant.now(clockService.getUtc()),
                checkinMoment,
                totalPrice,
                actualPrice.getCurrency().getCurrencyCode(),
                refundParams
        );

        // applying the effective penalty (see RefundCalculationService.calculateRefundForHotelItem)
        RefundRule rule = refundRules.getRuleAtInstant(clockService.getUtc().instant());
        Preconditions.checkNotNull(rule, "Unable to get cancellation penalty for order");

        Money penaltyAmount;
        switch (rule.getType()) {
            case FULLY_REFUNDABLE:
                penaltyAmount = Money.zero(actualPrice.getCurrency());
                break;
            case REFUNDABLE_WITH_PENALTY:
                penaltyAmount = rule.getPenalty();
                break;
            case NON_REFUNDABLE:
                penaltyAmount = Money.of(totalPrice, actualPrice.getCurrency());
                break;
            default:
                throw new IllegalStateException("Unsupported cancellation type: " + rule.getType());
        }

        var refundAmount = Money.of(totalPrice.subtract(penaltyAmount.getNumberStripped()), actualPrice.getCurrency());

        return new PenaltyAndRefund(
                ProviderHelper.ensureMoneyScale(penaltyAmount),
                ProviderHelper.ensureMoneyScale(refundAmount));
    }
}
