package ru.yandex.travel.orders.services.finances.providers;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EServiceType;

@Service
public class FinancialDataProviderRegistry implements ApplicationContextAware, InitializingBean {
    private Map<EServiceType, FinancialDataProvider> providers;
    private ApplicationContext applicationContext;

    public Set<EServiceType> getSupportedServiceTypes() {
        return ImmutableSet.copyOf(providers.keySet());
    }

    public boolean supportsProvider(EServiceType serviceType) {
        return providers.containsKey(serviceType);
    }

    public FinancialDataProvider getProvider(EServiceType serviceType) {
        FinancialDataProvider provider = providers.get(serviceType);
        if (provider == null) {
            throw new IllegalArgumentException("No financial provider for service type: " + serviceType);
        }
        return provider;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Map<EServiceType, FinancialDataProvider> providers = new HashMap<>();
        for (var bean : applicationContext.getBeansOfType(FinancialDataProvider.class).values()) {
            for (var t : bean.getServiceTypes()) {
                if (providers.containsKey(t)) {
                    throw new BeanInitializationException("Duplicate FinancialDataProvider for service type " + t);
                }
                providers.put(t, bean);
            }
        }
        this.providers = ImmutableMap.copyOf(providers);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
