package ru.yandex.travel.orders.services.finances.providers;

import javax.money.CurrencyUnit;

import com.google.common.annotations.VisibleForTesting;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.NonNull;
import lombok.Value;
import org.javamoney.moneta.Money;

/**
 * @see FullMoneySplitCalculator
 */
@Value
@Builder
@AllArgsConstructor
class FullMoneySplit {
    @NonNull MoneySplit userMoney;
    @NonNull MoneySplit plusMoney;
    @NonNull MoneySplit promoMoney;
    @NonNull MoneySplit userPostPayMoney;

    // legacy impl for tests only
    @VisibleForTesting
    FullMoneySplit(MoneySplit userMoney, MoneySplit promoMoney) {
        this(userMoney, MoneySplit.zero(userMoney.getCurrency()), promoMoney, MoneySplit.zero(userMoney.getCurrency()));
    }

    public static FullMoneySplit fromUserMoneyOnly(MoneySplit partnerSplit) {
        return FullMoneySplit.builder()
                .userMoney(partnerSplit)
                .plusMoney(MoneySplit.zero(partnerSplit.getCurrency()))
                .promoMoney(MoneySplit.zero(partnerSplit.getCurrency()))
                .userPostPayMoney(MoneySplit.zero(partnerSplit.getCurrency()))
                .build();
    }

    public String describe() {
        return String.format("%s [user cost = %s, user reward = %s, " +
                        "plus cost = %s, plus reward = %s, " +
                        "promo cost = %s, promo reward = %s" +
                        "user post pay = %s, user post pay reward = %s]",
                getTotal(), userMoney.getPartner(), userMoney.getFee(),
                plusMoney.getPartner(), plusMoney.getFee(),
                promoMoney.getPartner(), promoMoney.getFee(),
                userPostPayMoney.getPartner(), userPostPayMoney.getFee()
        );
    }

    public Money getTotal() {
        return userMoney.add(plusMoney).add(promoMoney).add(userPostPayMoney).getTotal();
    }

    public MoneySourceSplit getSourceView() {
        return MoneySourceSplit.builder()
                .user(userMoney.getTotal())
                .plus(plusMoney.getTotal())
                .promo(promoMoney.getTotal())
                .userPostPay(userPostPayMoney.getTotal().subtract(userPostPayMoney.getReverseFee()))
                .partnerReverseFee(userPostPayMoney.getReverseFee())
                .build();
    }

    public MoneySplit getPartnerView() {
        return new MoneySplit(
                userMoney.getPartner()
                        .add(plusMoney.getPartner())
                        .add(promoMoney.getPartner())
                        .add(userPostPayMoney.getPartner()),
                userMoney.getFee()
                        .add(plusMoney.getFee())
                        .add(promoMoney.getFee())
                        .add(userPostPayMoney.getFee()),
                userMoney.getReverseFee()
                        .add(plusMoney.getReverseFee())
                        .add(promoMoney.getReverseFee())
                        .add(userPostPayMoney.getReverseFee())
        );
    }

    public static FullMoneySplit zero(CurrencyUnit currencyUnit) {
        return new FullMoneySplit(
                MoneySplit.zero(currencyUnit),
                MoneySplit.zero(currencyUnit),
                MoneySplit.zero(currencyUnit),
                MoneySplit.zero(currencyUnit)
        );
    }

    public FullMoneySplit subtract(FullMoneySplit other) {
        return new FullMoneySplit(
                this.userMoney.subtract(other.userMoney),
                this.plusMoney.subtract(other.plusMoney),
                this.promoMoney.subtract(other.promoMoney),
                this.userPostPayMoney.subtract(other.userPostPayMoney)
        );
    }

    public FullMoneySplit add(FullMoneySplit other) {
        return new FullMoneySplit(
                this.userMoney.add(other.userMoney),
                this.plusMoney.add(other.plusMoney),
                this.promoMoney.add(other.promoMoney),
                this.userPostPayMoney.add(other.userPostPayMoney)
        );
    }

    public FullMoneySplit extractNegativeValues() {
        return new FullMoneySplit(
                userMoney.extractNegativeValues(),
                plusMoney.extractNegativeValues(),
                promoMoney.extractNegativeValues(),
                userPostPayMoney.extractNegativeValues()
        );
    }

    public FullMoneySplit negate() {
        return new FullMoneySplit(
                userMoney.negate(),
                plusMoney.negate(),
                promoMoney.negate(),
                userPostPayMoney.negate()
        );
    }

    public void ensureNoNegativeValues() {
        if (!extractNegativeValues().getTotal().isZero()) {
            throw new IllegalStateException("No negative values is expected: " + describe());
        }
    }

    public static FullMoneySplit mergeByMin(FullMoneySplit a, FullMoneySplit b) {
        return new FullMoneySplit(
                MoneySplit.mergeByMin(a.userMoney, b.userMoney),
                MoneySplit.mergeByMin(a.plusMoney, b.plusMoney),
                MoneySplit.mergeByMin(a.promoMoney, b.promoMoney),
                MoneySplit.mergeByMin(a.userPostPayMoney, b.userPostPayMoney)
        );
    }
}
