package ru.yandex.travel.orders.services.finances.providers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;

import static ru.yandex.travel.orders.services.finances.FinancialEventService.DEFAULT_MONEY_REFUND_MODE;
import static ru.yandex.travel.orders.services.finances.providers.LegacySplitHelper.calculatePenaltyPromoMoney;

/**
 * The service helps financial data providers to distribute promo code money after the base partner 'cost' & 'reward'
 * money calculations are done.
 *
 * <p/>
 * <b>Model</b>:
 * <pre>
 *      money source perspective (user payments)
 *           ─────────────────────────────
 *          │   user  │  plus   │  promo  │
 *          ╔═════════╦═════════╦═════════╗ ────────
 *          ║   (1)   ║   (3)   ║   (5)   ║ partner │
 *    8600  ║  cost   ║  plus   ║  promo  ║ part    │ money
 *          ║         ║  cost   ║  cost   ║         │ destination
 *          ╠═════════╬═════════╬═════════╣ ────────│ perspective
 *          ║   (2)   ║   (4)   ║   (6)   ║ yandex  │ (partner payouts)
 *    1400  ║ reward  ║  plus   ║ promo   ║ fee     │
 *          ║         ║  reward ║ reward  ║ part    │
 *          ╚═════════╩═════════╩═════════╝ ────────
 *              6000     1000     3000
 * </pre>
 *
 * In this example we have an order with the price of 10 000 roubles.
 * The diagram shows how the paid money is converted to partner payouts.
 * At first the user decides how to cover to order cost.
 * <p/>There supported options are:
 * <ul>
 *     <li>real money (a.k.a. card / direct money)</li>
 *     <li>Yandex Plus points (1 point = 1 currency unit)</li>
 *     <li>promo codes (covered by travel marketing expenses)</li>
 * </ul>
 * In our example user pays only 6 000 roubles with card
 * while using a promo code for 3 000 roubles and 1000 Plus points.
 *
 * <p/>On the other hand we know what split partner expects for every order.
 * In case 10k order it's 8 600 roubles to pay to the partner and 1 400 stays with us as our commission.
 *
 * <p/>Our default strategy is to send all non-direct money to partner while keeping the 'real' money os the reward.
 * So in this example we get the payouts as follows:
 * <p/>
 * <b>Partner money (8 600 roubles)</b>
 * <ul>
 *     <li>4 600 - (1) cost</li>
 *     <li>1 000 - (3) plus cost</li>
 *     <li>3 000 - (5) promo cost</li>
 *     <li>(8 600 = 4 600 + 1 000 + 3 000)</li>
 * </ul>
 * <b>Yandex commission money (1 400 roubles)</b>
 * <ul>
 *     <li>1 400 - (2) reward</li>
 *     <li>0 - (4) plus reward</li>
 *     <li>0 - (6) promo reward</li>
 * </ul>
 */
@Component
@Slf4j
class FullMoneySplitCalculator {
    /**
     * The method works with 2 money perspectives: partner view (partner payout + agent fee) and
     * money source view (user + promo code). Given these values we need to distribute them in the following
     * manner:
     * <p/><pre>
     *         money source perspective
     *           ───────────────────
     *          │   user  │  promo  │
     *          ╔═════════╦═════════╗ ────────
     *          ║   (3)   ║   (1)   ║ partner │
     *    8600  ║  cost?  ║  promo  ║ part    │
     *          ║         ║  cost?  ║         │ partner
     *          ╠═════════╬═════════╣ ────────│ perspective
     *          ║   (4)   ║   (2)   ║ fee     │
     *    1400  ║ reward? ║ promo   ║ part    │
     *          ║         ║ reward? ║         │
     *          ╚═════════╩═════════╝ ────────
     *              7000     3000
     * </pre>
     * We decided to use promo code money to compensate the partner cost component first and then the reward component
     * if any promo code money is left.
     * <p>
     * In the example above we calculate the maximum possible 'promo cost' component (1) first. That let's use
     * to calculate 'promo reward' (2). These two values can be subtracted from the total partner money view
     * to get (3) and (4). All partner/source perspectives sum up to the same value (10000) and will look as follows:
     * <p/>
     * <pre>
     *           ───────────────────
     *          │   user  │  promo  │
     *          ╔═════════╦═════════╗ ────────
     *          ║   (3)   ║   (1)   ║ partner │
     *    8600  ║   5600  ║  3000   ║ part    │
     *          ║         ║         ║         │
     *          ╠═════════╬═════════╣ ────────│
     *          ║   (4)   ║   (2)   ║ fee     │
     *    1400  ║   1400  ║    0    ║ part    │
     *          ║         ║         ║         │
     *          ╚═════════╩═════════╝ ────────
     *              7000     3000
     * </pre>
     */
    public FullMoneySplit calculatePayment(MoneySourceSplit sourceView, MoneySplit partnerView) {
        Preconditions.checkArgument(partnerView.getTotal().equals(sourceView.getTotal()),
                "The specified money views don't match: source %s, destination %s",
                sourceView, partnerView);
        partnerView.ensureNoNegativeValues();
        sourceView.ensureNoNegativeValues();
        sourceView = sourceView.map(ProviderHelper::ensureMoneyScale);

        MoneySplit userSplit = partnerView;

        MoneySplit promoSplit = subtractMoneySource(userSplit, sourceView.getPromo());
        userSplit = userSplit.subtract(promoSplit);

        MoneySplit plusSplit = subtractMoneySource(userSplit, sourceView.getPlus());
        userSplit = userSplit.subtract(plusSplit);

        MoneySplit postPaySplit = subtractMoneySource(userSplit, sourceView.getUserPostPay(), true);
        userSplit = userSplit.subtract(postPaySplit);

        FullMoneySplit payment = FullMoneySplit.builder()
                .userMoney(userSplit)
                .plusMoney(plusSplit)
                .promoMoney(promoSplit)
                .userPostPayMoney(postPaySplit)
                .build();
        log.info("The requested money source view {} and partner view {} gets split into {}",
                sourceView.describe(), partnerView.describe(), payment.describe());

        Preconditions.checkArgument(payment.getTotal().isEqualTo(partnerView.getTotal()),
                "Payment split doesn't match the total payment sum: total %s, split %s",
                partnerView.getTotal(), payment);
        payment.ensureNoNegativeValues();
        return payment;
    }

    private MoneySplit subtractMoneySource(MoneySplit totalSplit, Money sourceMoney) {
        return subtractMoneySource(totalSplit, sourceMoney, false);
    }

    /**
     * Because reverse fee is going from partner view to source split (reverse),
     * we need to transfer it at specific moment
     */
    private MoneySplit subtractMoneySource(MoneySplit totalSplit, Money sourceMoney, boolean extractReverseFee) {
        // the specified source money covers the partner ('cost') component first
        Money sourcePartnerAmount = ComparatorUtils.min(sourceMoney, totalSplit.getPartner());
        Money sourceRemainder = sourceMoney.subtract(sourcePartnerAmount);

        // then trying to cover the 'fee' amount if any remainder source money is available
        Money sourceFeeAmount = ComparatorUtils.min(sourceRemainder, totalSplit.getFee());
        sourceRemainder = sourceRemainder.subtract(sourceFeeAmount);

        // reverse fee transfer is explicitly stated
        Money reverseFeeAmount = extractReverseFee ? totalSplit.getReverseFee() : Money.zero(sourceMoney.getCurrency());

        if (!sourceRemainder.isZero()) {
            throw new IllegalStateException(String.format("" +
                            "Some money from the specified source left after covering the payment values: " +
                            "payment %s, promo code remainder %s",
                    totalSplit.describe(), sourceRemainder));
        }

        return new MoneySplit(sourcePartnerAmount, sourceFeeAmount, reverseFeeAmount);
    }

    /**
     * <b>Case 1, Standard (EMoneyRefundMode.PROMO_MONEY_FIRST):</b>
     * <p/>
     * For refunds we should return all promo code money first. This is important in case of penalties or
     * partial refunds. All the money that is left as a penalty or a part of the initial payment should be distributed
     * according to our partner agreement.
     * <p>
     * <b>Case 2, Manual Support (EMoneyRefundMode.USER_MONEY_FIRST):</b>
     * <p/>
     * We have a special flag that inverts Case 1 logic:
     * The user money is returned first, then promo money is used. Corrections logic stays the same.
     * <p>
     * Let's examine the following sample payment and refund:
     * <p/><pre>
     *       ───────────────────
     *      │   user  │  promo  │
     *      ╔═════════╦═════════╗ ────────               ╔═════════╦═════════╗             ╔═════════╦═════════╗
     *      ║         ║         ║ partner │              ║   (?)   ║   (?)   ║             ║   (?)   ║   (?)   ║
     * 8600 ║    0    ║  8600   ║ part    │          430 ║  COST_2 ║  PROMO_ ║        8170 ║  COST_1 ║  PROMO_ ║
     *      ║         ║         ║         │              ║         ║  COST_2 ║             ║         ║  COST_1 ║
     *      ╠═════════╬═════════╣ ────────│   ─────      ╠═════════╬═════════╣     ═══     ╠═════════╬═════════╣
     *      ║         ║         ║ fee     │              ║   (?)   ║   (?)   ║             ║   (?)   ║   (?)   ║
     * 1400 ║   1000  ║   400   ║ part    │           70 ║  REW_2  ║ PROMO_  ║        1330 ║  REW_1  ║ PROMO_  ║
     *      ║         ║         ║         │              ║         ║ REW_2   ║             ║         ║ REW_1   ║
     *      ╚═════════╩═════════╝ ────────               ╚═════════╩═════════╝             ╚═════════╩═════════╝
     *          1000     9000                              USR_PEN?  PR_PEN?                 USR_REF?  PR_REF?
     *
     *             PAYMENT                                      PENALTY                           REFUND
     *             (10000)                              (500, the final PAYMENT)                  (9500)
     * </pre>
     * We know the initial payment components distribution (we re-calculate it on refund, based on the total amount
     * and the total promo code money) and the desired refund/penalty money distribution from the partner perspective.
     * <p>Having these values we can calculate how much promo code money should be returned (9000) and
     * how much is left (0). In addition to the defined penalty money split the left promo code money lets us
     * define the full result penalty money split. We simply re-use the payment promo money application logic here:
     * <p/><pre>
     *       ───────────────────
     *      │   user  │  promo  │
     *      ╔═════════╦═════════╗ ────────               ╔═════════╦═════════╗             ╔═════════╦═════════╗
     *      ║         ║         ║ partner │              ║         ║         ║             ║   (?)   ║   (?)   ║
     * 8600 ║    0    ║  8600   ║ part    │          430 ║   430   ║    0    ║        8170 ║  COST_1 ║  PROMO_ ║
     *      ║         ║         ║         │              ║         ║         ║             ║         ║  COST_1 ║
     *      ╠═════════╬═════════╣ ────────│   ─────      ╠═════════╬═════════╣     ═══     ╠═════════╬═════════╣
     *      ║         ║         ║ fee     │              ║         ║         ║             ║   (?)   ║   (?)   ║
     * 1400 ║   1000  ║   400   ║ part    │           70 ║    70   ║    0    ║        1330 ║  REW_1  ║ PROMO_  ║
     *      ║         ║         ║         │              ║         ║         ║             ║         ║ REW_1   ║
     *      ╚═════════╩═════════╝ ────────               ╚═════════╩═════════╝             ╚═════════╩═════════╝
     *          1000     9000                                500        0                      500       9000
     * </pre>
     * Now, all that's left is to subtract the according payment and penalty components to get the result refund ones.
     * The final order refund and payment (penalty) money will look like:
     * <p/><pre>
     *       ───────────────────
     *      │   user  │  promo  │
     *      ╔═════════╦═════════╗ ────────               ╔═════════╦═════════╗             ╔═════════╦═════════╗
     *      ║         ║         ║ partner │              ║         ║         ║             ║         ║         ║
     * 8600 ║    0    ║  8600   ║ part    │          430 ║   430   ║    0    ║        8170 ║  -430   ║   8600  ║
     *      ║         ║         ║         │              ║         ║         ║             ║         ║         ║
     *      ╠═════════╬═════════╣ ────────│   ─────      ╠═════════╬═════════╣     ═══     ╠═════════╬═════════╣
     *      ║         ║         ║ fee     │              ║         ║         ║             ║         ║         ║
     * 1400 ║   1000  ║   400   ║ part    │           70 ║    70   ║    0    ║        1330 ║   930   ║   400   ║
     *      ║         ║         ║         │              ║         ║         ║             ║         ║         ║
     *      ╚═════════╩═════════╝ ────────               ╚═════════╩═════════╝             ╚═════════╩═════════╝
     *          1000     9000                                500        0                      500       9000
     * </pre>
     * Here is where the source user money re-distribution problem comes from.
     * According to the initial user money distribution his payment of 1000 was split as cost = 0 and reward = 1000.
     * But the full promo code money refund has made us distribute the user's remaining penalty of 500 as
     * cost = 430 and reward = 70. This isn't possible unless we change some 'reward' money into 'cost'.
     * The negative COST_1 component shows us that it should be converter into a payment event.
     * <p>The example above should produce the following financial events:
     * <ul>
     *     <li>payment event: cost = 430</li>
     *     <li>refund event: reward = 930, promo_cost = 8600, promo_reward = 400</li>
     * </ul>
     */
    public FullMoneySplit calculateRefund(FullMoneySplit payment, MoneySourceSplit penaltySourceView,
                                          MoneySplit refundPartnerView) {
        Preconditions.checkArgument(payment.getTotal().isEqualTo(penaltySourceView.getTotal().add(refundPartnerView.getTotal())),
                "The initial payment %s can't be split into penalty %s and refund %s",
                payment, penaltySourceView, refundPartnerView);
        payment.ensureNoNegativeValues();
        penaltySourceView.ensureNoNegativeValues();
        refundPartnerView.ensureNoNegativeValues();

        // reconstructing the penalty (= the result payment in a more broad sense)
        MoneySplit penaltyPartnerView = payment.getPartnerView().subtract(refundPartnerView);
        FullMoneySplit penalty = calculatePayment(penaltySourceView, penaltyPartnerView);

        FullMoneySplit refund = payment.subtract(penalty);

        log.info("The initial payment split {} gets refunded for {} which splits as {}; the penalty is {}",
                payment.describe(), refundPartnerView.describe(), refund.describe(), penalty.describe());
        Preconditions.checkArgument(refund.getPartnerView().equals(refundPartnerView),
                "Full refund split doesn't match the requested one: full %s, requested %s",
                refund, refundPartnerView);

        return refund;
    }

    // should be used in legacy tests only, no plus points support
    @VisibleForTesting
    FullMoneySplit calculatePaymentWithPromoMoney(MoneySplit partnerView, Money promoMoney) {
        Money userMoney = partnerView.getTotal().subtract(promoMoney);
        MoneySourceSplit sourceView = MoneySourceSplit.builder()
                .user(userMoney)
                .plus(Money.zero(userMoney.getCurrency()))
                .promo(promoMoney)
                .userPostPay(Money.zero(userMoney.getCurrency()))
                .partnerReverseFee(Money.zero(userMoney.getCurrency()))
                .build();
        return calculatePayment(sourceView, partnerView);
    }

    // should be used in legacy tests only, no plus points support
    @VisibleForTesting
    FullMoneySplit calculateRefundWithPromoMoney(FullMoneySplit payment, MoneySplit refundPartnerView) {
        return calculateRefundWithPromoMoney(payment, refundPartnerView, DEFAULT_MONEY_REFUND_MODE);
    }

    // should be used in legacy tests only, no plus points support
    @VisibleForTesting
    FullMoneySplit calculateRefundWithPromoMoney(FullMoneySplit payment, MoneySplit refundPartnerView,
                                                 EMoneyRefundMode moneyRefundMode) {
        Money refundTotal = refundPartnerView.getTotal();
        Money penaltyTotal = payment.getTotal().subtract(refundTotal);
        Money penaltyPromoTotal = calculatePenaltyPromoMoney(payment.getTotal(), payment.getPromoMoney().getTotal(),
                refundTotal, moneyRefundMode);
        Money penaltyUserTotal = penaltyTotal.subtract(penaltyPromoTotal);
        MoneySourceSplit penaltySourceView = MoneySourceSplit.builder()
                .user(penaltyUserTotal)
                .plus(Money.zero(penaltyUserTotal.getCurrency()))
                .promo(penaltyPromoTotal)
                .userPostPay(Money.zero(penaltyUserTotal.getCurrency()))
                .partnerReverseFee(Money.zero(penaltyUserTotal.getCurrency()))
                .build();
        return calculateRefund(payment, penaltySourceView, refundPartnerView);
    }
}
