package ru.yandex.travel.orders.services.finances.providers;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.experimental.UtilityClass;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.services.finances.providers.ServiceBalance.PaymentBalance;

import static ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode.MRM_PROMO_MONEY_FIRST;
import static ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode.MRM_PROPORTIONAL;
import static ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode.MRM_UNDEFINED;
import static ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode.UNRECOGNIZED;

/**
 * The class provides helpers for implicit money distribution calculation.
 * The approach itself is very bug-prone and problematic.
 * <p/>
 * Ideally, we should get all needed source money splits from the application code performing money-related operations,
 * such as refunds and additional payments.
 * Then only partner splits will be calculated inside providers via simple and clear rules.
 * <p/>
 * The desired approach isn't implemented yet because current refund and fiscal items implementation doesn't track
 * enough information about actual order money amount and distribution.
 * It needs to be fixed in the near future.
 * <p/>
 * Items to remove:
 * <ul>
 *     <li>this class</li>
 *     <li>any usages of EMoneyRefundMode from code (but keep the structs in proto files for old events)</li>
 * </ul>
 */
@UtilityClass
public class LegacySplitHelper {
    public static MoneySourceSplit calculateLegacyPenaltySplit(ServiceBalance serviceBalance, Money penalty,
                                                               EMoneyRefundMode moneyRefundMode) {
        Preconditions.checkArgument(penalty.isPositiveOrZero(), "Negative penalty: %s", penalty);
        Money penaltyPromo;
        Money penaltyPlus;
        if (serviceBalance.hasEvents()) {
            PaymentBalance balance = serviceBalance.getOverallBalance();
            Money balanceTotal = balance.getTotal();
            Money totalRefund = balanceTotal.subtract(penalty);
            // not supporting negative refunds ('balance increasing') when some events have already been generated
            Preconditions.checkArgument(totalRefund.isPositiveOrZero(),
                    "Unexpected refund amount: %s, current balance - %s, penalty - %s",
                    totalRefund, balanceTotal, penalty);
            penaltyPromo = calculatePenaltyPromoMoney(
                    balanceTotal, balance.getTotalPromo(), totalRefund, moneyRefundMode);
            penaltyPlus = calculatePenaltyPlusMoney(
                    balance.getFullSplit(), totalRefund, penaltyPromo);
        } else {
            // old hack for deferred payments: before an order is FULLY_PAID there are no events to refund,
            // a simply assumption was made that there shouldn't be any promo money in the penalty of deferred payments
            // (this logic won't work well in case when the penalty should contain some promo or plus money)
            penaltyPromo = Money.zero(penalty.getCurrency());
            penaltyPlus = Money.zero(penalty.getCurrency());
        }
        return MoneySourceSplit.builder()
                .user(penalty.subtract(penaltyPlus).subtract(penaltyPromo))
                .plus(penaltyPlus)
                .promo(penaltyPromo)
                .userPostPay(Money.zero(penalty.getCurrency()))
                .partnerReverseFee(Money.zero(penalty.getCurrency()))
                .build();
    }

    private static EMoneyRefundMode ensureBackwardCompatibility(EMoneyRefundMode moneyRefundMode) {
        if (moneyRefundMode == null || moneyRefundMode == MRM_UNDEFINED || moneyRefundMode == UNRECOGNIZED) {
            // most callers don't specify the mode explicitly, we use the default one unless an explicit one is passed
            return MRM_PROMO_MONEY_FIRST;
        }
        return moneyRefundMode;
    }

    @VisibleForTesting
    static Money calculatePenaltyPromoMoney(Money paymentTotal, Money paymentPromoTotal, Money refundTotal,
                                            EMoneyRefundMode moneyRefundMode) {
        Preconditions.checkArgument(paymentTotal.isPositiveOrZero(), "Negative payment: %s", paymentTotal);
        Preconditions.checkArgument(paymentPromoTotal.isPositiveOrZero(), "Negative promo: %s", paymentPromoTotal);
        Preconditions.checkArgument(refundTotal.isPositiveOrZero(), "Negative refund: %s", refundTotal);
        moneyRefundMode = ensureBackwardCompatibility(moneyRefundMode);
        Money penaltyTotal = paymentTotal.subtract(refundTotal);
        Preconditions.checkArgument(penaltyTotal.isPositiveOrZero(), "Negative penalty: %s", penaltyTotal);

        Money penaltyPromoTotal;
        switch (moneyRefundMode) {
            case MRM_PROMO_MONEY_FIRST: {
                // return as much promo money as possible, the rest will be penalty
                Money refundPromoTotal = ComparatorUtils.min(paymentPromoTotal, refundTotal);
                penaltyPromoTotal = paymentPromoTotal.subtract(refundPromoTotal);
                break;
            }
            case MRM_USER_MONEY_FIRST: {
                // return as much user money as possible, the rest is a user money part of the penalty
                // after subtracting the user part from the total penalty we get the required promo penalty amount
                // by 'user' money we mean all non-promo money here (=user+plus+...)
                Money paymentUserTotal = paymentTotal.subtract(paymentPromoTotal);
                Money refundUserTotal = ComparatorUtils.min(paymentUserTotal, refundTotal);
                Money penaltyUserTotal = paymentUserTotal.subtract(refundUserTotal);
                penaltyPromoTotal = penaltyTotal.subtract(penaltyUserTotal);
                break;
            }
            case MRM_PROPORTIONAL: {
                penaltyPromoTotal = penaltyTotal
                        .multiply(paymentPromoTotal.getNumber())
                        .divide(paymentTotal.getNumber());
                break;
            }
            default:
                throw new IllegalArgumentException("Unsupported money refund mode: " + moneyRefundMode);
        }
        return penaltyPromoTotal;
    }

    @VisibleForTesting
    static Money calculatePenaltyPlusMoney(FullMoneySplit payment, Money refundTotal, Money penaltyPromo) {
        payment.ensureNoNegativeValues();
        Preconditions.checkArgument(refundTotal.isPositiveOrZero(), "Unexpected refund amount %s", refundTotal);
        Preconditions.checkArgument(penaltyPromo.isPositiveOrZero(),
                "Unexpected penalty promo amount %s", penaltyPromo);

        Money penaltyTotal = payment.getTotal().subtract(refundTotal);
        Preconditions.checkArgument(penaltyTotal.isPositiveOrZero(), "Unexpected penalty: %s", penaltyTotal);

        Money refundPromo = payment.getPromoMoney().getTotal().subtract(penaltyPromo);
        Preconditions.checkArgument(refundPromo.isPositiveOrZero(), "Unexpected refund promo: %s", refundPromo);
        Money refundPlus = ComparatorUtils.min(refundTotal.subtract(refundPromo), payment.getPlusMoney().getTotal());
        Preconditions.checkArgument(refundPlus.isPositiveOrZero(), "Unexpected refund plus: %s", refundPlus);

        return payment.getPlusMoney().getTotal().subtract(refundPlus);
    }
}
