package ru.yandex.travel.orders.services.finances.providers;

import java.util.function.Function;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import lombok.Builder;
import lombok.NonNull;
import lombok.Value;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.MoneyMarkup;

@Value
@Builder
class MoneySourceSplit {
    @NonNull Money user;
    @NonNull Money plus;
    @NonNull Money promo;
    @NonNull Money userPostPay;
    @NonNull Money partnerReverseFee;

    public String describe() {
        return String.format("%s [user = %s, plus = %s, promo = %s, userPostPay = %s, partnerReverseFee = %s]",
                getTotal(), user, plus, promo, userPostPay, partnerReverseFee);
    }

    public Money getTotal() {
        return user.add(plus).add(promo).add(userPostPay).add(partnerReverseFee);
    }

    public static MoneySourceSplit fromMarkupAndPromo(MoneyMarkup markup, Money promo) {
        return MoneySourceSplit.builder()
                .user(markup.getCard())
                .plus(markup.getYandexAccount())
                .promo(promo)
                .userPostPay(Money.zero(promo.getCurrency()))
                .partnerReverseFee(Money.zero(promo.getCurrency()))
                .build();
    }

    public static MoneySourceSplit fromPostPayOnly(Money postPayMoney, Money partnerReverseFee) {
        return MoneySourceSplit.builder()
                .user(Money.zero(postPayMoney.getCurrency()))
                .plus(Money.zero(postPayMoney.getCurrency()))
                .promo(Money.zero(postPayMoney.getCurrency()))
                .userPostPay(postPayMoney)
                .partnerReverseFee(partnerReverseFee)
                .build();
    }

    public static MoneySourceSplit zero(CurrencyUnit currency) {
        return MoneySourceSplit.builder()
                .user(Money.zero(currency))
                .plus(Money.zero(currency))
                .promo(Money.zero(currency))
                .userPostPay(Money.zero(currency))
                .partnerReverseFee(Money.zero(currency))
                .build();
    }

    public MoneySourceSplit subtract(MoneySourceSplit another) {
        return MoneySourceSplit.builder()
                .user(user.subtract(another.user))
                .plus(plus.subtract(another.plus))
                .promo(promo.subtract(another.promo))
                .userPostPay(userPostPay.subtract(another.userPostPay))
                .partnerReverseFee(partnerReverseFee.subtract(another.partnerReverseFee))
                .build();
    }

    public MoneySourceSplit map(Function<Money, Money> mapper) {
        return MoneySourceSplit.builder()
                .user(mapper.apply(user))
                .plus(mapper.apply(plus))
                .promo(mapper.apply(promo))
                .userPostPay(mapper.apply(userPostPay))
                .partnerReverseFee(mapper.apply(partnerReverseFee))
                .build();
    }

    public void ensureNoNegativeValues() {
        Preconditions.checkState(user.isPositiveOrZero(), "Negative user money: %s", user);
        Preconditions.checkState(plus.isPositiveOrZero(), "Negative plus money: %s", plus);
        Preconditions.checkState(promo.isPositiveOrZero(), "Negative promo money: %s", promo);
        Preconditions.checkState(userPostPay.isPositiveOrZero(), "Negative userPostPay money: %s", userPostPay);
        Preconditions.checkState(partnerReverseFee.isPositiveOrZero(), "Negative partnerReverseFee money: %s", partnerReverseFee);
    }
}
