package ru.yandex.travel.orders.services.finances.providers;

import javax.money.CurrencyUnit;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.NonNull;
import lombok.Value;
import org.javamoney.moneta.Money;
import org.jetbrains.annotations.NotNull;

import ru.yandex.travel.commons.lang.ComparatorUtils;

/**
 * A.k.a. partner or target/destination split.
 *
 * @see MoneySourceSplit
 * @see FullMoneySplit
 */
@Value
@Builder
@AllArgsConstructor
class MoneySplit {
    @NonNull
    Money partner;
    @NonNull
    Money fee;
    @NonNull
    Money reverseFee;

    MoneySplit(@NotNull Money partner, @NotNull Money fee) {
        this.partner = partner;
        this.fee = fee;
        this.reverseFee = Money.zero(partner.getCurrency());
    }

    public String describe() {
        return String.format("%s [partner = %s, fee = %s, reverseFee = %s]", getTotal(), partner, fee, reverseFee);
    }

    public Money getTotal() {
        return partner.add(fee).add(reverseFee);
    }

    public static MoneySplit zero(CurrencyUnit currency) {
        return new MoneySplit(
                Money.zero(currency),
                Money.zero(currency),
                Money.zero(currency)
        );
    }

    public MoneySplit add(MoneySplit other) {
        return new MoneySplit(
                this.partner.add(other.partner),
                this.fee.add(other.fee),
                this.reverseFee.add(other.reverseFee)
        );
    }

    public MoneySplit subtract(MoneySplit other) {
        return new MoneySplit(
                this.partner.subtract(other.partner),
                this.fee.subtract(other.fee),
                this.reverseFee.subtract(other.reverseFee)
        );
    }

    public MoneySplit extractNegativeValues() {
        return new MoneySplit(
                partner.isNegative() ? partner : Money.zero(partner.getCurrency()),
                fee.isNegative() ? fee : Money.zero(fee.getCurrency()),
                reverseFee.isNegative() ? reverseFee : Money.zero(reverseFee.getCurrency())
        );
    }

    public MoneySplit negate() {
        return new MoneySplit(partner.negate(), fee.negate(), reverseFee.negate());
    }

    public void ensureNoNegativeValues() {
        if (!extractNegativeValues().getTotal().isZero()) {
            throw new IllegalStateException("No negative values is expected: " + describe());
        }
    }

    public CurrencyUnit getCurrency() {
        return partner.getCurrency();
    }

    public static MoneySplit mergeByMin(MoneySplit a, MoneySplit b) {
        a.ensureNoNegativeValues();
        b.ensureNoNegativeValues();
        Money partner = ComparatorUtils.min(a.partner, b.partner);
        Money fee = ComparatorUtils.min(a.fee, b.fee);
        Money reverseFee = ComparatorUtils.min(a.reverseFee, b.reverseFee);
        return new MoneySplit(partner, fee, reverseFee);
    }
}
