package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.math.RoundingMode;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.finances.FinancialEvent;

@Slf4j
public class ProviderHelper {
    static Money ensureMoneyScale(Money money) {
        return Money.of(ensureMoneyScale(money.getNumberStripped()), money.getCurrency());
    }

    static BigDecimal ensureMoneyScale(BigDecimal money) {
        if (money.scale() > 2) {
            // we don't work with non integer kopeck values and simply round them
            log.warn("Suspicious money scale: sum={}, scale={}; will be half-up-rounded to 2 decimal places", money, money.scale());
        }
        return roundMoney(money);
    }

    public static BigDecimal roundMoney(BigDecimal money) {
        return money.setScale(2, RoundingMode.HALF_UP);
    }

    static MoneySplit splitMoney(MoneySourceSplit sourceSplit, BigDecimal splitRate) {
        boolean isReverse = !sourceSplit.getPartnerReverseFee().isZero();
        BigDecimal total = ensureMoneyScale(sourceSplit.getTotal().getNumberStripped());
        if (isReverse)
            total = total.subtract(sourceSplit.getPartnerReverseFee().getNumberStripped());
        BigDecimal feeAmount = isReverse ? BigDecimal.ZERO : roundMoney(total.multiply(splitRate));
        BigDecimal reverseFeeAmount = isReverse ? roundMoney(total.multiply(splitRate)) : BigDecimal.ZERO;
        BigDecimal partnerAmount = total.subtract(feeAmount);
        var currency = sourceSplit.getTotal().getCurrency();
        return new MoneySplit(
                Money.of(partnerAmount, currency),
                Money.of(feeAmount, currency),
                Money.of(reverseFeeAmount, currency)
        );
    }

    static MoneySplit splitMoney(Money totalSum, BigDecimal feeRate) {
        return splitMoney(totalSum.getNumberStripped(), totalSum.getCurrency(), feeRate, BigDecimal.ZERO);
    }

    static MoneySplit splitMoney(Money totalSum, BigDecimal feeRate, BigDecimal reverseFeeRate) {
        return splitMoney(totalSum.getNumberStripped(), totalSum.getCurrency(), feeRate, reverseFeeRate);
    }

    static MoneySplit splitMoney(BigDecimal totalSum, CurrencyUnit currency, BigDecimal feeRate, BigDecimal reverseFeeRate) {
        BigDecimal total = ensureMoneyScale(totalSum);
        BigDecimal feeAmount = roundMoney(total.multiply(feeRate));
        BigDecimal reverseFeeAmount = roundMoney(total.multiply(reverseFeeRate));
        BigDecimal partnerAmount = total.subtract(feeAmount);
        return new MoneySplit(
                Money.of(partnerAmount, currency),
                Money.of(feeAmount, currency),
                Money.of(reverseFeeAmount, currency)
        );
    }

    static void setPaymentMoney(FinancialEvent event, FullMoneySplit moneySplit, Boolean enablePromoFee) {
        Preconditions.checkArgument(event.isPayment(),
                "Only PAYMENT events are expected but got %s", event.getType());
        event.ensureNoMoneyValues();
        event.setPartnerAmount(moneySplit.getUserMoney().getPartner());
        event.setFeeAmount(moneySplit.getUserMoney().getFee());

        MoneySplit plusMoney = moneySplit.getPlusMoney();
        if (!plusMoney.getPartner().isZero()) {
            event.setPlusPartnerAmount(plusMoney.getPartner());
        }
        if (!plusMoney.getFee().isZero()) {
            event.setPlusFeeAmount(plusMoney.getFee());
        }

        MoneySplit promoMoney = moneySplit.getPromoMoney();
        if (!promoMoney.getPartner().isZero()) {
            event.setPromoCodePartnerAmount(promoMoney.getPartner());
        }
        if (!promoMoney.getFee().isZero()) {
            if (enablePromoFee) {
                event.setPromoCodeFeeAmount(promoMoney.getFee());
            } else {
                throw new RuntimeException("Promo code money can not be used with the reward payment type");
            }
        }

        MoneySplit postPayMoney = moneySplit.getUserPostPayMoney();
        if (!postPayMoney.getPartner().isZero()) {
            event.setPostPayUserAmount(postPayMoney.getPartner());
        }
        if (!postPayMoney.getReverseFee().isZero()) {
            event.setPostPayPartnerPayback(postPayMoney.getReverseFee());
        }
    }

    static void setRefundMoney(FinancialEvent event, FullMoneySplit moneySplit, Boolean enablePromoFee) {
        Preconditions.checkArgument(event.isRefund(),
                "Only REFUND events are expected but got %s", event.getType());
        event.ensureNoMoneyValues();
        event.setPartnerRefundAmount(moneySplit.getUserMoney().getPartner());
        event.setFeeRefundAmount(moneySplit.getUserMoney().getFee());

        MoneySplit plusMoney = moneySplit.getPlusMoney();
        if (!plusMoney.getPartner().isZero()) {
            event.setPlusPartnerRefundAmount(plusMoney.getPartner());
        }
        if (!plusMoney.getFee().isZero()) {
            event.setPlusFeeRefundAmount(plusMoney.getFee());
        }

        MoneySplit promoMoney = moneySplit.getPromoMoney();
        if (!promoMoney.getPartner().isZero()) {
            event.setPromoCodePartnerRefundAmount(promoMoney.getPartner());
        }
        if (!promoMoney.getFee().isZero()) {
            if (enablePromoFee) {
                event.setPromoCodeFeeRefundAmount(promoMoney.getFee());
            } else {
                throw new RuntimeException("Promo code money can not be used with the refund payment type");
            }
        }

        MoneySplit postPayMoney = moneySplit.getUserPostPayMoney();
        if (!postPayMoney.getPartner().isZero()) {
            event.setPostPayUserRefund(postPayMoney.getPartner());
        }
        if (!postPayMoney.getReverseFee().isZero()) {
            event.setPostPayPartnerRefund(postPayMoney.getReverseFee());
        }
    }
}
