package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import javax.money.CurrencyUnit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.services.finances.OverallServiceBalance;

import static ru.yandex.travel.orders.services.finances.FinancialEventService.DEFAULT_MONEY_REFUND_MODE;
import static ru.yandex.travel.orders.services.finances.providers.LegacySplitHelper.calculateLegacyPenaltySplit;

@Slf4j
public class ServiceBalance {

    private final List<FinancialEvent> allEvents;
    private final CurrencyUnit defaultCurrency;

    public ServiceBalance(List<FinancialEvent> allEvents, CurrencyUnit defaultCurrency) {
        this.allEvents = new ArrayList<>(allEvents);
        this.defaultCurrency = defaultCurrency;
    }

    Map<FinancialEvent, PaymentBalance> getPaymentBalances() {
        Map<FinancialEvent, PaymentBalance> result = new HashMap<>();
        for (var event : allEvents) {
            if (event.isYandexAccountTopup()) {
                // doesn't affect the balance
                continue;
            }
            if (event.isPayment()) {
                if (!result.containsKey(event)) {
                    result.put(event, new PaymentBalance(event, defaultCurrency));
                }
            } else if (event.isRefund()) {
                if (!result.containsKey(event.getOriginalEvent())) {
                    result.put(event.getOriginalEvent(), new PaymentBalance(event.getOriginalEvent(), defaultCurrency));
                }
                result.get(event.getOriginalEvent()).subtractOnRefund(event);
            } else {
                throw new IllegalArgumentException("Unsupported event: " + event.getDescription());
            }
        }
        return result;
    }

    public PaymentBalance getOverallBalance() {
        return getPaymentBalances().values().stream().reduce(PaymentBalance::add).orElse(new PaymentBalance(defaultCurrency));
    }

    public List<FinancialEvent> setBalanceTo(MoneySourceSplit targetSourcesSplit,
                                             BigDecimal splitRate,
                                             Supplier<FinancialEvent> paymentEventSupplier,
                                             Supplier<FinancialEvent> refundEventSupplier,
                                             FullMoneySplitCalculator calculator,
                                             Boolean enablePromoFee) {
        if (getOverallBalance().getTotal().isLessThan(targetSourcesSplit.getTotal())) {
            return increaseBalanceTo(targetSourcesSplit, splitRate, paymentEventSupplier, calculator, enablePromoFee);
        } else {
            return decreaseBalanceTo(targetSourcesSplit, splitRate, refundEventSupplier, calculator, enablePromoFee);
        }
    }

    public List<FinancialEvent> decreaseBalanceTo(MoneySourceSplit targetSourcesSplit,
                                                  BigDecimal splitRate,
                                                  Supplier<FinancialEvent> refundEventSupplier,
                                                  FullMoneySplitCalculator calculator,
                                                  Boolean enablePromoFee) {
        PaymentBalance currentBalance = getOverallBalance();
        MoneySourceSplit deltaSourcesSplit = currentBalance.getTotalSourceSplit().subtract(targetSourcesSplit);
        log.info("Requested balance decrease: from {} to {}, delta {}",
                currentBalance.getTotal(), targetSourcesSplit.getTotal(), deltaSourcesSplit.describe());
        deltaSourcesSplit.ensureNoNegativeValues();

        if (currentBalance.getTotal().isGreaterThan(targetSourcesSplit.getTotal())) {
            log.info("Balance decreasing, calculating refund events");
            Preconditions.checkState(refundEventSupplier != null, "No refund event supplier");
            return calculateRefunds(targetSourcesSplit, splitRate, refundEventSupplier, calculator, enablePromoFee);
        } else {
            return List.of();
        }
    }

    public List<FinancialEvent> increaseBalanceTo(MoneySourceSplit targetSourcesSplit,
                                                  BigDecimal splitRate,
                                                  Supplier<FinancialEvent> paymentEventSupplier,
                                                  FullMoneySplitCalculator calculator,
                                                  Boolean enablePromoFee
    ) {
        PaymentBalance currentBalance = getOverallBalance();
        MoneySourceSplit deltaSourcesSplit = targetSourcesSplit.subtract(currentBalance.getTotalSourceSplit());
        log.info("Requested balance increase: from {} to {}, delta {}",
                currentBalance.getTotal(), targetSourcesSplit.getTotal(), deltaSourcesSplit.describe());
        deltaSourcesSplit.ensureNoNegativeValues();

        if (deltaSourcesSplit.getTotal().isPositive()) {
            Preconditions.checkState(paymentEventSupplier != null, "No payment event supplier");
            MoneySplit deltaPartnerSplit = ProviderHelper.splitMoney(deltaSourcesSplit, splitRate);
            FullMoneySplit split = calculator.calculatePayment(deltaSourcesSplit, deltaPartnerSplit);
            FinancialEvent event = paymentEventSupplier.get();
            ProviderHelper.setPaymentMoney(event, split, enablePromoFee);
            allEvents.add(event);
            return List.of(event);
        } else {
            log.info("No changes in balance, no events generated");
            return Collections.emptyList();
        }
    }

    public List<FinancialEvent> recalculateBalance(BigDecimal splitRate,
                                                   Supplier<FinancialEvent> refundEventSupplier,
                                                   FullMoneySplitCalculator calculator,
                                                   Boolean enablePromoFee
    ) {
        PaymentBalance currentBalance = getOverallBalance();
        var targetSourcesSplit = currentBalance.getTotalSourceSplit();
        log.info("Requested balance recalculate: total {}, split {}", currentBalance.getTotal(), targetSourcesSplit.getTotal());
        var targetPartnerSplit = ProviderHelper.splitMoney(targetSourcesSplit.getTotal(), splitRate);
        var targetSplit = calculator.calculatePayment(targetSourcesSplit, targetPartnerSplit);
        var diffSplit = targetSplit.subtract(currentBalance.getFullSplit());
        FullMoneySplit refundSplit = diffSplit.extractNegativeValues();
        FullMoneySplit paymentSplit = diffSplit.subtract(refundSplit);

        if (refundSplit.getTotal().isZero() && paymentSplit.getTotal().isZero()) {
            log.info("No changes in balance, no events generated");
            return Collections.emptyList();
        }

        Preconditions.checkState(refundEventSupplier != null, "No refund event supplier");
        return applyRefundSplits(paymentSplit.negate(), refundSplit.negate(), refundEventSupplier, enablePromoFee);
    }

    // backward compatibility, should be used only in legacy tests
    @VisibleForTesting
    List<FinancialEvent> calculateRefunds(Money penalty, BigDecimal refundRate,
                                          Supplier<FinancialEvent> eventSupplier,
                                          FullMoneySplitCalculator calculator) {
        MoneySourceSplit penaltySourceSplit = calculateLegacyPenaltySplit(this, penalty, DEFAULT_MONEY_REFUND_MODE);
        return calculateRefunds(penaltySourceSplit, refundRate, eventSupplier, calculator, false);
    }

    private List<FinancialEvent> calculateRefunds(MoneySourceSplit penaltySourceSplit,
                                                  BigDecimal refundRate,
                                                  Supplier<FinancialEvent> eventSupplier,
                                                  FullMoneySplitCalculator calculator,
                                                  Boolean enablePromoFee) {
        Money penalty = penaltySourceSplit.getTotal();
        Money initialBalance = getOverallBalance().getTotalRefundable();
        Money amountToRefund = initialBalance.subtract(penalty);
        Preconditions.checkState(amountToRefund.isPositiveOrZero(),
                "Total refundable balance %s is less then required penalty %s", initialBalance, penalty);

        MoneySplit paidSplit = getOverallBalance().getRefundableSplit().getPartnerView();
        MoneySplit penaltySplit = ProviderHelper.splitMoney(penaltySourceSplit, refundRate);
        MoneySplit toRefundPartnerSplit = paidSplit.subtract(penaltySplit);
        FullMoneySplit toRefundWithCorrection =
                calculator.calculateRefund(getOverallBalance().getRefundableSplit(), penaltySourceSplit,
                        toRefundPartnerSplit);
        FullMoneySplit correctionSplit = toRefundWithCorrection.extractNegativeValues();
        FullMoneySplit toRefundRemaining = toRefundWithCorrection.subtract(correctionSplit);

        var result = applyRefundSplits(correctionSplit, toRefundRemaining, eventSupplier, enablePromoFee);

        if (result.isEmpty()) { // backwards compatibility
            result.add(eventSupplier.get().toBuilder()
                    .partnerRefundAmount(Money.zero(defaultCurrency))
                    .feeRefundAmount(Money.zero(defaultCurrency))
                    .build());
        }
        return result;
    }

    private List<FinancialEvent> applyRefundSplits(FullMoneySplit correctionSplit,
                                                   FullMoneySplit refundSplit,
                                                   Supplier<FinancialEvent> refundEventSupplier,
                                                   Boolean enablePromoFee) {
        List<FinancialEvent> result = new ArrayList<>();
        while (refundSplit.getTotal().isPositive()) {
            var bestRefundableEventAndBalance = getBestRefundableEvent();
            FinancialEvent eventToRefund = bestRefundableEventAndBalance.get1();
            PaymentBalance balance = bestRefundableEventAndBalance.get2();
            FullMoneySplit split = FullMoneySplit.mergeByMin(balance.getRefundableSplit(), refundSplit);
            FinancialEvent event = refundEventSupplier.get();
            ProviderHelper.setRefundMoney(event, split, enablePromoFee);
            event.setOriginalEvent(eventToRefund);
            allEvents.add(event);
            result.add(event);
            refundSplit = refundSplit.subtract(split);
        }
        if (!correctionSplit.getTotal().isZero()) {
            correctionSplit = correctionSplit.negate();
            log.info("The refund operation requires an additional payment event " +
                            "to correct the payout money distribution: refund event {}",
                    correctionSplit.describe());
            FinancialEvent correctionEvent = refundEventSupplier.get().toBuilder()
                    .type(FinancialEventType.PAYMENT)
                    .comment("Correction event: refund with promo code money")
                    .build();
            ProviderHelper.setPaymentMoney(correctionEvent, correctionSplit, enablePromoFee);
            allEvents.add(correctionEvent);
            result.add(correctionEvent);
        }
        return result;
    }

    Tuple2<FinancialEvent, PaymentBalance> getBestRefundableEvent() {
        return getPaymentBalances().entrySet().stream()
                .filter(e -> e.getValue().getTotalRefundable().isPositive())
                .max(Map.Entry.comparingByValue())
                .map(e -> Tuple2.tuple(e.getKey(), e.getValue()))
                .orElse(null);
    }

    public boolean hasEvents() {
        return !allEvents.isEmpty();
    }

    @AllArgsConstructor(access = AccessLevel.PRIVATE)
    @Getter
    public static class PaymentBalance implements Comparable<PaymentBalance> {
        private CurrencyUnit currency;
        private FullMoneySplit fullSplit;
        // tech fee is a special thing, isn't properly supported at this moment
        private Money partnerFeeAmount;

        public PaymentBalance(CurrencyUnit defaultCurrency) {
            fullSplit = FullMoneySplit.zero(defaultCurrency);
            partnerFeeAmount = Money.zero(defaultCurrency);
        }

        PaymentBalance(FinancialEvent paymentEvent, CurrencyUnit defaultCurrency) {
            Preconditions.checkArgument(paymentEvent.isPayment(), "Payment event is not actually payment");
            currency = defaultCurrency;
            fullSplit = FullMoneySplit.builder()
                    .userMoney(MoneySplit.builder()
                            .partner(paymentEvent.getPartnerAmount())
                            .fee(paymentEvent.getFeeAmount())
                            .reverseFee(Money.zero(currency))
                            .build())
                    .plusMoney(MoneySplit.builder()
                            .partner(nullToZero(paymentEvent.getPlusPartnerAmount(), currency))
                            .fee(nullToZero(paymentEvent.getPlusFeeAmount(), currency))
                            .reverseFee(Money.zero(currency))
                            .build())
                    .promoMoney(MoneySplit.builder()
                            .partner(nullToZero(paymentEvent.getPromoCodePartnerAmount(), currency))
                            .fee(nullToZero(paymentEvent.getPromoCodeFeeAmount(), currency))
                            .reverseFee(Money.zero(currency))
                            .build())
                    .userPostPayMoney(MoneySplit.builder()
                            .partner(nullToZero(paymentEvent.getPostPayUserAmount(), currency))
                            .fee(Money.zero(currency))
                            .reverseFee(nullToZero(paymentEvent.getPostPayPartnerPayback(), currency))
                            .build())
                    .build();
            partnerFeeAmount = nullToZero(paymentEvent.getPartnerFeeAmount(), currency);
        }

        private static Money nullToZero(Money value, CurrencyUnit currency) {
            return value != null ? value : Money.zero(currency);
        }

        void subtractOnRefund(FinancialEvent refundEvent) {
            Preconditions.checkArgument(refundEvent.isRefund(), "Refund event is not actually refund");
            FullMoneySplit refundSplit = FullMoneySplit.builder()
                    .userMoney(MoneySplit.builder()
                            .partner(refundEvent.getPartnerRefundAmount())
                            .fee(refundEvent.getFeeRefundAmount())
                            .reverseFee(Money.zero(currency))
                            .build())
                    .plusMoney(MoneySplit.builder()
                            .partner(nullToZero(refundEvent.getPlusPartnerRefundAmount(), currency))
                            .fee(nullToZero(refundEvent.getPlusFeeRefundAmount(), currency))
                            .reverseFee(Money.zero(currency))
                            .build())
                    .promoMoney(MoneySplit.builder()
                            .partner(nullToZero(refundEvent.getPromoCodePartnerRefundAmount(), currency))
                            .fee(nullToZero(refundEvent.getPromoCodeFeeRefundAmount(), currency))
                            .reverseFee(Money.zero(currency))
                            .build())
                    .userPostPayMoney(MoneySplit.zero(currency))
                    .build();
            fullSplit = fullSplit.subtract(refundSplit);
            if (refundEvent.getPartnerFeeRefundAmount() != null) {
                partnerFeeAmount = partnerFeeAmount.subtract(refundEvent.getPartnerFeeRefundAmount());
            }
        }

        public PaymentBalance add(PaymentBalance other) {
            return new PaymentBalance(
                    this.currency,
                    this.fullSplit.add(other.fullSplit),
                    this.partnerFeeAmount.add(other.partnerFeeAmount));
        }

        public FullMoneySplit getRefundableSplit() {
            return fullSplit;
        }

        public Money getTotalPromo() {
            return fullSplit.getPromoMoney().getTotal();
        }

        public Money getTotalPlus() {
            return fullSplit.getPlusMoney().getTotal();
        }

        public Money getTotalUser() {
            return fullSplit.getUserMoney().getTotal();
        }

        public Money getTotalPartner() {
            return fullSplit.getPartnerView().getPartner();
        }

        public Money getTotalFee() {
            return fullSplit.getPartnerView().getFee();
        }

        public Money getTotalRefundable() {
            return fullSplit.getTotal();
        }

        public Money getTotal() {
            // should reconsider this method name/implementation in case the Train provider starts using ServiceBalance
            Preconditions.checkState(partnerFeeAmount.isZero(),
                    "Partner tech. fee is not supported: %s", partnerFeeAmount);
            return getTotalRefundable().add(partnerFeeAmount);
        }

        public MoneySourceSplit getTotalSourceSplit() {
            // should reconsider this method name/implementation in case the Train provider starts using ServiceBalance
            Preconditions.checkState(partnerFeeAmount.isZero(),
                    "Partner tech. fee is not supported: %s", partnerFeeAmount);
            return fullSplit.getSourceView();
        }

        @Override
        public int compareTo(PaymentBalance other) {
            if (this.getTotalPromo().isGreaterThan(other.getTotalPromo())) {
                return 1;
            }
            if (this.getTotalPromo().isLessThan(other.getTotalPromo())) {
                return -1;
            }
            return this.getTotalRefundable().compareTo(other.getTotalRefundable());
        }

        public OverallServiceBalance convertToOverall() {
            return OverallServiceBalance.builder()
                    .userPartner(fullSplit.getUserMoney().getPartner())
                    .userFee(fullSplit.getUserMoney().getFee())
                    .plusPartner(fullSplit.getPlusMoney().getPartner())
                    .plusFee(fullSplit.getPlusMoney().getFee())
                    .promoPartner(fullSplit.getPromoMoney().getPartner())
                    .promoFee(fullSplit.getPromoMoney().getFee())
                    .techFee(partnerFeeAmount)
                    .build();
        }
    }

}
