package ru.yandex.travel.orders.services.finances.providers;

import java.time.Instant;
import java.util.List;

import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.services.suburban.environment.SuburbanOrderItemEnvProvider;
import ru.yandex.travel.orders.services.suburban.providers.SuburbanProviderBase;
import ru.yandex.travel.utils.ClockService;

@RequiredArgsConstructor
public class SuburbanFinancialDataProvider implements FinancialDataProvider {
    /*
    Логика вычислений:
    https://st.yandex-team.ru/RASPFRONT-9208#602ee942da223c648b604a2c
     */
    private final ClockService clockService;
    private final SuburbanOrderItemEnvProvider suburbanEnvProvider;

    @Override
    public List<EServiceType> getServiceTypes() {
        return List.of(EServiceType.PT_SUBURBAN);
    }

    @Override
    public List<FinancialEvent> onConfirmation(OrderItem orderItem, Boolean enablePromoFee) {
        SuburbanOrderItem suburbanOrderItem = (SuburbanOrderItem) orderItem;
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(suburbanOrderItem).createSuburbanProvider();

        if (!provider.enableAddBillingTransactions())
            return List.of();

        FinancialEvent event = createPaymentEvent(suburbanOrderItem, FinancialEventType.PAYMENT);

        Money price = suburbanOrderItem.getPayload().getPrice();
        Money rewardFee = provider.getRewardFee(price);
        event.setPartnerAmount(price);
        event.setFeeAmount(ProviderHelper.ensureMoneyScale(rewardFee));  // round to kopecks

        return List.of(event);
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee) {
        SuburbanOrderItem suburbanOrderItem = (SuburbanOrderItem) orderItem;
        SuburbanProviderBase provider = suburbanEnvProvider.createEnv(suburbanOrderItem).createSuburbanProvider();

        if (!provider.enableAddBillingTransactions())
            return List.of();

        FinancialEvent event = createPaymentEvent(suburbanOrderItem, FinancialEventType.REFUND);

        Money price = suburbanOrderItem.getPayload().getPrice();
        Money rewardFeeRefund = provider.getRewardFeeRefund(price);
        event.setPartnerRefundAmount(price);
        event.setFeeRefundAmount(ProviderHelper.ensureMoneyScale(rewardFeeRefund));  // round to kopecks

        return List.of(event);
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee, Money penalty, Money refund, Instant refundedAt) {
        throw new UnsupportedOperationException("Refunds with specified penalty are not supported for suburban finevent providers");
    }

    private FinancialEvent createPaymentEvent(SuburbanOrderItem orderItem, FinancialEventType eventType) {
        FinancialEvent event = FinancialEvent.create(
                orderItem, eventType, FinancialEventPaymentScheme.SUBURBAN, Instant.now(clockService.getUtc()));

        event.setBillingClientId(orderItem.getBillingPartnerAgreement().getBillingClientId());

        // https://st.yandex-team.ru/RASPFRONT-9630#60d331822f24aa34b4fc2735
        event.setPayoutAt(orderItem.getConfirmedAt());
        event.setAccountingActAt(event.getPayoutAt());

        return event;
    }

    @Override
    public List<FinancialEvent> onExtraPayment(OrderItem orderItem, Payment extraPayment, Boolean enablePromoFee) {
        throw new UnsupportedOperationException(
                "Method onExtraPayment must not be called for SuburbanFinancialDataProvider");
    }

    @Override
    public List<FinancialEvent> onPartialRefund(OrderItem orderItem, OrderRefund partialRefund, Boolean enablePromoFee) {
        throw new UnsupportedOperationException(
                "Method onPartialRefund must not be called for SuburbanFinancialDataProvider");
    }

    @Override
    public List<FinancialEvent> onPaymentScheduleFullyPaid(OrderItem orderItem, PaymentSchedule schedule, Boolean enablePromoFee) {
        throw new UnsupportedOperationException("Payment schedules are not supported for suburban finevent providers");
    }

    @Override
    public VatType getYandexFeeVat(OrderItem orderItem) {
        return VatType.VAT_20;
    }
}
