package ru.yandex.travel.orders.services.finances.providers;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.VatType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.TrainBillingPartnerAgreement;
import ru.yandex.travel.orders.grpc.helpers.OrderProtoUtils;
import ru.yandex.travel.orders.services.finances.billing.BillingHelper;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;

@Service
@RequiredArgsConstructor
public class TrainFinancialDataProvider implements FinancialDataProvider {
    private final Clock clock;
    private final TrainWorkflowProperties trainWorkflowProperties;

    @Override
    public List<EServiceType> getServiceTypes() {
        return List.of(EServiceType.PT_TRAIN);
    }

    @Override
    public List<FinancialEvent> onConfirmation(OrderItem orderItem, Boolean enablePromoFee) {
        TrainOrderItem trainOrderItem = (TrainOrderItem) orderItem;
        TrainReservation payload = trainOrderItem.getPayload();
        List<FinancialEvent> events = new ArrayList<>();
        TrainBillingPartnerAgreement trainBillingAgreement =
                (TrainBillingPartnerAgreement) orderItem.getBillingPartnerAgreement();

        var orderCurrency = TrainModelHelpers.checkAndGetOrderCurrency(payload);
        Preconditions.checkState(payload.getPassengers().size() > 0,
                "There should be 1 or more passenger in successful train order");
        Preconditions.checkNotNull(orderCurrency, "OrderItem should have not null currency");
        MoneySplit totalTicketSplit = MoneySplit.zero(orderCurrency);
        MoneySplit totalInsuranceSplit = MoneySplit.zero(orderCurrency);
        Money totalPartnerFeeAmount = Money.zero(orderCurrency);

        for (TrainPassenger passenger : payload.getPassengers()) {
            var ticket = passenger.getTicket();
            var insurance = passenger.getInsurance();

            totalTicketSplit = totalTicketSplit.add(ticketMoneySplit(ticket));
            if (payload.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                totalInsuranceSplit = totalInsuranceSplit.add(
                        insuranceMoneySplit(insurance.getAmount(), trainBillingAgreement));
            }
            totalPartnerFeeAmount = totalPartnerFeeAmount.add(ticket.getPartnerFee());
        }

        FinancialEvent ticketEvent = createPaymentEvent(trainOrderItem, FinancialEventType.PAYMENT,
                trainBillingAgreement.getBillingClientId());
        FullMoneySplit fullTicketSplit = FullMoneySplit.fromUserMoneyOnly(totalTicketSplit);
        ProviderHelper.setPaymentMoney(ticketEvent, fullTicketSplit, enablePromoFee);
        if (!totalPartnerFeeAmount.isZero()) {
            ticketEvent.setPartnerFeeAmount(totalPartnerFeeAmount);
        }
        events.add(ticketEvent);

        if (payload.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
            FinancialEvent insuranceEvent = createPaymentEvent(trainOrderItem, FinancialEventType.INSURANCE_PAYMENT,
                    trainBillingAgreement.getInsuranceClientId());
            FullMoneySplit fullInsuranceSplit = FullMoneySplit.fromUserMoneyOnly(totalInsuranceSplit);
            ProviderHelper.setPaymentMoney(insuranceEvent, fullInsuranceSplit, enablePromoFee);
            events.add(insuranceEvent);
        }

        Preconditions.checkState(!events.isEmpty(), "No financial events for a confirmed order: %s",
                orderItem.getOrder().getId());
        return events;
    }

    @Override
    public List<FinancialEvent> onExtraPayment(OrderItem orderItem, Payment extraPayment, Boolean enablePromoFee) {
        // Extra payments for train order are not defined atm
        return List.of();
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee) {
        return List.of();
    }

    @Override
    public List<FinancialEvent> onRefund(OrderItem orderItem, EMoneyRefundMode moneyRefundMode, Boolean enablePromoFee, Money penalty, Money refund, Instant refundedAt) {
        throw new UnsupportedOperationException("Refunds with specified penalty are not supported for train finevent providers");
    }

    @Override
    public List<FinancialEvent> onPartialRefund(OrderItem orderItem, OrderRefund partialRefund, Boolean enablePromoFee) {
        TrainOrderItem trainOrderItem = (TrainOrderItem) orderItem;
        var orderCurrency = TrainModelHelpers.checkAndGetOrderCurrency(trainOrderItem.getPayload());
        Preconditions.checkNotNull(orderCurrency, "OrderItem should have not null currency");
        Map<UUID, TrainTicketRefund> ticketRefundsByOrderRefund =
                trainOrderItem.getTrainTicketRefunds().stream()
                        .collect(Collectors.toMap(refund -> refund.getOrderRefund().getId(), x -> x));
        Map<Integer, TrainPassenger> passengersByCustomerId =
                trainOrderItem.getPayload().getPassengers().stream()
                        .collect(Collectors.toMap(TrainPassenger::getCustomerId, tp -> tp));
        TrainBillingPartnerAgreement trainBillingAgreement =
                (TrainBillingPartnerAgreement) orderItem.getBillingPartnerAgreement();
        TrainTicketRefund ticketRefund = ticketRefundsByOrderRefund.get(partialRefund.getId());
        MoneySplit totalTicketSplit = MoneySplit.zero(orderCurrency);
        MoneySplit totalInsuranceSplit = MoneySplit.zero(orderCurrency);
        Money totalPartnerFeeAmount = Money.zero(orderCurrency);
        for (PassengerRefundInfo info : ticketRefund.getPayload().getRefundedItems()) {
            totalTicketSplit = totalTicketSplit.add(new MoneySplit(
                    info.getActualRefundTicketAmount(),
                    info.getCalculatedRefundFeeAmount(),
                    Money.zero(orderCurrency)
            ));
            if (info.getCalculatedRefundInsuranceAmount() != null) {
                totalInsuranceSplit = totalInsuranceSplit.add(
                        insuranceMoneySplit(info.getCalculatedRefundInsuranceAmount(), trainBillingAgreement));
            }
            TrainPassenger trainPassenger =
                    Optional.ofNullable(passengersByCustomerId.get(info.getCustomerId())).orElseThrow();
            totalPartnerFeeAmount = totalPartnerFeeAmount.add(trainPassenger.getTicket().getPartnerRefundFee());
        }

        List<FinancialEvent> events = new ArrayList<>();

        FullMoneySplit fullTicketSplit = FullMoneySplit.fromUserMoneyOnly(totalTicketSplit);
        FinancialEvent ticketRefundEvent = createRefundEvent(trainOrderItem, FinancialEventType.REFUND,
                trainBillingAgreement.getBillingClientId());
        ProviderHelper.setRefundMoney(ticketRefundEvent, fullTicketSplit, enablePromoFee);
        if (!totalPartnerFeeAmount.isZero()) {
            ticketRefundEvent.setPartnerFeeAmount(totalPartnerFeeAmount);
        }
        events.add(ticketRefundEvent);

        if (!totalInsuranceSplit.getPartner().isZero()) {
            FullMoneySplit fullInsuranceSplit = FullMoneySplit.fromUserMoneyOnly(totalInsuranceSplit);
            FinancialEvent insuranceEvent = createRefundEvent(trainOrderItem, FinancialEventType.INSURANCE_REFUND,
                    trainBillingAgreement.getInsuranceClientId());
            ProviderHelper.setRefundMoney(insuranceEvent, fullInsuranceSplit, enablePromoFee);
            events.add(insuranceEvent);
        }
        Preconditions.checkState(!events.isEmpty(), "No financial events for a refund order: %s",
                orderItem.getOrder().getId());
        return events;
    }

    @Override
    public List<FinancialEvent> onPaymentScheduleFullyPaid(OrderItem orderItem, PaymentSchedule schedule, Boolean enablePromoFee) {
        throw new UnsupportedOperationException("Payment schedules are not supported for train finevent providers");
    }

    @Override
    public VatType getYandexFeeVat(OrderItem orderItem) {
        return OrderProtoUtils.fromEVat(trainWorkflowProperties.getBilling().getFeeVat());
    }

    private FinancialEvent createPaymentEvent(TrainOrderItem orderItem, FinancialEventType type, Long partnerId) {
        Instant now = Instant.now(clock);
        FinancialEvent event = FinancialEvent.create(orderItem, type, FinancialEventPaymentScheme.TRAINS, now);

        TrainBillingPartnerAgreement agreement = orderItem.getBillingPartnerAgreement();
        event.setBillingClientId(partnerId);
        Instant payoutAt = getPayoutAtForOrderItem(orderItem);
        if (payoutAt == null) {
            payoutAt = now;
        }
        event.setPayoutAt(payoutAt);
        event.setAccountingActAt(payoutAt);

        return event;
    }

    private FinancialEvent createRefundEvent(TrainOrderItem orderItem, FinancialEventType type, Long partnerId) {
        Instant now = Instant.now(clock);
        FinancialEvent event = FinancialEvent.create(orderItem, type, FinancialEventPaymentScheme.TRAINS, now);

        TrainBillingPartnerAgreement agreement = orderItem.getBillingPartnerAgreement();
        event.setBillingClientId(partnerId);
        Instant payoutAt = orderItem.getRefundedAt();
        if (payoutAt == null || now.isAfter(payoutAt)) {
            payoutAt = now;
        }
        event.setPayoutAt(payoutAt);
        event.setAccountingActAt(payoutAt);

        return event;
    }

    private Instant getPayoutAtForOrderItem(TrainOrderItem orderItem) {
        return BillingHelper.getBillingDayStart(orderItem.getConfirmedAt());
    }

    private MoneySplit ticketMoneySplit(TrainTicket ticket) {
        Money partnerAmount = ticket.getTariffAmount().add(ticket.getServiceAmount());
        Money feeAmount = ticket.getFeeAmount();
        return new MoneySplit(partnerAmount, feeAmount, Money.zero(partnerAmount.getCurrency()));
    }

    private MoneySplit insuranceMoneySplit(Money insuranceAmount, TrainBillingPartnerAgreement trainBillingAgreement) {
        BigDecimal insuranceFeeCoefficient = trainBillingAgreement.getInsuranceFeeCoefficient();
        return ProviderHelper.splitMoney(insuranceAmount, insuranceFeeCoefficient);
    }
}
