package ru.yandex.travel.orders.services.finances.providers;

import java.time.Clock;
import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties;

@Service
@RequiredArgsConstructor
public class YandexPlusFinancialDataProvider {
    private final YandexPlusPromoProperties properties;
    private final Clock clock;

    public FinancialEvent onPlusTopup(YandexPlusTopup topup) {
        Instant now = Instant.now(clock);
        Instant topupDate = topup.getAuthorizedAt();
        FinancialEventType eventType = FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT;
        FinancialEvent event;
        Preconditions.checkState((topup.getExternalOrderId() == null || topup.getOrderItem() == null) &&
                        (topup.getExternalOrderId() != null || topup.getOrderItem() != null),
                "Topup should have either orderItem or orderPrettyIdOverride");
        if (topup.getOrderItem() != null) {
            FinancialEventPaymentScheme paymentScheme = getPaymentScheme(topup.getOrderItem());
            event = FinancialEvent.create(topup.getOrderItem(), eventType, paymentScheme, now);
        } else {
            FinancialEventPaymentScheme paymentScheme = getPaymentScheme(topup.getPaymentProfile());
            event = FinancialEvent.createWithoutOrder(topup.getExternalOrderId(), eventType, paymentScheme, now);
        }
        // using a separate billing account for topup operations as it's not actually used by Billing
        event.setBillingClientId(properties.getBillingClientId());
        event.setBillingContractId(null);
        event.setPayoutAt(topupDate);
        event.setAccountingActAt(topupDate);
        event.setPlusTopupAmount(topup.getAmountMoney());
        event.setTrustPaymentId(topup.getTrustPaymentId());
        return event;
    }

    private FinancialEventPaymentScheme getPaymentScheme(OrderItem orderItem) {
        EDisplayOrderType orderType = orderItem.getOrder().getDisplayType();
        //noinspection SwitchStatementWithTooFewBranches
        switch (orderType) {
            case DT_HOTEL:
                return FinancialEventPaymentScheme.HOTELS;
            default:
                throw new IllegalArgumentException("Unsupported topup initiating order type: " + orderType);
        }
    }

    private FinancialEventPaymentScheme getPaymentScheme(PaymentProfile paymentProfile) {
        if (paymentProfile == PaymentProfile.HOTEL) {
            return FinancialEventPaymentScheme.HOTELS;
        }
        throw new IllegalArgumentException("Unsupported topup initiating payment profile: " + paymentProfile);
    }
}
