package ru.yandex.travel.orders.services.finances.tasks;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.validation.annotation.Validated;

import ru.yandex.inside.yt.kosher.CloseableYt;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.integration.balance.BillingApiProperties;
import ru.yandex.travel.integration.balance.BillingCsvApiClient;
import ru.yandex.travel.orders.cache.HotelAgreementDictionary;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingPartnerAgreementSynchronizer;
import ru.yandex.travel.orders.services.finances.billing.BillingPartnerAgreementSynchronizerProperties;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionActCommitter;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionActCommitterProperties;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionGenerator;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtExporter;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtIdGenerator;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtTableClient;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtTableClientProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorHelper;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.tvm.TvmWrapper;
import ru.yandex.travel.utils.ClockService;

@Configuration
@RequiredArgsConstructor
public class FinancialEventTaskProcessorsConfiguration {
    @ConditionalOnProperty("billing-transactions.generator-task.enabled")
    @Configuration
    public static class BillingTransactionGenerateTaskConfiguration {
        @Bean
        public BillingTransactionGenerator billingTransactionGenerator(
                FinancialEventRepository financialEventRepository,
                BillingTransactionRepository billingTransactionRepository,
                ClockService clockService
        ) {
            return new BillingTransactionGenerator(financialEventRepository, billingTransactionRepository, clockService);
        }

        @ConfigurationProperties(value = "billing-transactions.generator-task", ignoreUnknownFields = false)
        @Bean
        public TaskProcessorProperties billingTransactionGeneratorTaskProcessorProperties() {
            return TaskProcessorProperties.builder()
                    .name("BillingTransactionGenerator")
                    .daemonPoolThreads(true)
                    .gracefulPoolShutdown(true)
                    .build();
        }

        @Bean
        public TaskProcessor<Long> billingTransactionGeneratorTaskProcessor(
                @Qualifier("billingTransactionGeneratorTaskProcessorProperties") TaskProcessorProperties properties,
                BillingTransactionGenerator billingTransactionGenerator,
                ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
        ) {
            TaskKeyProvider<Long> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                    billingTransactionGenerator::fetchEventIdsWaitingProcessing,
                    billingTransactionGenerator::countEventsWaitingProcessing
            );
            return new TaskProcessor<>(taskKeyProvider,
                    billingTransactionGenerator::processPendingEvent,
                    forcedRollbackTxManagerWrapper,
                    TaskProcessorHelper.createDefaultTxDefinition("billingTransactionGeneratorTaskTxDef"),
                    properties
            );
        }
    }

    @ConditionalOnProperty("billing-transactions.yt-id-generator-task.enabled")
    @Configuration
    public static class BillingTransactionYtIdGeneratorTaskConfiguration {
        @Bean
        public BillingTransactionYtIdGenerator billingTransactionYtIdGenerator(
                BillingTransactionRepository billingTransactionRepository,
                ClockService clockService
        ) {
            return new BillingTransactionYtIdGenerator(billingTransactionRepository, clockService);
        }

        @ConfigurationProperties(value = "billing-transactions.yt-id-generator-task", ignoreUnknownFields = false)
        @Bean
        public TaskProcessorProperties billingTransactionYtIdGeneratorTaskProcessorProperties() {
            return TaskProcessorProperties.builder()
                    .name("BillingTransactionYtIdGenerator")
                    .daemonPoolThreads(true)
                    .gracefulPoolShutdown(true)
                    .build();
        }

        @Bean
        public TaskProcessor<String> billingTransactionYtIdGeneratorTaskProcessor(
                @Qualifier("billingTransactionYtIdGeneratorTaskProcessorProperties") TaskProcessorProperties properties,
                BillingTransactionYtIdGenerator billingTransactionYtIdGenerator,
                ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
        ) {
            TaskKeyProvider<String> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                    (activeTasks, limit) -> billingTransactionYtIdGenerator.getYtIdBulkGenerationTaskIds(activeTasks),
                    (activeTasks) -> billingTransactionYtIdGenerator.countYdIdGenerationTasks()
            );
            return new TaskProcessor<>(taskKeyProvider,
                    billingTransactionYtIdGenerator::bulkGenerateNewYtIds,
                    forcedRollbackTxManagerWrapper,
                    TaskProcessorHelper.createDefaultTxDefinition("billingTransactionYtIdGeneratorTaskTxDef"),
                    properties
            );
        }
    }

    @ConditionalOnProperty("billing-transactions.export-task.enabled")
    @Configuration
    public static class BillingTransactionYtExportTaskConfiguration {
        @ConfigurationProperties(value = "billing-transactions.yt-exporter", ignoreUnknownFields = false)
        @Bean
        public BillingTransactionYtTableClientProperties billingTransactionYtTableClientProperties() {
            return new BillingTransactionYtTableClientProperties();
        }

        @Bean
        public CloseableYt billingTransactionExporterYtClient(BillingTransactionYtTableClientProperties properties) {
            return (CloseableYt) YtUtils.http(properties.getCluster(), properties.getToken());
        }

        @Bean
        public BillingTransactionYtTableClient billingTransactionYtTableClient(
                BillingTransactionYtTableClientProperties properties,
                @Qualifier("billingTransactionExporterYtClient") Yt yt
        ) {
            return new BillingTransactionYtTableClient(properties, yt);
        }

        @Bean
        public BillingTransactionYtExporter billingTransactionYtExporter(
                BillingTransactionYtTableClientProperties properties,
                BillingTransactionRepository billingTransactionRepository,
                BillingTransactionYtTableClient billingTransactionYtTableClient,
                ClockService clockService
        ) {
            return new BillingTransactionYtExporter(properties, billingTransactionRepository,
                    billingTransactionYtTableClient, clockService);
        }

        @ConfigurationProperties(value = "billing-transactions.export-task", ignoreUnknownFields = false)
        @Bean
        public TaskProcessorProperties billingTransactionExportTaskProcessorProperties() {
            return TaskProcessorProperties.builder()
                    .name("BillingTransactionExport")
                    .daemonPoolThreads(true)
                    .gracefulPoolShutdown(true)
                    .build();
        }

        @Bean
        public TaskProcessor<String> billingTransactionExportTaskProcessor(
                @Qualifier("billingTransactionExportTaskProcessorProperties") TaskProcessorProperties properties,
                BillingTransactionYtExporter billingTransactionYtExporter,
                ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
        ) {
            TaskKeyProvider<String> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                    (activeTasks, limit) -> billingTransactionYtExporter.getExportTasks(activeTasks),
                    (activeTasks) -> billingTransactionYtExporter.countNotExportedTransactions()
            );
            return new TaskProcessor<>(taskKeyProvider,
                    billingTransactionYtExporter::startExportTask,
                    forcedRollbackTxManagerWrapper,
                    TaskProcessorHelper.createDefaultTxDefinition("billingTransactionExportTaskTxDef"),
                    properties
            );
        }
    }

    @ConditionalOnProperty("billing-api.enabled")
    @Configuration
    public static class BillingApiConfiguration {
        @ConfigurationProperties(value = "billing-api", ignoreUnknownFields = false)
        @Bean
        @Validated
        public BillingApiProperties billingApiProperties() {
            return new BillingApiProperties();
        }

        @Bean
        public BillingApiClient billingApiClient(
                @Qualifier("billingApiAhcClientWrapper") AsyncHttpClientWrapper ahcWrapper,
                BillingApiProperties billingApiProperties,
                @Autowired(required = false) TvmWrapper tvm
        ) {
            return new BillingApiClient(ahcWrapper, billingApiProperties, tvm);
        }
    }

    @ConditionalOnProperty("billing-csv-api.enabled")
    @Configuration
    public static class BillingCsvApiConfiguration {
        @ConfigurationProperties(value = "billing-csv-api", ignoreUnknownFields = false)
        @Bean
        @Validated
        public BillingApiProperties billingCsvApiProperties() {
            return new BillingApiProperties();
        }

        @Bean
        public BillingCsvApiClient billingApiCsvClient(
                @Qualifier("billingApiAhcClientWrapper") AsyncHttpClientWrapper ahcWrapper,
                BillingApiProperties billingCsvApiProperties,
                @Autowired(required = false) TvmWrapper tvm
        ){
            return new BillingCsvApiClient(ahcWrapper, billingCsvApiProperties, tvm);
        }
    }

    @ConditionalOnProperty("billing-transactions.act-commit-task.enabled")
    @Configuration
    public static class BillingTransactionActCommitTaskConfiguration {
        @ConfigurationProperties(value = "billing-transactions.act-committer", ignoreUnknownFields = false)
        @Bean
        public BillingTransactionActCommitterProperties billingTransactionActCommitterProperties() {
            return new BillingTransactionActCommitterProperties();
        }

        @Bean
        public BillingTransactionActCommitter billingTransactionActCommitter(
                BillingTransactionRepository billingTransactionRepository,
                BillingTransactionActCommitterProperties actCommitterProperties,
                BillingApiClient billingApiClient,
                ClockService clockService
        ) {
            return new BillingTransactionActCommitter(billingTransactionRepository, billingApiClient,
                    clockService, actCommitterProperties.getActCommitDelay());
        }

        @ConfigurationProperties(value = "billing-transactions.act-commit-task", ignoreUnknownFields = false)
        @Bean
        public TaskProcessorProperties billingTransactionActCommitTaskProcessorProperties() {
            return TaskProcessorProperties.builder()
                    .name("BillingTransactionActCommit")
                    .daemonPoolThreads(true)
                    .gracefulPoolShutdown(true)
                    .build();
        }

        @Bean
        public TaskProcessor<Long> billingTransactionActCommitTaskProcessor(
                @Qualifier("billingTransactionActCommitTaskProcessorProperties") TaskProcessorProperties properties,
                BillingTransactionActCommitter billingTransactionActCommitter,
                ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
        ) {
            TaskKeyProvider<Long> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                    billingTransactionActCommitter::getTransactionIdsWaitingForCommit,
                    billingTransactionActCommitter::countTransactionsWaitingForCommit
            );
            return new TaskProcessor<>(taskKeyProvider,
                    billingTransactionActCommitter::processTransactionWaitingForCommit,
                    forcedRollbackTxManagerWrapper,
                    TaskProcessorHelper.createDefaultTxDefinition("billingTransactionActCommitTaskTxDef"),
                    properties
            );
        }
    }

    @ConditionalOnProperty("billing-partners.agreement-sync-task.enabled")
    @Configuration
    public static class BillingPartnerAgreementSynchronizerTaskConfiguration {
        @ConfigurationProperties(value = "billing-partners.agreement-synchronizer", ignoreUnknownFields = false)
        @Bean
        public BillingPartnerAgreementSynchronizerProperties billingPartnerAgreementSynchronizerProperties() {
            return new BillingPartnerAgreementSynchronizerProperties();
        }

        @Bean
        public BillingPartnerAgreementSynchronizer billingPartnerAgreementSynchronizer(
                BillingPartnerAgreementSynchronizerProperties synchronizerProperties,
                BillingPartnerConfigRepository billingPartnerConfigRepository,
                BillingApiClient billingApiClient,
                ClockService clockService,
                HotelAgreementDictionary hotelAgreementDictionary
        ) {
            return new BillingPartnerAgreementSynchronizer(
                    synchronizerProperties, billingPartnerConfigRepository, billingApiClient, clockService,
                    hotelAgreementDictionary
            );
        }

        @ConfigurationProperties(value = "billing-partners.agreement-sync-task", ignoreUnknownFields = false)
        @Bean
        public TaskProcessorProperties billingPartnerAgreementSynchronizerTaskProcessorProperties() {
            return TaskProcessorProperties.builder()
                    .name("BillingPartnerAgreementSynchronizer")
                    .daemonPoolThreads(true)
                    .gracefulPoolShutdown(true)
                    .build();
        }

        @Bean
        public TaskProcessor<Long> billingPartnerAgreementSynchronizerTaskProcessor(
                @Qualifier("billingPartnerAgreementSynchronizerTaskProcessorProperties") TaskProcessorProperties properties,
                BillingPartnerAgreementSynchronizer billingPartnerAgreementSynchronizer,
                ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper
        ) {
            TaskKeyProvider<Long> taskKeyProvider = TaskProcessorHelper.createTaskKeyProvider(
                    billingPartnerAgreementSynchronizer::getReadyTasks,
                    billingPartnerAgreementSynchronizer::countCountTasks
            );
            return new TaskProcessor<>(taskKeyProvider,
                    billingPartnerAgreementSynchronizer::processTask,
                    forcedRollbackTxManagerWrapper,
                    TaskProcessorHelper.createDefaultTxDefinition("billingPartnerAgreementSynchronizerTaskTxDef"),
                    properties
            );
        }
    }
}
