package ru.yandex.travel.orders.services.hotels;

import java.time.LocalDate;
import java.util.EnumSet;
import java.util.List;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.repository.TrustRefundRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;


@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(BusinessTripDocProperties.class)
public class BusinessTripDocService {
    private final TrustRefundRepository trustRefundRepository;
    private final BusinessTripDocProperties config;
    private final VoucherRepository voucherRepository;
    private final WorkflowRepository workflowRepository;
    private final WorkflowMessageSender workflowMessageSender;


    public CheckResult canGenerateBusinessTripDoc(Order order) {
        if (!OrderCompatibilityUtils.isHotelOrder(order) || !OrderCompatibilityUtils.isConfirmed(order)) {
            return new CheckResult(false, "order must be confirmed");
        }
        HotelOrderItem orderItem = OrderCompatibilityUtils.getOnlyHotelOrderItem(order);
        LocalDate checkoutDate = orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate();
        boolean enoughDaysPassed = checkoutDate.plusDays(config.getMinDaysAfterCheckout() - 1).isBefore(LocalDate.now());
        if (!enoughDaysPassed) {
            return new CheckResult(false,
                    String.format("%s days must be passed after checkout", config.getMinDaysAfterCheckout()));
        }

        if (isOrderRefunded(order)) {
            return new CheckResult(false, "order must not be refunded");
        }

        return new CheckResult(true, "OK to generate business trip doc");
    }

    private boolean isOrderRefunded(Order order) {
        if( ((HotelOrder)order).getEntityState() == EHotelOrderState.OS_REFUNDED ) {
            return true;
        }

        return order.getInvoices().stream().mapToInt(invoice ->
            trustRefundRepository.findAllByInvoiceId(invoice.getId()).size()
        ).sum() > 0;
    }

    public void maybeStartGenerateForOrder(Order order) {
        List<Voucher> vouchers = voucherRepository.findForOrderByTypesAndStates(
                order.getId(),
                EVoucherType.VT_HOTELS_BUSINESS_TRIP,
                EnumSet.of(EVoucherState.VS_NEW, EVoucherState.VS_CREATING));

        if (vouchers.size() == 0) {
            String fileName = String.format("hotels/business_trip_%s.pdf", order.getPrettyId());
            Voucher voucher = Voucher.createDocument(EVoucherType.VT_HOTELS_BUSINESS_TRIP, fileName, order);
            voucher.setCrashWorkflowOnError(false);
            startVoucherWorkflow(voucher);
        }
    }

    private void startVoucherWorkflow(Voucher voucher) {
        voucher = voucherRepository.saveAndFlush(voucher);
        Workflow voucherWorkflow = Workflow.createWorkflowForEntity(voucher,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(voucherWorkflow);
        workflowMessageSender.scheduleEvent(voucher.getWorkflow().getId(), TGenerateVoucher.getDefaultInstance());
    }

    @Data
    @RequiredArgsConstructor
    public static class CheckResult {
        private final boolean canGenerate;
        private final String message;
    }
}
