package ru.yandex.travel.orders.services.hotels_extranet;


import java.time.Instant;

import lombok.RequiredArgsConstructor;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.extranet.TBankOrderInfo;
import ru.yandex.travel.hotels.extranet.TBankOrderInfoDetail;
import ru.yandex.travel.hotels.extranet.TGetDBoyReq;
import ru.yandex.travel.hotels.extranet.TGetDBoyRsp;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;

import static ru.yandex.travel.commons.proto.ProtoUtils.fromInstant;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromLocalDate;
import static ru.yandex.travel.commons.proto.ProtoUtils.toTDate;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getLimit;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getPaymentType;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getTransactionType;

@Service
@RequiredArgsConstructor
public class DBoyFinancialDataProvider {
    private final BankOrderRepository bankOrderRepository;

    public TGetDBoyRsp getBankOrdersData(TGetDBoyReq req) {
        var bankOrders = bankOrderRepository.getRecenttlyUpdatedBankOrders(
                ProtoUtils.toInstant(req.getUpdatedAtFrom()),
                req.hasUpdatedAtTo() ? ProtoUtils.toInstant(req.getUpdatedAtTo()) : Instant.now(),
                PageRequest.of(
                        0, getLimit(req) + 1,
                        Sort.by(Sort.Direction.ASC, "updatedAt")
                )
        );

        var response = TGetDBoyRsp.newBuilder();
        bankOrders.stream()
                .limit(getLimit(req))
                .map(this::mapBankOrder)
                .forEach(response::addBankOrders);
        response.setHasMore(bankOrders.size() > getLimit(req));

        return response.build();
    }

    private TBankOrderInfo mapBankOrder(BankOrder it) {
        var proto = TBankOrderInfo.newBuilder();
        proto.setBankOrderId(it.getBankOrderId());
        proto.setPaymentBatchId(it.getBankOrderPayment().getPaymentBatchId());
        proto.setStatus(it.getStatus().getValue());
        proto.setOebsStatus(it.getOebsPaymentStatus().getValue());
        proto.setSum(it.getSum().toString());
        proto.setDescription(it.getDescription());
        proto.setUpdatedAt(fromInstant(it.getUpdatedAt()));
        proto.setEventTime(toTDate(it.getEventtime()));
        proto.setFetchStatus(it.getBankOrderPayment().getStatus().getValue());

        it.getBankOrderPayment()
                .getDetails()
                .stream().map(this::mapBankOrderDetail)
                .forEach(proto::addDetails);

        return proto.build();
    }

    private TBankOrderInfoDetail mapBankOrderDetail(BankOrderDetail it) {
        var proto = TBankOrderInfoDetail.newBuilder();

        proto.setTrustPaymentId(
                Long.parseLong(
                        (it.getTrustPaymentId() != null ? it.getTrustPaymentId() : it.getTrustRefundId()),
                        16)
        );
        proto.setSum(it.getSum().toString());
        proto.setAgencyCommission(it.getAgencyCommission().toString());
        proto.setContractId(it.getContractId());
        proto.setHandlingTime(fromLocalDate(it.getHandlingTime()));
        proto.setPaymentTime(fromLocalDate(it.getPaymentTime()));
        proto.setPaymentType(getPaymentType(it.getPaymentType()));
        proto.setTransactionType(getTransactionType(it.getTransactionType()));

        proto.setYtId(it.getYtId());
        proto.setId(it.getId());

        return proto.build();
    }
}
