package ru.yandex.travel.orders.services.hotels_extranet;

import java.time.Instant;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.extranet.TBankOrderTransactionInfo;
import ru.yandex.travel.hotels.extranet.TDBoyOrder;
import ru.yandex.travel.hotels.extranet.TFinancialEventInfo;
import ru.yandex.travel.hotels.extranet.TGetDBoyReq;
import ru.yandex.travel.hotels.extranet.TGetDBoyRsp;
import ru.yandex.travel.hotels.extranet.TGuest;
import ru.yandex.travel.hotels.extranet.TPaymentInfo;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.repository.HotelDBoyOrderItemRepository;

import static ru.yandex.travel.commons.proto.ProtoUtils.fromInstant;
import static ru.yandex.travel.commons.proto.ProtoUtils.toTPrice;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getLimit;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getPaymentType;
import static ru.yandex.travel.orders.services.hotels_extranet.HotelsExtranetMappingUtils.getTransactionType;

@Service
@RequiredArgsConstructor
public class DBoyOrderDataProvider {
    private final HotelDBoyOrderItemRepository orderItemRepository;
    private final FinancialEventRepository financialEventRepository;
    private final BillingTransactionRepository billingTransactionRepository;


    public TGetDBoyRsp getOrderData(TGetDBoyReq req) {
        var orders = orderItemRepository.getRecentDBoyOrders(
                ProtoUtils.toInstant(req.getUpdatedAtFrom()),
                req.hasUpdatedAtTo() ? ProtoUtils.toInstant(req.getUpdatedAtTo()) : Instant.now(),
                PageRequest.of(
                        0, getLimit(req) + 1,
                        Sort.by(Sort.Direction.ASC, "updatedAt")
                )
        );

        var response = TGetDBoyRsp.newBuilder();
        orders.stream()
                .limit(getLimit(req))
                .map(this::buildProto)
                .forEach(response::addOrders);
        response.setHasMore(orders.size() > getLimit(req));
        return response.build();
    }

    private TDBoyOrder buildProto(HotelOrderItem hoi) {
        var proto = TDBoyOrder.newBuilder();
        var order = (HotelOrder) hoi.getOrder();
        proto.setTravelOrderGuid(order.getId().toString());
        proto.setTravelPrettyOrderId(order.getPrettyId());
        Optional.ofNullable(hoi.getProviderId())
                .ifPresent(proto::setPartnerOrderId);
        proto.setState(order.getState());
        proto.setCreatedAt(fromInstant(order.getCreatedAt()));
        proto.setUpdatedAt(fromInstant(order.getUpdatedAt()));
        var itinerary = hoi.getHotelItinerary();

        itinerary.getGuests().stream().map(this::mapGuest).forEach(proto::addGuests);

        if (hoi.getBillingPartnerAgreement() != null) {
            proto.setHotelAgreement(
                    ((DirectHotelBillingPartnerAgreement) hoi.getBillingPartnerAgreement())
                            .toProto(hoi.getPartnerId())
            );
        }
        proto.setCheckInDate(ProtoUtils.toTDate(itinerary.getOrderDetails().getCheckinDate()));
        proto.setCheckOutDate(ProtoUtils.toTDate(itinerary.getOrderDetails().getCheckoutDate()));
        Optional.ofNullable(hoi.getRefundedAt()).map(ProtoUtils::fromInstant).ifPresent(proto::setCancellationDateTime);
        proto.setHotelPrice(toTPrice(
                itinerary.getRealHotelPrice()
        ));
        proto.setFiscalPrice(toTPrice(itinerary.getFiscalPrice()));
        Optional.ofNullable(itinerary.getDiscount())
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setDiscount);

        proto.setCostAfterReservation(ProtoUtils.toTPrice(order.calculateTotalCost()));
        proto.setPaymentInfo(TPaymentInfo.newBuilder()
                .setUseDeferred(order.getUseDeferredPayment())
                .setPaidAmount(ProtoUtils.toTPrice(order.calculatePaidAmount()))
                .setPaymentState(order.getPaymentState())
                .build());
        financialEventRepository.findAllByOrderItem(hoi)
                .stream()
                .map(this::mapFinancialEvent)
                .forEach(proto::addFinancialEventInfo);
        return proto.build();
    }

    private TGuest mapGuest(Guest guest) {
        TGuest.Builder proto = TGuest.newBuilder();
        if (guest.hasFilledName()) {
            proto.setFirstName(guest.getFirstName())
                    .setLastName(guest.getLastName());
        }
        proto.setIsChild(guest.isChild());
        Optional.ofNullable(guest.getAge()).ifPresent(proto::setAge);
        return proto.build();
    }

    private TFinancialEventInfo mapFinancialEvent(FinancialEvent financialEvent) {
        var proto = TFinancialEventInfo.newBuilder();
        proto.setFinancialEventId(financialEvent.getId());
        proto.setBillingClientId(financialEvent.getBillingClientId());
        if (financialEvent.getBillingContractId() != null) {
            proto.setBillingContractId(financialEvent.getBillingContractId());
        }
        Optional.ofNullable(financialEvent.getPartnerAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getPartnerRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setPartnerAmount);
        Optional.ofNullable(financialEvent.getFeeAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getFeeRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setFeeAmount);

        // promo code
        // - fee
        Optional.ofNullable(financialEvent.getPromoCodeFeeAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getPromoCodeFeeRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setPromoCodeFeeAmount);
        // - partner
        Optional.ofNullable(financialEvent.getPromoCodePartnerAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getPromoCodePartnerRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setPromoCodePartnerAmount);

        // Plus
        // - fee
        Optional.ofNullable(financialEvent.getPlusFeeAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getPlusFeeRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setPlusFeeAmount);
        // - partner
        Optional.ofNullable(financialEvent.getPlusPartnerAmount())
                .or(() -> Optional.ofNullable(
                                financialEvent.getPlusPartnerRefundAmount())
                        .map(Money::negate))
                .map(ProtoUtils::toTPrice)
                .ifPresent(proto::setPlusPartnerAmount);

        // Topup
        // we don't care about topup in extranet

        billingTransactionRepository.findAllBySourceFinancialEvent(financialEvent)
                .stream()
                .map(this::mapBillingTransaction)
                .forEach(proto::addBankOrderTransactions);
        return proto.build();
    }

    private TBankOrderTransactionInfo mapBillingTransaction(BillingTransaction bt) {
        var proto = TBankOrderTransactionInfo.newBuilder();
        proto.setId(bt.getId());
        proto.setTransactionType(getTransactionType(bt.getTransactionType()));
        proto.setPaymentType(getPaymentType(bt.getPaymentType()));
        proto.setValueAmount(toTPrice(bt.getValue()));
        proto.setCreatedAt(fromInstant(bt.getCreatedAt()));
        proto.setPayoutAt(fromInstant(bt.getPayoutAt()));
        proto.setAccountingActAt(fromInstant(bt.getAccountingActAt()));
        proto.setExportedToYt(bt.isExportedToYt());
        proto.setActCommitted(bt.isActCommitted());
        Optional.ofNullable(bt.getYtId()).ifPresent(proto::setYtId);
        return proto.build();
    }
}
