package ru.yandex.travel.orders.services.indexing;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.repository.OrderChangeRepository;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(OrdersIndexingConfigurationProperties.class)
public class IndexingTaskProcessorConfiguration {
    private final OrdersIndexingService ordersIndexingService;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final OrderChangeRepository orderChangeRepository;
    private final OrdersIndexingConfigurationProperties config;

    @Bean
    @ConditionalOnProperty(name = "orders-indexing.enabled", havingValue = "true", matchIfMissing = true)
    public TaskProcessor<UUID> ordersAwaitingIndexingProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return orderChangeRepository.getOrdersWaitingForIndexing(getLockedTaskKeys(),
                        PageRequest.of(0, maxResultSize));
            }

            @Override
            public long getPendingTasksCount() {
                return orderChangeRepository.countOrdersWaitingForIndexing(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("OrdersIndexer");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_READ_COMMITTED);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, ordersIndexingService::indexOrder,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("OrdersIndexer").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(config.getNumberOfOrdersToIndex())
                        .initialStartDelay(config.getInitialDelay()).scheduleRate(config.getRate()).build());
    }
}
