package ru.yandex.travel.orders.services.indexing;

import java.util.Set;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.hibernate.event.spi.PostUpdateEvent;
import org.hibernate.event.spi.PostUpdateEventListener;
import org.hibernate.persister.entity.EntityPersister;
import org.springframework.stereotype.Component;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderChange;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.workflow.entities.Workflow;

@Component
@Slf4j
public class IndexingUpdateEventListener implements PostUpdateEventListener {
    private static final Set<String> orderSet = Set.of(
            WellKnownWorkflowEntityType.GENERIC_ORDER.getDiscriminatorValue(),
            WellKnownWorkflowEntityType.HOTEL_ORDER.getDiscriminatorValue(),
            WellKnownWorkflowEntityType.AEROFLOT_ORDER.getDiscriminatorValue(),
            WellKnownWorkflowEntityType.TRAIN_ORDER.getDiscriminatorValue());

    @Override
    public void onPostUpdate(PostUpdateEvent event) {
        try {
            final Object entity = event.getEntity();

            if (entity instanceof Order) {
                Order order = (Order) entity;
                insertIntoChanges(event, order.getId());
            } else if (entity instanceof OrderItem) {
                OrderItem orderItem = (OrderItem) entity;
                if (orderItem.getOrder() != null) {
                    insertIntoChanges(event, orderItem.getOrder().getId());
                }
            } else if (entity instanceof Invoice) {
                Invoice inv = (Invoice) entity;
                if (inv.getOrder() != null) {
                    insertIntoChanges(event, inv.getOrder().getId());
                }
            } else if (entity instanceof Workflow) {
                Workflow workflow = (Workflow) entity;
                if (orderSet.contains(workflow.getEntityType())) {
                    insertIntoChanges(event, workflow.getEntityId());
                }
            }
        } catch (Exception e) {
            log.error("An exception has occurred while post update", e);
        }
    }

    private void insertIntoChanges(PostUpdateEvent event, UUID orderId) {
        var orderChange = OrderChange.createOrderChange(orderId);
        event.getSession().persist(orderChange);
    }

    @Override
    public boolean requiresPostCommitHanding(EntityPersister persister) {
        return false;
    }

    @Override
    public boolean requiresPostCommitHandling(EntityPersister persister) {
        return false;
    }
}
