package ru.yandex.travel.orders.services.indexing;

import java.math.BigInteger;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.admin.proto.EBrokenStateEnum;
import ru.yandex.travel.orders.admin.proto.EPartnerType;
import ru.yandex.travel.orders.admin.proto.EPaymentScheduleType;
import ru.yandex.travel.orders.admin.proto.TAdminSorterType;
import ru.yandex.travel.orders.admin.proto.TListOrdersReq;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ESortDirection;
import ru.yandex.travel.orders.entities.AdminFilterValues;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.repository.OrderChangeRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.orders.services.OrdersAdminEnumMapper;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@Service
@RequiredArgsConstructor
public class OrdersIndexingService {
    private final OrdersAdminEnumMapper ordersAdminEnumMapper;
    private final OrderRepository orderRepository;
    private final OrderChangeRepository orderChangeRepository;
    private final EntityManager em;
    private final OrdersIndexingConfigurationProperties properties;
    private final AuthorizationService authorizationService;

    @TransactionMandatory
    public void indexOrder(UUID orderId) {
        Long maxChangeId = orderChangeRepository.findMaxIdForOrder(orderId);
        Optional<Order> optOrder = orderRepository.findById(orderId);
        if (optOrder.isEmpty()) {
            throw new RuntimeException("Order with id = " + orderId + " is not present");
        }
        Order order = optOrder.get();
        boolean indexOrder = !properties.getDisabledOrderTypes().contains(order.getDisplayType());
        if (indexOrder) {
            String sqlString = "INSERT INTO order_search_documents " +
                    "(id, order_id, pretty_id, order_created_at, order_updated_at, provider_ids, " +
                    "phone, email, participants_names, " +
                    "created_at, updated_at, order_type, order_state, partner_type, " +
                    "card_mask, rrn, purchase_token, broken, uses_deferred_payment, display_order_type, " +
                    "yandex_uid, ticket_number, carrier, referral_partner_id) " +
                    "VALUES (:id, :order_id, :pretty_id, :order_created_at, :order_updated_at, " +
                    "to_tsvector(:provider_ids), :phone, :email, to_tsvector(:participants_names), " +
                    ":created_at, :updated_at, :order_type, :order_state, :partner_type, " +
                    "to_tsvector(:card_mask), to_tsvector(:rrn), to_tsvector(:purchase_token), :broken, " +
                    ":uses_deferred_payment, :display_order_type, :yandex_uid, :ticket_number, :carrier, " +
                    ":referral_partner_id)" +
                    "ON CONFLICT (order_id) DO UPDATE SET " +
                    "(pretty_id, order_created_at, order_updated_at, provider_ids, phone, email, " +
                    "participants_names, updated_at, order_type, order_state, partner_type, " +
                    "card_mask, rrn, purchase_token, broken, uses_deferred_payment, display_order_type, " +
                    "yandex_uid, ticket_number, carrier, referral_partner_id) = " +
                    "(:pretty_id, :order_created_at, :order_updated_at, to_tsvector(:provider_ids), :phone, :email, " +
                    "to_tsvector(:participants_names), :updated_at, :order_type, :order_state, :partner_type, " +
                    "to_tsvector(:card_mask), to_tsvector(:rrn), to_tsvector(:purchase_token), :broken, " +
                    ":uses_deferred_payment, :display_order_type, :yandex_uid, :ticket_number, :carrier, " +
                    ":referral_partner_id)";

            Query query = em.createNativeQuery(sqlString);

            // ProviderId
            String provider_ids = order.getOrderItems().stream()
                    .map(item -> {
                        switch (item.getPublicType()) {
                            case PT_EXPEDIA_HOTEL:
                                ExpediaOrderItem expediaItem = (ExpediaOrderItem) item;
                                ExpediaHotelItinerary itinerary = expediaItem.getItinerary();
                                List<String> ids = new ArrayList<>();
                                if (!Strings.isNullOrEmpty(itinerary.getExpediaItineraryId())) {
                                    ids.add(itinerary.getExpediaItineraryId());
                                }
                                if (itinerary.getConfirmation() != null) {
                                    if (!Strings.isNullOrEmpty(itinerary.getConfirmation().getHotelConfirmationId())) {
                                        ids.add(itinerary.getConfirmation().getHotelConfirmationId());
                                    }
                                    if (!Strings.isNullOrEmpty(itinerary.getConfirmation().getPartnerConfirmationId())) {
                                        ids.add(itinerary.getConfirmation().getPartnerConfirmationId());
                                    }
                                }
                                return String.join(", ", ids);
                            case PT_FLIGHT:
                                AeroflotOrderItem aeroflotItem = (AeroflotOrderItem) item;
                                String aviaPnr = aeroflotItem.getAviaPnr();
                                if (!Strings.isNullOrEmpty(aviaPnr)) {
                                    return aviaPnr;
                                } else {
                                    return "";
                                }
                            case PT_DOLPHIN_HOTEL:
                                DolphinOrderItem dolphinOrderItem = (DolphinOrderItem) item;
                                DolphinHotelItinerary dolphinItinerary = dolphinOrderItem.getItinerary();
                                List<String> dolphinIds = new ArrayList<>();
                                if (dolphinItinerary.getConfirmation() != null) {
                                    if (!Strings.isNullOrEmpty(dolphinItinerary.getConfirmation().getHotelConfirmationId())) {
                                        dolphinIds.add(dolphinItinerary.getConfirmation().getHotelConfirmationId());
                                    }
                                    if (!Strings.isNullOrEmpty(dolphinItinerary.getConfirmation().getPartnerConfirmationId())) {
                                        dolphinIds.add(dolphinItinerary.getConfirmation().getPartnerConfirmationId());
                                    }
                                }
                                return String.join(", ", dolphinIds);
                            case PT_TRAVELLINE_HOTEL:
                                TravellineOrderItem travellineOrderItem = (TravellineOrderItem) item;
                                TravellineHotelItinerary travellineHotelItinerary = travellineOrderItem.getItinerary();
                                List<String> travellineIds = new ArrayList<>();
                                if (travellineHotelItinerary.getConfirmation() != null) {
                                    if (!Strings.isNullOrEmpty(travellineHotelItinerary.getConfirmation().getHotelConfirmationId())) {
                                        travellineIds.add(travellineHotelItinerary.getConfirmation().getHotelConfirmationId());
                                    }
                                    if (!Strings.isNullOrEmpty(travellineHotelItinerary.getConfirmation().getPartnerConfirmationId())) {
                                        travellineIds.add(travellineHotelItinerary.getConfirmation().getPartnerConfirmationId());
                                    }
                                }
                                return String.join(", ", travellineIds);
                            case PT_BNOVO_HOTEL:
                                BNovoOrderItem bNovoOrderItem = (BNovoOrderItem) item;
                                BNovoHotelItinerary bNovoHotelItinerary = bNovoOrderItem.getItinerary();
                                List<String> bNovoIds = new ArrayList<>();
                                if (bNovoHotelItinerary.getConfirmation() != null) {
                                    if (!Strings.isNullOrEmpty(bNovoHotelItinerary.getConfirmation().getHotelConfirmationId())) {
                                        bNovoIds.add(bNovoHotelItinerary.getConfirmation().getHotelConfirmationId());
                                    }
                                    if (!Strings.isNullOrEmpty(bNovoHotelItinerary.getConfirmation().getPartnerConfirmationId())) {
                                        bNovoIds.add(bNovoHotelItinerary.getConfirmation().getPartnerConfirmationId());
                                    }
                                }
                                return String.join(", ", bNovoIds);
                            case PT_BRONEVIK_HOTEL:
                                BronevikOrderItem bronevikOrderItem = (BronevikOrderItem) item;
                                BronevikHotelItinerary bronevikHotelItinerary = bronevikOrderItem.getItinerary();
                                List<String> bronevikIds = new ArrayList<>();
                                if (bronevikHotelItinerary.getConfirmation() != null) {
                                    if (!Strings.isNullOrEmpty(bronevikHotelItinerary.getConfirmation().getHotelConfirmationId())) {
                                        bronevikIds.add(bronevikHotelItinerary.getConfirmation().getHotelConfirmationId());
                                    }
                                    if (!Strings.isNullOrEmpty(bronevikHotelItinerary.getConfirmation().getPartnerConfirmationId())) {
                                        bronevikIds.add(bronevikHotelItinerary.getConfirmation().getPartnerConfirmationId());
                                    }
                                }
                                return String.join(", ", bronevikIds);
                            case PT_TRAIN:
                                TrainOrderItem trainOrderItem = (TrainOrderItem) item;
                                return Strings.nullToEmpty(trainOrderItem.getReservation().getReservationNumber());
                            case PT_SUBURBAN:
                                return ((SuburbanOrderItem)item).getPayload().getProviderOrderId();
                            case PT_BUS:
                                List<String> ticketIds = new ArrayList<>();
                                BusReservation reservation = ((BusOrderItem) item).getReservation();
                                if (reservation.getOrder() != null && !reservation.getOrder().getTickets().isEmpty()) {
                                    ticketIds = reservation.getOrder().getTickets().stream()
                                            .map(BusesTicket::getId)
                                            .collect(Collectors.toList());
                                }
                                return String.join(", ", ticketIds);
                            default:
                                return "";
                        }
                    })
                    .collect(Collectors.joining(","));

            // ParticipantNames
            String participantNames = order.getOrderItems().stream()
                    .map(item -> {
                        switch (item.getPublicType()) {
                            case PT_EXPEDIA_HOTEL:
                                return listOfGuests((ExpediaOrderItem) item);
                            case PT_FLIGHT:
                                AeroflotOrderItem aviaItem = (AeroflotOrderItem) item;
                                return aviaItem.getPayload().getTravellers().stream()
                                        .map(traveller -> traveller.getFirstName() + " " + traveller.getMiddleName() + " "
                                                + traveller.getLastName() + " " + traveller.getLastNameSuffix())
                                        .collect(Collectors.toList());
                            case PT_DOLPHIN_HOTEL:
                                return listOfGuests((DolphinOrderItem) item);
                            case PT_TRAVELLINE_HOTEL:
                                return listOfGuests((TravellineOrderItem) item);
                            case PT_BNOVO_HOTEL:
                                return listOfGuests((BNovoOrderItem) item);
                            case PT_BRONEVIK_HOTEL:
                                return listOfGuests((BronevikOrderItem) item);
                            case PT_TRAIN:
                                TrainOrderItem trainOrderItem = (TrainOrderItem) item;
                                return trainOrderItem.getReservation().getPassengers().stream()
                                        .map(passenger -> passenger.getFirstName() + " " + passenger.getLastName())
                                        .collect(Collectors.toList());
                            case PT_BUS:
                                BusOrderItem busOrderItem = (BusOrderItem) item;
                                return busOrderItem.getReservation().getRequestPassengers().stream()
                                        .map(passenger -> passenger.getFirstName() + " " + passenger.getLastName())
                                        .collect(Collectors.toList());
                            default:
                                return new ArrayList<String>();
                        }
                    })
                    .flatMap(List::stream)
                    .collect(Collectors.joining(","));

            EPartnerType partner = null;
            String referralPartnerId = null;
            String ticketNumber = null;
            String carrier = null;
            if (order.getOrderItems().size() > 0) {
                OrderItem item = order.getOrderItems().get(0);
                partner = ordersAdminEnumMapper.getPartnerByItem(item);

                if (item instanceof HotelOrderItem) {
                    HotelItinerary itinerary = ((HotelOrderItem) item).getHotelItinerary();
                    if (itinerary.getActivePromoCampaigns() != null
                            && itinerary.getActivePromoCampaigns().getWhiteLabel() != null
                            && itinerary.getActivePromoCampaigns().getWhiteLabel().getPartnerId() != null)
                        referralPartnerId = itinerary.getActivePromoCampaigns().getWhiteLabel().getPartnerId().name();
                }

                if (item.getPublicType() == EServiceType.PT_SUBURBAN) {
                    SuburbanReservation reservation = ((SuburbanOrderItem) item).getPayload();
                    ticketNumber = reservation.getProviderTicketNumber();
                    carrier = reservation.getCarrierCode();
                }
            }

            AuthorizedUser orderOwner = authorizationService.getOrderOwner(orderId);
            String yandexUid = orderOwner != null ? orderOwner.getYandexUid() : null;

            Instant now = Instant.now();
            query.setParameter("id", UUID.randomUUID());
            query.setParameter("order_id", order.getId());
            query.setParameter("pretty_id", analyzePrettyId(order.getPrettyId()));
            query.setParameter("order_created_at", order.getCreatedAt());
            query.setParameter("order_updated_at", order.getUpdatedAt());
            query.setParameter("provider_ids", provider_ids);
            query.setParameter("phone", order.getPhone());
            query.setParameter("email", order.getEmail().toLowerCase());
            query.setParameter("participants_names", participantNames);
            query.setParameter("created_at", now);
            query.setParameter("updated_at", now);
            query.setParameter("order_type", order.getPublicType().getNumber());
            query.setParameter("display_order_type", order.getDisplayType().getNumber());
            query.setParameter("order_state", EDisplayOrderStateMapper.fromOrder(order).getNumber());
            query.setParameter("partner_type", partner != null ? partner.getNumber() : null);
            query.setParameter("yandex_uid", yandexUid);
            query.setParameter("ticket_number", ticketNumber);
            query.setParameter("carrier", carrier);
            query.setParameter("referral_partner_id", referralPartnerId);

            List<String> rrnBuilder = new ArrayList<>();
            List<String> maskBuilder = new ArrayList<>();
            List<String> purchaseTokenBuilder = new ArrayList<>();
            if (order.getInvoices() != null) {
                order.getInvoices().forEach(invoice -> {
                    if (!Strings.isNullOrEmpty(invoice.getPurchaseToken())) {
                        purchaseTokenBuilder.add(invoice.getPurchaseToken());
                    }
                    if (!Strings.isNullOrEmpty(invoice.getUserAccount())) {
                        maskBuilder.add(invoice.getUserAccount());
                    }
                    if (invoice.getPublicType() == EInvoiceType.IT_TRUST) {
                        TrustInvoice trustInvoice = (TrustInvoice) invoice;
                        if (!Strings.isNullOrEmpty(trustInvoice.getRrn())) {
                            rrnBuilder.add(trustInvoice.getRrn());
                        }
                    }
                });
            }

            query.setParameter("rrn", String.join(", ", rrnBuilder));
            query.setParameter("card_mask", String.join(", ", maskBuilder));
            query.setParameter("purchase_token", String.join(", ", purchaseTokenBuilder));
            query.setParameter("broken", order.isBroken());
            query.setParameter("uses_deferred_payment", order.getUseDeferredPayment());
            query.executeUpdate();
        }

        orderChangeRepository.cleanupByOrderIdAndIdLessThanEqual(orderId, maxChangeId);
    }

    private List<String> listOfGuests(HotelOrderItem orderItem) {
        return orderItem.getHotelItinerary()
                .getGuests()
                .stream()
                .filter(Guest::hasFilledName)
                .map(Guest::getFullName)
                .collect(Collectors.toList());
    }

    private String analyzePrettyId(String prettyId) {
        return StringUtils.replaceChars(prettyId.toUpperCase(), "-", "");
    }

    @TransactionMandatory
    public List<UUID> getListOfOrdersWithAdminFilters(TListOrdersReq req) {
        List<String> filters = prepareFilters(req);

        List<String> sorters = prepareSorters(req);

        String sqlString = "SELECT cast(osd.order_id as varchar) FROM order_search_documents osd";
        if (filters.size() > 0) {
            sqlString = sqlString + "  WHERE " + String.join(" and ", filters);
        }
        if (sorters.size() > 0) {
            sqlString = sqlString + "  ORDER BY " + String.join(", ", sorters);
        }
        sqlString = sqlString + "  OFFSET :offset LIMIT :limit";

        Query query = em.createNativeQuery(sqlString);
        if (req.hasCreatedAtFromFilter()) {
            query.setParameter("created_at_from", ProtoUtils.toInstant(req.getCreatedAtFromFilter()));
        }
        if (req.hasCreatedAtToFilter()) {
            query.setParameter("created_at_to", ProtoUtils.toInstant(req.getCreatedAtToFilter()));
        }
        query.setParameter("offset", req.getPage().getOffset());
        query.setParameter("limit", req.getPage().getLimit());
        @SuppressWarnings("unchecked")
        List<String> resultList = query.getResultList();
        return resultList.stream()
                .map(UUID::fromString)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public Long countOrdersWithAdminFilters(TListOrdersReq request) {
        String sqlString = "SELECT count(osd.order_id) FROM order_search_documents osd";
        Query query = applyFiltersAndCreateQuery(request, sqlString);
        return ((BigInteger) query.getSingleResult()).longValue();
    }

    private Query applyFiltersAndCreateQuery(TListOrdersReq request, String selectQuery) {
        List<String> filters = prepareFilters(request);
        if (filters.size() > 0) {
            selectQuery = selectQuery + "  WHERE " + String.join(" and ", filters);
        }
        Query query = em.createNativeQuery(selectQuery);
        if (request.hasCreatedAtFromFilter()) {
            query.setParameter("created_at_from", ProtoUtils.toInstant(request.getCreatedAtFromFilter()));
        }
        if (request.hasCreatedAtToFilter()) {
            query.setParameter("created_at_to", ProtoUtils.toInstant(request.getCreatedAtToFilter()));
        }
        return query;
    }

    @TransactionMandatory
    public AdminFilterValues getActualStateValues(TListOrdersReq req) {
        String sql = "SELECT DISTINCT osd.display_order_type, osd.partner_type, osd.order_state " +
                "FROM order_search_documents osd";
        Query query = applyFiltersAndCreateQuery(req, sql);
        AdminFilterValues result = new AdminFilterValues();
        @SuppressWarnings("unchecked")
        List<Object[]> resultList = query.getResultList();
        if (resultList != null) {
            resultList.forEach(row -> {
                EDisplayOrderType displayOrderType = EDisplayOrderType.forNumber((Integer) row[0]);
                result.addOrderType(ordersAdminEnumMapper.getOrderTypeFromDisplayOrderType(displayOrderType));
                result.addPartnerType(EPartnerType.forNumber((Integer) row[1]));
                result.addOrderState(EDisplayOrderState.forNumber((Integer) row[2]));
            });
        }
        return result;
    }

    private List<String> prepareFilters(TListOrdersReq request) {
        var filters = new ArrayList<String>();
        // getting direct filters
        if (!Strings.isNullOrEmpty(request.getOrderIdFilter())) {
            filters.add("osd.order_id = '" + request.getOrderIdFilter() + "'");
        }
        if (orderTypeFilterIsPresent(request)) {
            // we keep using EOrderType in API for backward compatibility with frontend
            // while business logic should rely on EDisplayOrderType
            EOrderType orderType = request.getOrderTypeFilter();
            EDisplayOrderType displayOrderType = ordersAdminEnumMapper.getDisplayOrderTypeFromOrderType(orderType);
            filters.add("osd.display_order_type = " + displayOrderType.getNumber());
        }
        if (displayOrderStateFilterIsPresent(request)) {
            filters.add("osd.order_state = " + request.getOrderStateFilter().getNumber());
        }
        if (partnerTypeFilterIsPresent(request)) {
            filters.add("osd.partner_type = " + request.getPartnerTypeFilter().getNumber());
        }
        if (BrokenFlagFilterIsPresent(request)) {
            filters.add("osd.broken = " + (request.getBrokenFlagFilter() == EBrokenStateEnum.BS_BROKEN ? "TRUE" :
                    "FALSE"));
        }
        if (paymentScheduleTypeFilterIsPresent(request)) {
            filters.add("osd.uses_deferred_payment = " +
                    (request.getPaymentScheduleTypeFilter() == EPaymentScheduleType.PST_DEFERRED ? "TRUE" : "FALSE"));
        }
        // getting trgm filters
        if (!Strings.isNullOrEmpty(request.getPrettyIdFilter())) {
            filters.add("osd.pretty_id LIKE '%" + analyzePrettyId(request.getPrettyIdFilter()) + "%'");
        }
        if (!Strings.isNullOrEmpty(request.getEmailFilter())) {
            filters.add("osd.email LIKE '%" + request.getEmailFilter().toLowerCase() + "%'");
        }
        if (!Strings.isNullOrEmpty(request.getPhoneFilter())) {
            filters.add("osd.phone LIKE '%" + request.getPhoneFilter() + "%'");
        }
        if (!Strings.isNullOrEmpty(request.getTicketNumberFilter())) {
            filters.add("osd.ticket_number LIKE '%" + request.getTicketNumberFilter() + "%'");
        }
        if (!Strings.isNullOrEmpty(request.getYandexUidFilter())) {
            filters.add("osd.yandex_uid = '" + request.getYandexUidFilter() + "'");
        }
        if (!Strings.isNullOrEmpty(request.getCarrierFilter())) {
            filters.add("osd.carrier = '" + request.getCarrierFilter() + "'");
        }
        if (!Strings.isNullOrEmpty(request.getReferralPartnerIdFilter())) {
            filters.add(request.getReferralPartnerIdFilter().equals("WL_EMPTY")
                    ? "osd.referral_partner_id IS NULL"
                    : "osd.referral_partner_id = '" + request.getReferralPartnerIdFilter() + "'");
        }
        // setting date filter
        if (request.hasCreatedAtFromFilter()) {
            filters.add("osd.order_created_at >= :created_at_from");
        }
        if (request.hasCreatedAtToFilter()) {
            filters.add("osd.order_created_at <= :created_at_to");
        }
        // setting full text search filters
        if (!Strings.isNullOrEmpty(request.getPassengerNamesFilter())) {
            String fio = Arrays.stream(request.getPassengerNamesFilter().split(" "))
                    .filter(name -> !Strings.isNullOrEmpty(name))
                    .collect(Collectors.joining(" & "));
            filters.add("to_tsquery('" + fio + "') @@ osd.participants_names");
        }
        if (!Strings.isNullOrEmpty(request.getProviderIdFilter())) {
            filters.add("to_tsquery('" + request.getProviderIdFilter() + "') @@ osd.provider_ids");
        }
        if (!Strings.isNullOrEmpty(request.getRrnFilter())) {
            filters.add("to_tsquery('" + request.getRrnFilter() + "') @@ osd.rrn");
        }
        if (!Strings.isNullOrEmpty(request.getPurchaseTokenFilter())) {
            filters.add("to_tsquery('" + request.getPurchaseTokenFilter() + "') @@ osd.purchase_token");
        }
        if (!Strings.isNullOrEmpty(request.getCardMaskFilter())) {
            filters.add("to_tsquery('" + request.getCardMaskFilter() + "') @@ osd.card_mask");
        }
        return filters;
    }

    private boolean BrokenFlagFilterIsPresent(TListOrdersReq request) {
        return enumIsNotOneOf(request.getBrokenFlagFilter(), EBrokenStateEnum.UNRECOGNIZED,
                EBrokenStateEnum.BS_UNKNOWN);
    }

    private boolean partnerTypeFilterIsPresent(TListOrdersReq request) {
        return enumIsNotOneOf(request.getPartnerTypeFilter(), EPartnerType.UNRECOGNIZED, EPartnerType.PT_UNKNOWN);
    }

    private boolean displayOrderStateFilterIsPresent(TListOrdersReq request) {
        return enumIsNotOneOf(request.getOrderStateFilter(), EDisplayOrderState.UNRECOGNIZED,
                EDisplayOrderState.OS_UNKNOWN);
    }

    private boolean orderTypeFilterIsPresent(TListOrdersReq request) {
        return enumIsNotOneOf(request.getOrderTypeFilter(), EOrderType.UNRECOGNIZED, EOrderType.OT_UNKNOWN);
    }

    private boolean paymentScheduleTypeFilterIsPresent(TListOrdersReq request) {
        return enumIsNotOneOf(request.getPaymentScheduleTypeFilter(),
                EPaymentScheduleType.UNRECOGNIZED, EPaymentScheduleType.PST_UNKNOWN);
    }

    private <T> boolean enumIsNotOneOf(T requestEnum, T valueOne, T valueTwo) {
        return requestEnum != valueOne && requestEnum != valueTwo;
    }

    private List<String> prepareSorters(TListOrdersReq request) {
        var sorters = new ArrayList<String>();
        if (request.getSorterList().isEmpty()) {
            sorters.add("osd.order_created_at desc");
            return sorters;
        }
        request.getSorterList().forEach(sorter -> {
            String direction = sorter.getSortDirection() == ESortDirection.SD_ASC ? "ASC" : "DESC";
            if (sorter.getSortField() == TAdminSorterType.ESortField.SF_CREATED_AT) {
                sorters.add("osd.order_created_at " + direction);
            }
        });
        return sorters;
    }
}
