package ru.yandex.travel.orders.services.migrations;

import java.time.Instant;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.Hibernate;

import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.OrderItemMigration;
import ru.yandex.travel.orders.repository.OrderItemMigrationRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@RequiredArgsConstructor
@Slf4j
public class OrderDetailsMigrationService {
    private static final String EXPEDIA_PROVIDER_ID = "3501497177";
    private static final String DOLPHIN_PROVIDER_ID = "3501630980";
    private static final String NOT_FOUND = "NOT_FOUND";

    private final OrderItemRepository itemRepository;
    private final OrderItemMigrationRepository migrationRepository;

    @TransactionMandatory
    public void migrate(UUID itemId) {
        HotelOrderItem hotelOrderItem = (HotelOrderItem) Hibernate.unproxy(itemRepository.getOne(itemId));
        OrderItemMigration migration = migrationRepository.getOne(itemId);
        if (!migration.isDetailsMigrated()) {
            OrderDetails details = null;
            if (migration.getPayload().getOrderInfo() != null) {
                OrderInfo orderInfo = migration.getPayload().getOrderInfo();
                var detailsBuilder = OrderDetails.builder()
                        .checkinDate(orderInfo.getRequestInfo().getCheckinDate())
                        .checkoutDate(orderInfo.getRequestInfo().getCheckoutDate())
                        .hotelName(orderInfo.getPartnerHotelInfo().getName())
                        .roomName(orderInfo.getPartnerRoomInfo().getName())
                        .locationType(orderInfo.getBasicHotelInfo().getLocationType())
                        .permalink(orderInfo.getBasicHotelInfo().getPermalink())
                        .originalId(orderInfo.getPartnerHotelInfo().getPropertyId());
                if (orderInfo.getPartnerHotelInfo().getCheckin() != null) {
                    detailsBuilder.checkinBegin(orderInfo.getPartnerHotelInfo().getCheckin().getBeginTime())
                            .checkinEnd(orderInfo.getPartnerHotelInfo().getCheckin().getEndTime());
                }
                if (orderInfo.getPartnerHotelInfo().getCheckout() != null) {
                    detailsBuilder.checkoutEnd(orderInfo.getPartnerHotelInfo().getCheckout().getTime());
                }
                if (hotelOrderItem instanceof ExpediaOrderItem) {
                    detailsBuilder.providerId(EXPEDIA_PROVIDER_ID);
                } else
                if (hotelOrderItem instanceof DolphinOrderItem) {
                    detailsBuilder.providerId(DOLPHIN_PROVIDER_ID);
                } else {
                    detailsBuilder.providerId(NOT_FOUND);
                }
                details = detailsBuilder.build();
            }
            if (hotelOrderItem.getHotelItinerary().getOrderDetails() == null) {
                log.info("Item {} has no order details, will be migrated", itemId);
                if (details != null) {
                    hotelOrderItem.getHotelItinerary().setOrderDetails(details);
                } else {
                    throw new RuntimeException("Unable to retrieve order details from payload backup");
                }
            } else {
                log.info("Item already {} has some order details, migration is not needed", itemId);
            }
        } else {
            log.warn("Item already {} is already migrated", itemId);
        }
        migration.setDetailsMigrated(true);
        migration.setDetailsMigratedAt(Instant.now());
    }
}
