package ru.yandex.travel.orders.services.migrations;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.commons.jackson.MoneySerializersModule;
import ru.yandex.travel.orders.entities.OrderItemMigration;
import ru.yandex.travel.orders.repository.OrderItemMigrationRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@EnableConfigurationProperties(OrderDetailsMigratorConfigurationProperties.class)
@RequiredArgsConstructor
@Slf4j
public class OrderDetailsMigratorConfiguration {
    private final OrderItemRepository itemRepository;
    private final OrderItemMigrationRepository migrationRepository;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final OrderDetailsMigratorConfigurationProperties properties;

    @Bean
    public OrderDetailsMigrationService migrationService() {
        ObjectMapper mapper = new ObjectMapper();
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(
                LocalDate.class,
                new LocalDateDeserializer(DateTimeFormatter.ISO_DATE));
        javaTimeModule.addSerializer(
                LocalDate.class,
                new LocalDateSerializer(DateTimeFormatter.ISO_DATE));
        mapper.registerModule(javaTimeModule);
        mapper.registerModule(new MoneySerializersModule());
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return new OrderDetailsMigrationService(itemRepository, migrationRepository);
    }

    @Bean
    @ConditionalOnProperty("migration.order-details.enabled")
    public TaskProcessor<UUID> payloadMigratorProcessor(OrderDetailsMigrationService migrationService) {
        TaskKeyProvider<UUID> provider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                var res = migrationRepository.findAllByDetailsMigratedFalse(PageRequest.of(0, maxResultSize));
                return res.stream().map(OrderItemMigration::getId).collect(Collectors.toList());
            }

            @Override
            public long getPendingTasksCount() {
                return migrationRepository.countAllByDetailsMigratedFalse();
            }
        };

        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("LegacyHotelPayloadMigration");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);
        return new TaskProcessor<>(provider, migrationService::migrate, forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder()
                        .name("LegacyHotelPayloadMigration")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(properties.getConcurrentMigrations())
                        .initialStartDelay(properties.getInitialStartDelay())
                        .scheduleRate(properties.getPeriod())
                        .build());
    }
}
