package ru.yandex.travel.orders.services.migrations.generic;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.entities.context.OrderStateContext;
import ru.yandex.travel.orders.entities.migrations.TrainOrderMigration;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.repository.migrations.TrainOrderMigrationRepository;
import ru.yandex.travel.orders.repository.migrations.TrainToGenericMigrationRepository;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.Workflow;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_CANCELLED;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_CONFIRMED;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_NEW;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_REFUNDED;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_REFUNDING;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_WAITING_CANCELLATION;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_WAITING_CONFIRMATION;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_WAITING_PAYMENT;
import static ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState.OS_WAITING_RESERVATION;

@Component
@RequiredArgsConstructor
@Slf4j
public class TrainToGenericMigrationProcessor extends AbstractTaskKeyProvider<UUID> {
    private static final List<UUID> NO_LOCK_UUIDS = List.of(UUID.fromString("0-0-0-0-0"));

    private final EntityManager em;
    private final TrainToGenericMigrationRepository migrationRepository;
    private final TrainOrderMigrationRepository logRepository;

    private Collection<UUID> getLockedKeysSafe() {
        Collection<UUID> keys = getLockedTaskKeys();
        return !keys.isEmpty() ? keys : NO_LOCK_UUIDS;
    }

    @TransactionMandatory
    @Override
    public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
        return migrationRepository.findOldTrainOrderIds(getLockedKeysSafe(), PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    @Override
    public long getPendingTasksCount() {
        return migrationRepository.countOldTrainOrderIds(getLockedKeysSafe());
    }

    @TransactionMandatory
    public void migrate(UUID orderId) {
        try (NestedMdc ignored = NestedMdc.forEntityId(orderId)) {
            log.info("Migrating old train order to the new generic flow");

            TrainOrder oldOrder = em.getReference(TrainOrder.class, orderId);
            boolean moneyAcquired = oldOrder.getMoneyAcquired() == Boolean.TRUE;
            ETrainOrderState oldState = oldOrder.getState();
            EOrderState newState = convertState(oldState);
            log.info("Will change the state from ETrainOrderState.{} to EOrderState.{}", oldState, newState);

            List<OrderRefund> userRefunds = getActiveUserRefunds(oldOrder);
            if (userRefunds.stream().anyMatch(r -> !r.isCompleted()) && newState == OS_CONFIRMED) {
                newState = OS_REFUNDING;
                log.info("Active refunds detected, will use the {} state instead of {}", newState, OS_CONFIRMED);
            }

            // removing from the cache to reload the updated instance
            em.detach(oldOrder);
            migrationRepository.changeOldTrainOrderTypeAndState(orderId, newState);
            GenericOrder newOrder = migrationRepository.getOne(orderId);

            OrderStateContext stateContext = generateStateContext(newOrder, moneyAcquired);
            newOrder.setStateContext(stateContext);
            log.info("Generated state context: {}", stateContext);

            Workflow workflow = newOrder.getWorkflow();
            workflow.setEntityType(WellKnownWorkflowEntityType.GENERIC_ORDER.getDiscriminatorValue());
            log.info("Workflow entity type changed to generic_order too");

            if (!userRefunds.isEmpty()) {
                Collection<UUID> refundIds = userRefunds.stream().map(OrderRefund::getId).collect(toList());
                log.info("Migrating user refunds: {}", refundIds);
                migrationRepository.changeOldTrainRefundTypesStrict(refundIds);
            }

            logMigration(orderId, oldState, newState, stateContext);

            // the call causes all our listeners to be performed before the final commit
            // that's needed to be done as ORM listeners are performed in the following order:
            // [..., insert, update, ..., delete]
            // we generate some extra INSERT actions during the UPDATE actions processing phase,
            // that causes our new INSERT actions to be added to a phase that's already completed;
            // by flushing early we let the INSERTS with indexer events to be processed during the final commit stage
            em.flush();
        }
    }

    private void logMigration(UUID orderId, ETrainOrderState oldState, EOrderState newState,
                              OrderStateContext newStateContext) {
        TrainOrderMigration logRecord = new TrainOrderMigration();
        logRecord.setId(UUID.randomUUID());
        logRecord.setOrderId(orderId);
        logRecord.setOldState(oldState);
        logRecord.setNewState(newState);
        logRecord.setNewStateContext(newStateContext);
        logRecord.setMigratedAt(Instant.now());
        logRepository.save(logRecord);
        log.info("A separate migration log record {} added", logRecord.getId());
    }

    private EOrderState convertState(ETrainOrderState state) {
        switch (state) {
            case OS_NEW:
                return OS_NEW;
            case OS_WAITING_RESERVATION:
                return OS_WAITING_RESERVATION;
            case OS_WAITING_PAYMENT:
                return OS_WAITING_PAYMENT;
            case OS_CANCELLED:
                return OS_CANCELLED;
            case OS_WAITING_CANCELLATION:
                return OS_WAITING_CANCELLATION;
            case OS_WAITING_CONFIRMATION:
                return OS_WAITING_CONFIRMATION;
            case OS_CONFIRMED:
                return OS_CONFIRMED;
            case OS_WAITING_INSURANCE_REFUND:
                return OS_REFUNDING;
            case OS_WAITING_REFUND_AFTER_CANCELLATION:
                return OS_REFUNDING;
            case OS_REFUNDED:
                return OS_REFUNDED;
            case OS_MANUAL_PROCESSING:
                return OS_REFUNDING;
            default:
                throw new UnsupportedOperationException("Unsupported old state: " + state);
        }
    }

    private List<OrderRefund> getActiveUserRefunds(TrainOrder order) {
        return order.getOrderRefunds().stream()
                .filter(r -> r.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND)
                .collect(toList());
    }

    private OrderStateContext generateStateContext(GenericOrder order, boolean moneyAcquired) {
        OrderStateContext context = new OrderStateContext();
        context.init(order.getDisplayType());

        TrainOrderItem orderItem = OrderCompatibilityUtils.getOnlyTrainOrderItem(order);
        context.getOrderItems().add(new OrderItemContextState(
                orderItem.getId(),
                orderItem.getWorkflow().getId(),
                // the processor is disabled already
                // todo(tlg-13): remove the comment during task archiving
                //convertItemState(orderItem.getState())
                null
        ));
        context.setMoneyAcquired(moneyAcquired);

        return context;
    }
/*
    private EOrderItemState convertItemState(ETrainOrderItemState state) {
        switch (state) {
            case IS_NEW_GENERIC:
            case IS_RESERVING_GENERIC:
                return EOrderItemState.IS_RESERVING;
            case IS_RESERVED_GENERIC:
                return EOrderItemState.IS_RESERVED;
            case IS_CANCELLING_GENERIC:
                return EOrderItemState.IS_CANCELLING;
            case IS_CANCELLED_GENERIC:
                return EOrderItemState.IS_CANCELLED;
            case IS_CONFIRMING_GENERIC:
                return EOrderItemState.IS_CONFIRMING;
            case IS_CONFIRMED_GENERIC:
                return EOrderItemState.IS_CONFIRMED;
            case IS_REFUNDING_GENERIC:
                return EOrderItemState.IS_REFUNDING;
            case IS_REFUNDED_GENERIC:
                return EOrderItemState.IS_REFUNDED;
            case IS_CHECKING_CONFIRMATION_TRAINS:
                return EOrderItemState.IS_CONFIRMING;
            case IS_CALCULATING_FEE_TRAINS:
            case IS_INSURANCE_PRICING_TRAINS:
            case IS_RESERVING_INSURANCE_LEGACY:
                return EOrderItemState.IS_RESERVING;
            default:
                throw new UnsupportedOperationException("Unsupported item state: " + state);
        }
    }
*/
}
