package ru.yandex.travel.orders.services.mock;

import java.io.StringWriter;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.travel.orders.commons.proto.EAviaMqEventOutcome;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.mock.MockAeroflotOrder;
import ru.yandex.travel.orders.repository.AeroflotOrderRepository;
import ru.yandex.travel.orders.repository.mock.MockAeroflotOrderRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqRawData;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqService;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Service
@Slf4j
@RequiredArgsConstructor
public class MockAeroflotMqService {
    private final static UUID ZERO_UUID = UUID.fromString("00000000-0000-0000-0000-000000000000");
    private static final DateTimeFormatter mqDateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private final MockAeroflotOrderRepository mockedOrderRepository;
    private final AeroflotOrderRepository orderRepository;
    private final AeroflotMqService aeroflotMqService;

    @TransactionMandatory
    public List<UUID> fetchOrdersWaitingForEvent(Set<UUID> active, int maxResultSize) {
        return mockedOrderRepository.findAllByMqEventResultInAndIdNotInAndSendMqEventAtBeforeOrderBySendMqEventAt(
                List.of(EAviaMqEventOutcome.MEO_SUCCESS),
                active.size() != 0 ? active : List.of(ZERO_UUID),
                Instant.now(),
                PageRequest.of(0, maxResultSize)).stream()
                .map(MockAeroflotOrder::getId)
                .collect(Collectors.toUnmodifiableList());
    }

    @TransactionMandatory
    public long countOrdersWaitingForEvent(Set<UUID> active) {
        return mockedOrderRepository.countAllByMqEventResultInAndIdNotInAndSendMqEventAtNotNull(
                List.of(EAviaMqEventOutcome.MEO_SUCCESS),
                active.size() != 0 ? active : List.of(ZERO_UUID));
    }

    @TransactionMandatory
    public void scheduleMqEvent(UUID orderId) {
        AeroflotOrder aeroflotOrder = orderRepository.getOne(orderId);
        AeroflotServicePayload payload = aeroflotOrder.getAeroflotOrderItem().getPayload();
        String data = makeMockedEventData(payload);
        aeroflotMqService.scheduleMessageProcessing(AeroflotMqRawData.builder()
                .id(UUID.randomUUID().toString())
                .data(data)
                .sendTimestamp(Instant.now())
                .build());
        MockAeroflotOrder mockedOrder = mockedOrderRepository.getOne(orderId);
        mockedOrder.setSendMqEventAt(null);
    }

    private String makeMockedEventData(AeroflotServicePayload payload) {
        try {
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newDefaultInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document document = db.newDocument();

            Element root = document.createElement("cepMessage");
            Element content = document.createElement("content");
            Element bookings = document.createElement("bookings");
            Element booking = document.createElement("booking");

            Element rloc = document.createElement("rloc");
            rloc.appendChild(document.createTextNode(payload.getBookingRef().getPnr()));
            Element bookingDate = document.createElement("bookingDate");
            String pnrDate = LocalDate.parse(payload.getBookingRef().getPnrDate().replace("PNR_date_",
                    "")).atStartOfDay().format(mqDateTimeFormatter);
            bookingDate.appendChild(document.createTextNode(pnrDate));
            Element email = document.createElement("IATA_Notification_Email");
            email.appendChild(document.createTextNode("test@test.ru"));
            Element phone = document.createElement("IATA_Notification_Phone");
            phone.appendChild(document.createTextNode("+79991234556"));
            booking.appendChild(rloc);
            booking.appendChild(bookingDate);
            booking.appendChild(phone);
            booking.appendChild(email);

            Element bookingNames = document.createElement("bookingNames");
            Element bookingName = document.createElement("bookingName");
            Element lastName = document.createElement("lastName");
            lastName.appendChild(document.createTextNode(payload.getTravellers().get(0).getLastName()));
            Element firstName = document.createElement("firstName");
            firstName.appendChild(document.createTextNode(payload.getTravellers().get(0).getFirstName()));
            Element passengerType = document.createElement("passengerType");
            passengerType.appendChild(document.createTextNode("1"));
            bookingName.appendChild(lastName);
            bookingName.appendChild(firstName);
            bookingName.appendChild(passengerType);
            bookingNames.appendChild(bookingName);
            booking.appendChild(bookingNames);

            Element ticketDocuments = document.createElement("TicketDocuments");
            payload.getTravellers().forEach(traveller -> {
                Element ticketDocument = document.createElement("TicketDocument");
                Element ticketNumber = document.createElement("TicketNumber");
                ticketNumber.appendChild(document.createTextNode(String.valueOf(RandomUtils.nextInt(0, 256))));
                ticketDocument.appendChild(ticketNumber);
                ticketDocuments.appendChild(ticketDocument);
            });
            booking.appendChild(ticketDocuments);

            bookings.appendChild(booking);
            content.appendChild(bookings);
            root.appendChild(content);

            Element header = document.createElement("header");
            Element dateSent = document.createElement("dateSent");
            dateSent.appendChild(document.createTextNode(LocalDateTime.now().format(mqDateTimeFormatter)));
            header.appendChild(dateSent);
            root.appendChild(header);

            document.appendChild(root);

            TransformerFactory tf = TransformerFactory.newInstance();
            Transformer transformer = tf.newTransformer();
            StringWriter writer = new StringWriter();
            transformer.transform(new DOMSource(document), new StreamResult(writer));
            return writer.getBuffer().toString();
        } catch (ParserConfigurationException | TransformerException e) {
            log.error("Failed to construct XmlEvent", e);
            return null;
        }
    }
}
