package ru.yandex.travel.orders.services.mock;

import java.time.Duration;
import java.time.Instant;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import kotlin.NotImplementedError;
import lombok.RequiredArgsConstructor;
import org.javamoney.moneta.Money;
import org.jetbrains.annotations.NotNull;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.bus.service.BusesService;
import ru.yandex.travel.bus.service.BusesServiceImpl;
import ru.yandex.travel.buses.backend.proto.api.CreateRideOfferRequest;
import ru.yandex.travel.buses.backend.proto.api.CreateRideOfferResponse;
import ru.yandex.travel.buses.backend.proto.api.GetOfferRequest;
import ru.yandex.travel.buses.backend.proto.api.GetOfferResponse;
import ru.yandex.travel.buses.backend.proto.worker.EOrderStatus;
import ru.yandex.travel.buses.backend.proto.worker.ETicketStatus;
import ru.yandex.travel.buses.backend.proto.worker.ETicketVAT;
import ru.yandex.travel.buses.backend.proto.worker.TBookRequest;
import ru.yandex.travel.buses.backend.proto.worker.TBookResponse;
import ru.yandex.travel.buses.backend.proto.worker.TCancelBookingRequest;
import ru.yandex.travel.buses.backend.proto.worker.TCancelBookingResponse;
import ru.yandex.travel.buses.backend.proto.worker.TConfirmRequest;
import ru.yandex.travel.buses.backend.proto.worker.TConfirmResponse;
import ru.yandex.travel.buses.backend.proto.worker.TOrder;
import ru.yandex.travel.buses.backend.proto.worker.TRefund;
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfo;
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoRequest;
import ru.yandex.travel.buses.backend.proto.worker.TRefundInfoResponse;
import ru.yandex.travel.buses.backend.proto.worker.TRefundRequest;
import ru.yandex.travel.buses.backend.proto.worker.TRefundResponse;
import ru.yandex.travel.buses.backend.proto.worker.TResponseHeader;
import ru.yandex.travel.buses.backend.proto.worker.TTicket;
import ru.yandex.travel.buses.backend.proto.worker.TTicketPassenger;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EBusBookOutcome;
import ru.yandex.travel.orders.commons.proto.EBusConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.EBusRefundInfoOutcome;
import ru.yandex.travel.orders.commons.proto.EBusRefundOutcome;
import ru.yandex.travel.orders.commons.proto.TBusTestContext;
import ru.yandex.travel.orders.entities.mock.MockBusOrder;
import ru.yandex.travel.orders.repository.mock.MockBusOrderRepository;

@RequiredArgsConstructor
public class MockBusesService implements BusesService {
    private final MockBusOrderRepository mockBusOrderRepository;
    private final TransactionTemplate transactionTemplate;

    private static final Duration EXPIRE_TIME = Duration.ofMinutes(15);
    private static final Money TICKET_PRICE = Money.of(970.0, ProtoCurrencyUnit.RUB);
    private static final Money TICKET_REVENUE = Money.of(19.4, ProtoCurrencyUnit.RUB);
    private static final Money REFUND_PRICE = Money.of(550.0, ProtoCurrencyUnit.RUB);
    private static final Set<EOrderStatus> CONFIRMED_ORDER_STATUSES = Set.of(EOrderStatus.OS_SOLD,
            EOrderStatus.OS_PARTIAL_RETURNED, EOrderStatus.OS_RETURNED);

    @NotNull
    @Override
    public TBookResponse book(@NotNull TBookRequest req, Message testContextMsg) {
        Preconditions.checkArgument(testContextMsg instanceof TBusTestContext);
        var testContext = (TBusTestContext) testContextMsg;
        var rsp = withTx(req, (request) -> {
            var now = Instant.now();
            var rspHeader = TResponseHeader.newBuilder();
            String workerOrderId = request.getSupplierId() + ":" + mockBusOrderRepository.getNextOrderId();
            MockBusOrder mockBusOrder = new MockBusOrder();
            mockBusOrder.setId(workerOrderId);
            mockBusOrder.setTestContext(testContext);
            mockBusOrder.getPayload().setRideId(request.getRideId());
            if (mockBusOrder.getTestContext().getBookOutcome() == EBusBookOutcome.BBO_FAILURE) {
                rspHeader.setCode(EErrorCode.EC_GENERAL_ERROR);
            }
            Duration expireTime = EXPIRE_TIME;
            if (testContext.getExpireAfterSeconds() != 0) {
                expireTime = Duration.ofSeconds(testContext.getExpireAfterSeconds());
            }
            var rspOrder = TOrder.newBuilder()
                    .setExpiresAt(ProtoUtils.fromInstant(now.plus(expireTime)))
                    .setStatus(testContext.getBookOutcome() == EBusBookOutcome.BBO_SUCCESS ?
                            EOrderStatus.OS_BOOKED : EOrderStatus.OS_CANCELLED)
                    .setId(workerOrderId);
            for (var p : request.getPassengersList()) {
                rspOrder.addTickets(TTicket.newBuilder()
                        .setId(mockBusOrderRepository.getNextOrderTicketId().toString())
                        .setStatus(testContext.getBookOutcome() == EBusBookOutcome.BBO_SUCCESS ?
                                ETicketStatus.TS_BOOKED : ETicketStatus.TS_CANCELLED)
                        .setPrice(ProtoUtils.toTPrice(TICKET_PRICE))
                        .setRevenue(ProtoUtils.toTPrice(TICKET_REVENUE))
                        .setPriceVat(ETicketVAT.TV_NDS_NONE)
                        .setFeeVat(ETicketVAT.TV_NDS_NONE)
                        .setPassenger(TTicketPassenger.newBuilder()
                                .setGender(p.getGender().getId())
                                .setCitizenship(p.getCitizenship().getId())
                                .setDocumentType(p.getDocumentType().getId())
                                .setDocumentNumber(p.getDocumentNumber())
                                .setTicketType(p.getTicketType().getId())
                                .setSeat(p.getSeat().getId())
                                .setFirstName(p.getFirstName())
                                .setMiddleName(p.getMiddleName())
                                .setLastName(p.getLastName())
                                .setBirthDate(p.getBirthDate())
                                .build())
                        .build());
            }
            mockBusOrder.setProtoOrder(rspOrder.build());
            mockBusOrderRepository.save(mockBusOrder);
            return TBookResponse.newBuilder()
                    .setHeader(rspHeader.build())
                    .setOrder(mockBusOrder.getProtoOrder())
                    .build();
        });
        BusesServiceImpl.rethrowHeaderErrors(rsp.getHeader());
        return rsp;
    }

    @NotNull
    @Override
    public TConfirmResponse confirm(@NotNull TConfirmRequest req) {
        var rsp = withTx(req, (request) -> {
            MockBusOrder mockBusOrder = mockBusOrderRepository.getOne(req.getOrderId());
            var rspHeader = TResponseHeader.newBuilder();
            if (mockBusOrder.getProtoOrder().getStatus() == EOrderStatus.OS_BOOKED) {
                if (mockBusOrder.getTestContext().getConfirmOutcome() == EBusConfirmOutcome.BCO_FAILURE) {
                    rspHeader.setCode(EErrorCode.EC_GENERAL_ERROR);
                } else {
                    var protoOrder = mockBusOrder.getProtoOrder().toBuilder();
                    protoOrder.setStatus(EOrderStatus.OS_SOLD);
                    for (var t : protoOrder.getTicketsBuilderList()) {
                        t.setStatus(ETicketStatus.TS_SOLD)
                                .setCode("1")
                                .setSeries("1")
                                .setNumber("1")
                                .setBarcode("")
                                .setPlatform("9 3/4");
                    }
                    mockBusOrder.setProtoOrder(protoOrder.build());
                }
            } else {
                rspHeader.setCode(EErrorCode.EC_FAILED_PRECONDITION);
            }

            return TConfirmResponse.newBuilder()
                    .setHeader(rspHeader.build())
                    .setOrder(mockBusOrder.getProtoOrder())
                    .build();
        });
        BusesServiceImpl.rethrowHeaderErrors(rsp.getHeader());
        return rsp;
    }

    @NotNull
    @Override
    public TCancelBookingResponse cancelBooking(@NotNull TCancelBookingRequest req) {
        var rsp = withTx(req, (request) -> {
            MockBusOrder mockBusOrder = mockBusOrderRepository.getOne(req.getOrderId());
            var rspHeader = TResponseHeader.newBuilder();
            if (mockBusOrder.getProtoOrder().getStatus() == EOrderStatus.OS_BOOKED) {
                var protoOrder = mockBusOrder.getProtoOrder().toBuilder();
                protoOrder.setStatus(EOrderStatus.OS_CANCELLED);
                for (var t : protoOrder.getTicketsBuilderList()) {
                    t.setStatus(ETicketStatus.TS_CANCELLED);
                }
                mockBusOrder.setProtoOrder(protoOrder.build());
            } else {
                rspHeader.setCode(EErrorCode.EC_FAILED_PRECONDITION);
            }
            return TCancelBookingResponse.newBuilder()
                    .setHeader(rspHeader)
                    .build();
        });
        BusesServiceImpl.rethrowHeaderErrors(rsp.getHeader());
        return rsp;
    }

    @NotNull
    @Override
    public TRefundInfoResponse refundInfo(@NotNull TRefundInfoRequest req) {
        var rsp = withTx(req, (request) -> {
            MockBusOrder mockBusOrder = mockBusOrderRepository.getOne(req.getOrderId());
            var rspHeader = TResponseHeader.newBuilder();
            var refundInfo = TRefundInfo.newBuilder();
            if (mockBusOrder.getTestContext().getRefundInfoOutcome() == EBusRefundInfoOutcome.BRIO_FAILURE) {
                rspHeader.setCode(EErrorCode.EC_GENERAL_ERROR);
            } else if (CONFIRMED_ORDER_STATUSES.contains(mockBusOrder.getProtoOrder().getStatus())) {
                var ticket = mockBusOrder.getProtoOrder().getTicketsList().stream()
                        .filter(x -> x.getId().equals(req.getTicketId())).findFirst().orElse(null);
                if (ticket == null) {
                    rspHeader.setCode(EErrorCode.EC_NOT_FOUND);
                } else if (ticket.getStatus() == ETicketStatus.TS_SOLD) {
                    refundInfo.setAvailable(true);
                    refundInfo.setPrice(ProtoUtils.toTPrice(REFUND_PRICE));
                }
            } else {
                rspHeader.setCode(EErrorCode.EC_FAILED_PRECONDITION);
            }
            return TRefundInfoResponse.newBuilder()
                    .setHeader(rspHeader)
                    .setRefundInfo(refundInfo.build())
                    .build();
        });
        BusesServiceImpl.rethrowHeaderErrors(rsp.getHeader());
        return rsp;
    }

    @NotNull
    @Override
    public TRefundResponse refund(@NotNull TRefundRequest req) {
        var rsp = withTx(req, (request) -> {
            MockBusOrder mockBusOrder = mockBusOrderRepository.getOne(req.getOrderId());
            var rspHeader = TResponseHeader.newBuilder();
            var refund = TRefund.newBuilder();
            if (mockBusOrder.getTestContext().getRefundInfoOutcome() == EBusRefundInfoOutcome.BRIO_FAILURE) {
                rspHeader.setCode(EErrorCode.EC_GENERAL_ERROR);
            } else if (mockBusOrder.getTestContext().getRefundOutcome() == EBusRefundOutcome.BRO_FAILURE) {
                rspHeader.setCode(EErrorCode.EC_GENERAL_ERROR);
            } else if (CONFIRMED_ORDER_STATUSES.contains(mockBusOrder.getProtoOrder().getStatus())) {
                var protoOrder = mockBusOrder.getProtoOrder().toBuilder();
                var ticket = protoOrder.getTicketsBuilderList().stream()
                        .filter(x -> x.getId().equals(req.getTicketId())).findFirst().orElse(null);
                if (ticket == null) {
                    rspHeader.setCode(EErrorCode.EC_NOT_FOUND);
                } else if (ticket.getStatus() == ETicketStatus.TS_SOLD) {
                    refund.setPrice(ProtoUtils.toTPrice(REFUND_PRICE));
                    ticket.setStatus(ETicketStatus.TS_RETURNED);
                    mockBusOrder.setProtoOrder(protoOrder.build());
                    if (protoOrder.getTicketsBuilderList().stream().allMatch(x -> x.getStatus() == ETicketStatus.TS_RETURNED)) {
                        protoOrder.setStatus(EOrderStatus.OS_RETURNED);
                    } else {
                        protoOrder.setStatus(EOrderStatus.OS_PARTIAL_RETURNED);
                    }
                } else {
                    rspHeader.setCode(EErrorCode.EC_FAILED_PRECONDITION);
                }
            } else {
                rspHeader.setCode(EErrorCode.EC_FAILED_PRECONDITION);
            }
            return TRefundResponse.newBuilder()
                    .setHeader(rspHeader.build())
                    .setRefund(refund.build())
                    .build();
        });
        BusesServiceImpl.rethrowHeaderErrors(rsp.getHeader());
        return rsp;
    }

    private <ReqT, RspT> RspT withTx(ReqT request, Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(request));
    }

    @NotNull
    @Override
    public CompletableFuture<CreateRideOfferResponse> rideDetails(@NotNull CreateRideOfferRequest request) {
        throw new NotImplementedError();
    }

    @NotNull
    @Override
    public CompletableFuture<GetOfferResponse> getOffer(@NotNull GetOfferRequest request) {
        throw new NotImplementedError();
    }
}
