package ru.yandex.travel.orders.services.mock;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.commons.proto.EAviaTokenizationOutcome;
import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.TAviaPaymentTestContext;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.mock.MockTrustBasket;
import ru.yandex.travel.orders.entities.mock.MockTrustBasketInternalState;
import ru.yandex.travel.orders.entities.mock.MockTrustBasketType;
import ru.yandex.travel.orders.entities.mock.MockTrustInternalOrderRecord;
import ru.yandex.travel.orders.entities.mock.MockTrustRefund;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustAccountRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustBasketRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustRefundRepository;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponseOrder;
import ru.yandex.travel.orders.services.payments.model.TrustBindingToken;
import ru.yandex.travel.orders.services.payments.model.TrustClearResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketOrder;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.payments.model.TrustRefundState;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.services.payments.model.TrustResizeResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResponseStatus;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustUnholdResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStartResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Slf4j
public class MockDBTrustClient implements TrustClient {

    private final MockTrustBasketRepository mockTrustBasketRepository;

    private final MockTrustRefundRepository mockTrustRefundRepository;

    @SuppressWarnings("FieldCanBeLocal")
    private final MockTrustAccountRepository accountRepository;

    private final TrustInvoiceRepository trustInvoiceRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final String receiptUrl;

    private final Clock clock;

    public MockDBTrustClient(MockTrustBasketRepository mockTrustBasketRepository,
                             MockTrustRefundRepository mockTrustRefundRepository,
                             MockTrustAccountRepository accountRepository, TrustInvoiceRepository trustInvoiceRepository,
                             WorkflowMessageSender workflowMessageSender,
                             Clock clock,
                             TrustDBMockProperties trustDBMockProperties) {
        this.mockTrustBasketRepository = mockTrustBasketRepository;
        this.mockTrustRefundRepository = mockTrustRefundRepository;
        this.accountRepository = accountRepository;
        this.trustInvoiceRepository = trustInvoiceRepository;
        this.workflowMessageSender = workflowMessageSender;
        this.receiptUrl = trustDBMockProperties.getReceiptUrl();
        this.clock = clock;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustCreateOrderResponse createOrder(String productId, TrustUserInfo userInfo) {
        Long nextOrderId = mockTrustBasketRepository.getNextOrderId();
        TrustCreateOrderResponse response = new TrustCreateOrderResponse();
        response.setProductId(productId);
        response.setOrderId("order" + (nextOrderId));
        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustCreateBasketResponse createBasket(TrustCreateBasketRequest request,
                                                  TrustUserInfo userInfo,
                                                  Object testContext) {

        MockTrustBasket basket = new MockTrustBasket();
        basket.setId(UUID.randomUUID());
        basket.setState(PaymentStatusEnum.NOT_STARTED);

        MockTrustBasketInternalState payload = new MockTrustBasketInternalState();
        payload.setOrders(new HashMap<>());

        if (FiscalItemType.FLIGHT_AEROFLOT.getTrustId().equals(request.getProductId())) {
            basket.setType(MockTrustBasketType.CARD_TOKENIZATION);

            MockTrustInternalOrderRecord intOrder = new MockTrustInternalOrderRecord();
            intOrder.setOrderId(request.getProductId());
            BigDecimal price = BigDecimal.valueOf(Double.parseDouble(request.getAmount()));
            intOrder.setPrice(price);
            intOrder.setOriginalPrice(price);
            payload.getOrders().put(intOrder.getOrderId(), intOrder);
        } else {
            basket.setType(MockTrustBasketType.STANDARD);

            for (TrustCreateBasketOrder order : request.getOrders()) {
                MockTrustInternalOrderRecord intOrder = new MockTrustInternalOrderRecord();
                intOrder.setOrderId(order.getOrderId());
                intOrder.setPrice(order.getPrice());
                intOrder.setOriginalPrice(order.getPrice());
                payload.getOrders().put(intOrder.getOrderId(), intOrder);
            }
        }

        basket.setPayload(payload);

        if (testContext instanceof TAviaPaymentTestContext) {
            basket.setTokenizationOutcome(((TAviaPaymentTestContext) testContext).getTokenizationOutcome());
        }

        if (testContext instanceof TPaymentTestContext) {
            basket.setTestContext((TPaymentTestContext) testContext);
        }

        mockTrustBasketRepository.save(basket);

        TrustCreateBasketResponse response = new TrustCreateBasketResponse();
        response.setPurchaseToken(basket.getId().toString());
        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustStartPaymentResponse startPayment(String purchaseToken, TrustUserInfo userInfo) {

        MockTrustBasket mockTrustBasket = getTrustBasket(purchaseToken);

        TrustStartPaymentResponse response = new TrustStartPaymentResponse();
        // our crowd testers must not see the payment form.
        // so we don't return any payment url
        // response.setPaymentUrl("http://test.payments/payments/?token=" + purchaseToken);

        mockTrustBasket.setState(PaymentStatusEnum.STARTED);
        mockTrustBasket.getPayload().setTrustPaymentId(UUID.randomUUID().toString());

        TPaymentTestContext paymentTestContext = getPaymentTestContext(mockTrustBasket);

        if (paymentTestContext != null && paymentTestContext.hasUserActionDelay() &&
                paymentTestContext.getUserActionDelay().getDelayMax() > 0) {
            long sleepTime =
                    ThreadLocalRandom.current().nextLong(paymentTestContext.getUserActionDelay().getDelayMin(),
                            paymentTestContext.getUserActionDelay().getDelayMax() + 1);
            mockTrustBasket.setUserActionMustOccurAt(Instant.now(clock).plusMillis(sleepTime));
        } else {
            mockTrustBasket.setUserActionMustOccurAt(Instant.now(clock));
        }

        if (paymentTestContext != null && !Strings.isNullOrEmpty(paymentTestContext.getPaymentUrl())) {
            response.setPaymentUrl(paymentTestContext.getPaymentUrl());
        }

        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustClearResponse clear(String purchaseToken, TrustUserInfo userInfo) {
        MockTrustBasket basket = getTrustBasket(purchaseToken);
        basket.setState(PaymentStatusEnum.CLEARED);
        basket.setClearedAt(Instant.now(clock));
        return new TrustClearResponse();
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustUnholdResponse unhold(String purchaseToken, TrustUserInfo userInfo) {
        MockTrustBasket basket = getTrustBasket(purchaseToken);
        Preconditions.checkState(basket.getState() == PaymentStatusEnum.AUTHORIZED, "Basket state must be AUTHORIZED");
        basket.getPayload().getOrders().forEach((k, v) ->
                v.setPrice(BigDecimal.ZERO));
        basket.setState(PaymentStatusEnum.CANCELED);
        return new TrustUnholdResponse();
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustResizeResponse resize(String purchaseToken, String orderId,
                                      TrustResizeRequest resizeRequest, TrustUserInfo userInfo) {

        MockTrustBasket basket = getTrustBasket(purchaseToken);
        Preconditions.checkState(basket.getState() == PaymentStatusEnum.AUTHORIZED, "Basket state must be AUTHORIZED");
        MockTrustInternalOrderRecord order = Preconditions.checkNotNull(basket.getPayload().getOrders().get(orderId),
                "Order %s not found in basket", orderId);
        order.setPrice(resizeRequest.getAmount());
        if (basket.getPayload().getTotal().compareTo(BigDecimal.ZERO) == 0) {
            basket.setState(PaymentStatusEnum.CANCELED);
        }
        return new TrustResizeResponse();
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustCreateRefundResponse createRefund(TrustCreateRefundRequest request,
                                                  TrustUserInfo userInfo) {

        MockTrustBasket basket = getTrustBasket(request.getPurchaseToken());

        MockTrustRefund refund = MockTrustRefund.refundFor(
                request.getReasonDesc(),
                request.getOrders().stream().collect(
                        Collectors.toMap(TrustCreateRefundRequest.Order::getOrderId,
                                TrustCreateRefundRequest.Order::getDeltaAmount)
                )
        );

        basket.addTrustRefund(refund);

        TrustCreateRefundResponse response = new TrustCreateRefundResponse();
        response.setTrustRefundId(refund.getId().toString());
        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustStartRefundResponse startRefund(String refundId, TrustUserInfo userInfo) {
        UUID id = UUID.fromString(refundId);
        MockTrustRefund refund = mockTrustRefundRepository.getOne(id);

        MockTrustBasket basket = refund.getTrustBasket();

        for (Map.Entry<String, BigDecimal> entry : refund.getPayload().getOrderDeltas().entrySet()) {
            // we don't care about sum integrity here, as the exception will stop the test
            MockTrustInternalOrderRecord order =
                    Preconditions.checkNotNull(basket.getPayload().getOrders().get(entry.getKey()),
                            "Order %s not found", entry.getKey());
            order.setPrice(order.getPrice().subtract(entry.getValue()));
        }
        refund.setState(TrustRefundState.SUCCESS);
        return new TrustStartRefundResponse();
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustRefundStatusResponse getRefundStatus(String refundId, TrustUserInfo userInfo) {
        UUID id = UUID.fromString(refundId);
        MockTrustRefund refund = mockTrustRefundRepository.getOne(id);
        TrustRefundStatusResponse response = new TrustRefundStatusResponse();
        if (refund.getState() == TrustRefundState.SUCCESS) {
            response.setFiscalReceiptUrl(receiptUrl);
        }
        response.setStatus(refund.getState());
        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustBasketStatusResponse getBasketStatus(String purchaseToken, TrustUserInfo userInfo) {
        log.info("Getting payment info for purchase token {}", purchaseToken);
        MockTrustBasket basket = getTrustBasket(purchaseToken);
        TPaymentTestContext paymentTestContenxt = getPaymentTestContext(basket);
        TrustBasketStatusResponse response = new TrustBasketStatusResponse();
        response.setPaymentStatus(basket.getState());
        response.setOrders(
                basket.getPayload().getOrders().values().stream().map(internalOrder -> {
                    TrustBasketStatusResponseOrder order = new TrustBasketStatusResponseOrder();
                    order.setPaidAmount(internalOrder.getPrice());
                    order.setOrderId(internalOrder.getOrderId());
                    order.setOrigAmount(internalOrder.getOriginalPrice());
                    return order;
                }).collect(Collectors.toList()));
        if (Strings.emptyToNull(basket.getPayload().getTrustPaymentId()) != null) {
            response.setTrustPaymentId(basket.getPayload().getTrustPaymentId());
        }
        switch (basket.getState()) {
            case AUTHORIZED:
                response.setFiscalReceiptUrl(receiptUrl);
                response.setPaymentTs(basket.getAuthorizedAt());
                break;
            case CLEARED:
                response.setFiscalReceiptUrl(receiptUrl);
                response.setFiscalReceiptClearingUrl(receiptUrl);
                response.setClearTs(basket.getClearedAt());
                if (basket.getType() != null && basket.getType() == MockTrustBasketType.CARD_TOKENIZATION) {
                    response.setBindingToken(TrustBindingToken.builder()
                            .expiration(LocalDateTime.ofInstant(basket.getClearedAt().plus(1, ChronoUnit.HOURS),
                                    ZoneId.systemDefault()))
                            .value(UUID.randomUUID().toString())
                            .build());
                }
                break;
            case NOT_AUTHORIZED:
                response.setPaymentRespCode("USER_CANCELLED");
                response.setPaymentRespDesc("User cancelled due to mocked trust");
                if (paymentTestContenxt != null) {
                    if (!paymentTestContenxt.getPaymentFailureResponseCode().isEmpty()) {
                        response.setPaymentRespCode(paymentTestContenxt.getPaymentFailureResponseCode());
                    }
                    if (!paymentTestContenxt.getPaymentFailureResponseDescription().isEmpty()) {
                        response.setPaymentRespDesc(paymentTestContenxt.getPaymentFailureResponseDescription());
                    }
                }
                break;
        }
        return response;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public TrustPaymentMethodsResponse getPaymentMethods(TrustUserInfo userInfo) {
        return TrustPaymentMethodsResponse.builder()
                .boundPaymentMethods(List.of())
                .build();
    }

    @Override
    public TrustCreateAccountResponse createAccount(TrustCreateAccountRequest request, TrustUserInfo userInfo) {
        throw new UnsupportedOperationException("Mock accounts aren't supported yet");
    }

    @Override
    public TrustCreateTopupResponse createTopup(TrustTopupRequest request, TrustUserInfo userInfo) {
        // we need to use a test context to determine useful topup outcomes
        /*
        CreateTopupResponse response = new CreateTopupResponse();
        // we do everything in one step in mock, no need to track
        response.setPurchaseToken(UUID.randomUUID().toString());
        if (request.getAmount() == null || userInfo.getUid() == null) {
            response.setStatus(TrustResponseStatus.ERROR);
            return response;
        }
        MockTrustAccount existing = accountRepository.findByUid(userInfo.getUid());
        if (existing == null) {
            MockTrustAccount newAccount = new MockTrustAccount();
            accountRepository.save(newAccount);
        } else {
            existing.add(BigDecimal.valueOf(request.getAmount()));
            accountRepository.save(existing);
        }
        response.setStatus(TrustResponseStatus.SUCCESS);
        return response;
        */
        return TrustCreateTopupResponse.builder().build();
    }

    @Override
    public TrustTopupStatusResponse getTopupStatus(String purchaseToken, TrustUserInfo userInfo) {
        TrustTopupStatusResponse response = new TrustTopupStatusResponse();
        response.setStatus(TrustResponseStatus.SUCCESS);
        response.setPurchaseToken(purchaseToken);
        return response;
    }

    @Override
    public TrustTopupStartResponse startTopup(String purchaseToken, TrustUserInfo userInfo) {
        // we do everything in one step for simplicity
        TrustTopupStartResponse response = new TrustTopupStartResponse();
        response.setPurchaseToken(purchaseToken);
        response.setStatus(TrustResponseStatus.SUCCESS);
        return response;
    }

    @Override
    public TrustPaymentReceiptResponse getReceipt(String purchaseToken, String receiptId, TrustUserInfo userInfo) {
        throw new UnsupportedOperationException("Mock getReceipt is not supported");
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public CompletableFuture<TrustPaymentMethodsResponse> getPaymentMethodsAsync(TrustUserInfo userInfo) {
        return CompletableFuture.completedFuture(getPaymentMethods(userInfo));
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void paymentAuthorized(String purchaseToken) {
        MockTrustBasket mockTrustBasket = getTrustBasket(purchaseToken);
        Preconditions.checkState(mockTrustBasket.getState() == PaymentStatusEnum.STARTED,
                "Only payment in STARTED state can be authorized");
        if (mockTrustBasket.getType() != null && mockTrustBasket.getType() == MockTrustBasketType.CARD_TOKENIZATION) {
            mockTrustBasket.setState(PaymentStatusEnum.CLEARED);
            mockTrustBasket.setClearedAt(Instant.now());
        } else {
            mockTrustBasket.setState(PaymentStatusEnum.AUTHORIZED);
            mockTrustBasket.setAuthorizedAt(Instant.now());
        }
    }

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void paymentNotAuthorized(String purchaseToken) {
        MockTrustBasket mockTrustBasket = getTrustBasket(purchaseToken);
        Preconditions.checkState(mockTrustBasket.getState() == PaymentStatusEnum.STARTED,
                "Only payment in STARTED state can be authorized");
        mockTrustBasket.setState(PaymentStatusEnum.NOT_AUTHORIZED);
    }

    private MockTrustBasket getTrustBasket(String purchaseToken) {
        UUID basketId = UUID.fromString(purchaseToken);
        return mockTrustBasketRepository.getOne(basketId);
    }

    /**
     * Returns null for aeroflot tokenization baskets.
     */
    private TPaymentTestContext getPaymentTestContext(MockTrustBasket basket) {
        return basket.getTestContext();
    }

    @TransactionMandatory
    public Collection<UUID> fetchBasketsWaitingUserAction(Set<UUID> lockedTaskKeys, int maxResultSize) {
        Set<UUID> excludes = lockedTaskKeys.isEmpty() ? MockTrustBasketRepository.NO_EXCLUDE_IDS : lockedTaskKeys;
        return mockTrustBasketRepository.findIdsReadyUserAction(
                Instant.now(clock), PaymentStatusEnum.STARTED,
                excludes, PageRequest.of(0, maxResultSize)
        );
    }

    @TransactionMandatory
    public Long countBasketsWaitingUserAction(Set<UUID> lockedTaskKeys) {
        Set<UUID> excludes = lockedTaskKeys.isEmpty() ? MockTrustBasketRepository.NO_EXCLUDE_IDS : lockedTaskKeys;
        return mockTrustBasketRepository.countBasketsReadyUserAction(
                Instant.now(clock), PaymentStatusEnum.STARTED, excludes
        );
    }

    @TransactionMandatory
    public void performUserAction(UUID key) {
        MockTrustBasket mockTrustBasket = mockTrustBasketRepository.getOne(key);
        TPaymentTestContext pctx = getPaymentTestContext(mockTrustBasket);

        if (mockTrustBasket.getType() != null && mockTrustBasket.getType() == MockTrustBasketType.CARD_TOKENIZATION) {

            if (mockTrustBasket.getTokenizationOutcome() == EAviaTokenizationOutcome.TO_FAILURE
                    || (pctx != null && pctx.getPaymentOutcome() == EPaymentOutcome.PO_FAILURE)) {
                paymentNotAuthorized(mockTrustBasket.getId().toString());
            } else {
                paymentAuthorized(mockTrustBasket.getId().toString());
            }

        } else {
            if (pctx != null && pctx.getPaymentOutcome() == EPaymentOutcome.PO_FAILURE) {
                paymentNotAuthorized(mockTrustBasket.getId().toString());
            } else {
                paymentAuthorized(mockTrustBasket.getId().toString());
            }
        }

        mockTrustBasket.setSendCallback(true);
    }

    @TransactionMandatory
    public Collection<UUID> fetchBasketsWaitingCallback(Set<UUID> lockedTaskKeys, int maxResultSize) {
        Set<UUID> excludes = lockedTaskKeys.isEmpty() ? MockTrustBasketRepository.NO_EXCLUDE_IDS : lockedTaskKeys;
        return mockTrustBasketRepository.findIdsReadyForCallback(
                Set.of(PaymentStatusEnum.AUTHORIZED, PaymentStatusEnum.NOT_AUTHORIZED),
                excludes, PageRequest.of(0, maxResultSize)
        );
    }

    @TransactionMandatory
    public Long countBasketsWaitingForCallback(Set<UUID> lockedTaskKeys) {
        Set<UUID> excludes = lockedTaskKeys.isEmpty() ? MockTrustBasketRepository.NO_EXCLUDE_IDS : lockedTaskKeys;
        return mockTrustBasketRepository.countBasketsReadyForCallback(
                Set.of(PaymentStatusEnum.AUTHORIZED, PaymentStatusEnum.NOT_AUTHORIZED), excludes
        );
    }

    @TransactionMandatory
    public void performCallbackEmulation(UUID key) {
        MockTrustBasket mockTrustBasket = mockTrustBasketRepository.getOne(key);
        TPaymentTestContext pctx = getPaymentTestContext(mockTrustBasket);

        String status = "success";
        if (mockTrustBasket.getState() == PaymentStatusEnum.NOT_AUTHORIZED) {
            status = "cancelled";
        }


        final String finalStatus = status;
        trustInvoiceRepository.findByPurchaseTokenEquals(mockTrustBasket.getId().toString()).ifPresentOrElse(invoice -> {
                    // we're only scheduling event for invoice in case we didn't get it via polling
                    if (invoice.getInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT && invoice.isBackgroundJobActive()) {
                        try (var ignored = NestedMdc.forEntity(invoice)) {
                            // stop background polling, and get rid of race when we got status change via
                            // polling
                            invoice.setBackgroundJobActive(false);
                            invoice.setNextCheckStatusAt(null);
                            workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                                    TTrustInvoiceCallbackReceived.newBuilder()
                                            .setStatus(finalStatus)
                                            .build());
                        }
                    }
                },
                () -> log.error("Couldn't find invoice corresponding to mock trust basket {} with appropriate state. " +
                        "Skipping callback", key));

        mockTrustBasket.setCallbackSentAt(Instant.now(clock));
    }
}


