package ru.yandex.travel.orders.services.mock;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.validation.constraints.NotNull;

import com.google.common.base.Preconditions;
import com.google.common.io.Resources;
import org.apache.commons.lang3.NotImplementedException;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsurancePricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundPricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainOfficeAction;
import ru.yandex.travel.orders.commons.proto.TTrainOfficeRefundAction;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.mock.MockTrainOrder;
import ru.yandex.travel.orders.entities.mock.MockTrainOrderItem;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderItemRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderRepository;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.train.MockImGenericOfficeRefundStartService;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.ImClientValidationException;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationRequest;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeResponse;
import ru.yandex.travel.train.partners.im.model.PendingElectronicRegistration;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.RailwayBlankInfo;
import ru.yandex.travel.train.partners.im.model.RailwayConfirmResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReservationRequest;
import ru.yandex.travel.train.partners.im.model.RailwayReservationResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.ReservationConfirmResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountRequest;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.UpdateBlanksResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListRequest;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListResponse;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

public class MockImClient implements ImClient {
    public static final String INVALID_EMAIL = "mock-im@invalid.email";

    private final MockTrainOrderRepository mockTrainOrderRepository;
    private final MockTrainOrderItemRepository mockTrainOrderItemRepository;
    private final TransactionTemplate transactionTemplate;
    private final SingleOperationService singleOperationService;
    private final MockImResponseGenerator mockImResponseGenerator;

    static private class TrainOrderData {
        public UUID orderId;
        public List<TrainOrderItemData> items;
    }

    static private class TrainOrderItemData {
        public TTrainTestContext testContext;
        public TrainReservation reservation;

        void copyReservation(TrainReservation reservation) {
            this.reservation = ProtoUtils.fromTJson(ProtoUtils.toTJson(reservation), TrainReservation.class);
        }
    }

    public static Object getTrainOrderData(TrainOrderItem orderItem, List<TrainOrderItem> slaves) {
        var res = new TrainOrderData();
        res.orderId = orderItem.getOrder().getId();
        res.items = Stream.concat(
                        Stream.of(orderItem),
                        slaves.stream()
                )
                .filter(i -> i.getTestContext() instanceof TTrainTestContext)
                .map(i -> {
                    var data = new TrainOrderItemData();
                    data.testContext = (TTrainTestContext) i.getTestContext();
                    data.copyReservation(i.getReservation());
                    return data;
                }).collect(Collectors.toList());
        return res;
    }

    public MockImClient(MockTrainOrderRepository mockTrainOrderRepository,
                        MockTrainOrderItemRepository mockTrainOrderItemRepository,
                        TransactionTemplate transactionTemplate,
                        SingleOperationService singleOperationService,
                        MockImResponseGenerator mockImResponseGenerator) {
        this.mockTrainOrderRepository = mockTrainOrderRepository;
        this.mockTrainOrderItemRepository = mockTrainOrderItemRepository;
        this.transactionTemplate = transactionTemplate;
        this.singleOperationService = singleOperationService;
        this.mockImResponseGenerator = mockImResponseGenerator;
    }

    @Override
    public AutoReturnResponse autoReturn(AutoReturnRequest request) {
        var rsp = withTx(request.getServiceAutoReturnRequest().getOrderItemId(),
                orderItemId -> {
                    MockTrainOrderItem orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
                    List<Integer> autoReturnBlankIds = request.getServiceAutoReturnRequest().getOrderItemBlankIds();
                    Map<Integer, Optional<BigDecimal>> blankIdsWithAmounts;
                    if (autoReturnBlankIds == null || autoReturnBlankIds.isEmpty()) {
                        blankIdsWithAmounts = orderItem.getItemData().getItemInfo().getOrderItemBlanks().stream()
                                .collect(Collectors.toMap(OrderItemBlank::getOrderItemBlankId,
                                        blankId -> Optional.empty()));
                    } else {
                        blankIdsWithAmounts = autoReturnBlankIds.stream()
                                .collect(Collectors.toMap(Function.identity(), blankId -> Optional.empty()));
                    }

                    MockTrainOrderItem refundItem = new MockTrainOrderItem();
                    refundItem.setId(mockTrainOrderRepository.getNextOrderItemId());
                    refundItem.setType(MockTrainOrderItem.MockTrainOrderItemType.REFUND);
                    refundItem.setOrder(orderItem.getOrder());
                    refundItem.setTestContext(orderItem.getTestContext());
                    boolean success =
                            orderItem.getTestContext().getRefundCheckoutOutcome() == ETrainRefundCheckoutOutcome.RCO_SUCCESS || (
                            orderItem.getTestContext().getRefundCheckoutOutcome() == ETrainRefundCheckoutOutcome.RCO_ODD_REFUND_FAILURE &&
                                    orderItem.getOrder().getItems().stream().filter(x -> x.getType() == MockTrainOrderItem.MockTrainOrderItemType.REFUND)
                                            .count() % 2 == 0
                    );
                    refundItem.setItemData(new MockTrainOrderItem.MockTrainOrderItemData(
                            mockImResponseGenerator.createRefundRailwayItemResponse(refundItem.getId(),
                                    orderItem.getItemData().getItemInfo(),
                                    request.getServiceAutoReturnRequest().getAgentReferenceId(),
                                    false, blankIdsWithAmounts,
                                    success)));
                    mockTrainOrderItemRepository.save(refundItem);
                    if (success) {
                        for (OrderItemBlank b : orderItem.getItemData().getItemInfo().getOrderItemBlanks()) {
                            if (blankIdsWithAmounts.containsKey(b.getOrderItemBlankId())) {
                                b.setBlankStatus(ImBlankStatus.REFUNDED);
                            }
                        }

                        AutoReturnResponse response = new AutoReturnResponse();
                        RailwayAutoReturnResponse serviceReturnResponse = new RailwayAutoReturnResponse();
                        serviceReturnResponse.setBlanks(blankIdsWithAmounts.keySet().stream()
                                .map(mockImResponseGenerator::createRailwayReturnBlankResponse)
                                .collect(Collectors.toList()));
                        response.setServiceReturnResponse(serviceReturnResponse);
                        return response;
                    } else {
                        return null;
                    }
                });
        if (rsp == null) {
            throw new ImClientException(3, "Mocked fail scenario");
        }
        return rsp;
    }

    @Override
    public ElectronicRegistrationResponse changeElectronicRegistration(ElectronicRegistrationRequest request) {
        var result = withTx(request.getOrderItemId(), orderItemId -> {
            var orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
            var testContext = orderItem.getTestContext();
            if (testContext.getTrainChangeElectronicRegistrationOutcome() != ETrainChangeElectronicRegistrationOutcome.CERO_SUCCESS) {
                return null;
            }
            var response = new ElectronicRegistrationResponse();
            response.setExpirationElectronicRegistrationDateTime(orderItem.getItemData().getItemInfo().getElectronicRegistrationExpirationDateTime());
            response.setBlanks(new ArrayList<>());
            var blankIds = Set.copyOf(request.getOrderItemBlankIds());
            for (var blank : orderItem.getItemData().getItemInfo().getOrderItemBlanks()) {
                if (!blankIds.contains(blank.getOrderItemBlankId())) {
                    continue;
                }
                ImBlankStatus targetStatus = request.getSet() ? ImBlankStatus.REMOTE_CHECK_IN :
                        ImBlankStatus.NO_REMOTE_CHECK_IN;
                if (blank.getBlankStatus() == targetStatus) {
                    continue;
                }
                blank.setBlankStatus(targetStatus);
                response.getBlanks().add(toRailwayBlankInfo(blank));
            }
            return response;
        });

        if (result == null) {
            throw new ImClientException(3, "Mocked fail scenario");
        }
        if (result.getBlanks().size() == 0) {
            throw new ImClientException(3, "No blanks to change ER status");
        }
        return result;
    }

    private RailwayBlankInfo toRailwayBlankInfo(OrderItemBlank blank) {
        var result = new RailwayBlankInfo();
        result.setOrderItemBlankId(blank.getOrderItemBlankId());
        result.setBlankStatus(blank.getBlankStatus());
        return result;
    }

    @Override
    public ReturnAmountResponse getReturnAmount(ReturnAmountRequest request) {
        ReturnAmountResponse rsp = withTx(request.getServiceReturnAmountRequest().getOrderItemId(),
                orderItemId -> {
                    MockTrainOrderItem orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
                    if (orderItem.getTestContext().getRefundPricingOutcome() == ETrainRefundPricingOutcome.RPO_SUCCESS) {
                        ReturnAmountResponse response = new ReturnAmountResponse();
                        RailwayReturnAmountResponse serviceResponse = new RailwayReturnAmountResponse();
                        serviceResponse.setBlanks(orderItem.getItemData().getItemInfo().getOrderItemBlanks().stream()
                                .map(blank -> mockImResponseGenerator.createRailwayReturnBlankResponse(blank.getOrderItemBlankId()))
                                .collect(Collectors.toList()));
                        response.setServiceReturnResponse(serviceResponse);
                        return response;
                    }
                    return null;
                });
        if (rsp == null) {
            throw new ImClientException(3, "Mocked fail scenario");
        }
        return rsp;
    }

    @Override
    public InsuranceCheckoutResponse insuranceCheckout(InsuranceCheckoutRequest request) {
        var result = withTx(request.getMainServiceReference().getOrderItemId(), orderItemId -> {
            var orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
            if (orderItem.getTestContext().getInsuranceCheckoutOutcome() == ETrainInsuranceCheckoutOutcome.ICO_SUCCESS) {
                MockTrainOrderItem insurance = new MockTrainOrderItem();
                insurance.setId(mockTrainOrderRepository.getNextOrderItemId());
                insurance.setType(MockTrainOrderItem.MockTrainOrderItemType.INSURANCE);
                insurance.setOrder(orderItem.getOrder());
                insurance.setTestContext(orderItem.getTestContext());
                insurance.setItemData(new MockTrainOrderItem.MockTrainOrderItemData(
                        mockImResponseGenerator.createBuyInsuranceItemResponse(insurance.getId())));
                mockTrainOrderItemRepository.save(insurance);

                var response = new InsuranceCheckoutResponse();
                response.setAmount(insurance.getItemData().getItemInfo().getAmount());
                response.setOrderCustomerId(request.getMainServiceReference().getOrderCustomerId());
                response.setOrderItemId(insurance.getId());
                return response;
            } else {
                return null;
            }
        });

        if (result != null) {
            return result;
        } else {
            throw new ImClientException(3, "Mocked fail scenario");
        }
    }

    @Override
    public InsurancePricingResponse insurancePricing(InsurancePricingRequest request) {
        var result = withTx(request.getProduct().getMainServiceReference().getOrderItemId(), orderItemId -> {
            var orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
            if (orderItem.getTestContext().getInsurancePricingOutcome() == ETrainInsurancePricingOutcome.IPO_SUCCESS) {
                return mockImResponseGenerator.createInsurancePricingResponse(orderItem.getOrder().getData()
                        .getReservationCreateResponse().getCustomers());
            } else {
                return null;
            }
        });

        if (result != null) {
            return result;
        } else {
            throw new ImClientException(3, "Mocked fail scenario");
        }
    }

    @Override
    public InsuranceReturnResponse insuranceReturn(InsuranceReturnRequest request) {
        return withTx(request.getOrderItemId(), orderItemId -> {
            MockTrainOrderItem orderItem = mockTrainOrderItemRepository.getOne(orderItemId);
            MockTrainOrderItem insuranceRefund = new MockTrainOrderItem();
            insuranceRefund.setId(mockTrainOrderRepository.getNextOrderItemId());
            insuranceRefund.setType(MockTrainOrderItem.MockTrainOrderItemType.INSURANCE_REFUND);
            insuranceRefund.setOrder(orderItem.getOrder());
            insuranceRefund.setTestContext(orderItem.getTestContext());
            insuranceRefund.setItemData(new MockTrainOrderItem.MockTrainOrderItemData(mockImResponseGenerator
                    .createRefundInsuranceItemResponse(insuranceRefund.getId(),
                            orderItem.getItemData().getItemInfo())));
            mockTrainOrderItemRepository.save(insuranceRefund);
            InsuranceReturnResponse response = new InsuranceReturnResponse();
            response.setAmount(orderItem.getItemData().getItemInfo().getAmount());
            response.setOrderId(orderItem.getOrder().getId());
            response.setOrderItemId(orderItemId);
            return response;
        });
    }

    @Override
    public OrderListResponse orderList(OrderListRequest request) {
        throw new java.lang.UnsupportedOperationException();
    }

    @Override
    public byte[] orderReservationBlank(OrderReservationBlankRequest request, Duration timeout) {
        return getStaticBlankPdf();
    }

    public static byte[] getStaticBlankPdf() {
        try (InputStream is = Resources.getResource("mock/blank.pdf").openStream()) {
            return is.readAllBytes();
        } catch (IOException e) {
            throw new RuntimeException("Error getting blank from resource", e);
        }
    }

    @Override
    public void reservationCancel(int orderId) {
        withTx(orderId, (id) -> {
            var order = mockTrainOrderRepository.getOne(id);
            if (order.getData().isConfirmed()) {
                throw new ImClientException(3, "Can not cancel confirmed order");
            }
            order.getItems().forEach(x -> {
                x.getItemData().getItemInfo().setSimpleOperationStatus(ImOperationStatus.FAILED);
                x.getItemData().getItemInfo().getOrderItemBlanks()
                        .forEach(b -> b.setBlankStatus(ImBlankStatus.CANCELLED));
            });
            order.getData().setCancelled(true);
            mockTrainOrderRepository.save(order);
            return null;
        });
    }

    @Override
    public ReservationConfirmResponse reservationConfirm(int orderId) {
        var response = withTx(orderId, (id) -> {
            var order = mockTrainOrderRepository.getOne(id);
            if (order.getData().isCancelled()) {
                throw new ImClientException(3, "Can not confirm cancelled order");
            }
            LocalDateTime electroRegistrationExpiration = LocalDateTime.now().plusDays(1);
            LocalDateTime onlineTicketReturnExpiration = LocalDateTime.now().plusDays(2);
            order.getData().setConfirmed(true);
            var resp = new ReservationConfirmResponse();
            resp.setConfirmResults(new ArrayList<>());
            for (var item : order.getItems()) {
                OrderItemResponse itemInfo = item.getItemData().getItemInfo();
                if (item.getTestContext().getReservationConfirmOutcome() == ETrainReservationConfirmOutcome.RCOO_SUCCESS) {
                    itemInfo.setSimpleOperationStatus(ImOperationStatus.OK);
                    itemInfo.setElectronicRegistrationExpirationDateTime(electroRegistrationExpiration);
                    itemInfo.getOrderItemBlanks().forEach(blank -> {
                        blank.setBlankStatus(ImBlankStatus.REMOTE_CHECK_IN);
                        blank.setPendingElectronicRegistration(PendingElectronicRegistration.NO_VALUE);
                        blank.setElectronicRegistrationExpirationDateTime(electroRegistrationExpiration);
                        blank.setOnlineTicketReturnExpirationDateTime(onlineTicketReturnExpiration);
                    });

                    if (item.getType() == MockTrainOrderItem.MockTrainOrderItemType.PURCHASE) {
                        var railwayResp = new RailwayConfirmResponse();
                        railwayResp.setOrderItemId(item.getId());
                        resp.getConfirmResults().add(railwayResp);

                        for (TTrainOfficeRefundAction officeRefundCase :
                                item.getTestContext().getOfficeActions().getRefundsList()) {
                            Preconditions.checkArgument(officeRefundCase.getDelayInSeconds() > 0,
                                    "office refund action should have positive delay");
                            Instant officeRefundAt = Instant.now().plus(officeRefundCase.getDelayInSeconds(),
                                    ChronoUnit.SECONDS);
                            Optional<BigDecimal> officeRefundAmount = officeRefundCase.getAmount() != 0 ?
                                    Optional.of(BigDecimal.valueOf(officeRefundCase.getAmount())) : Optional.empty();
                            item.getItemData().setOfficeRefundAt(officeRefundAt);
                            var opData = new MockImGenericOfficeRefundStartService.StartOfficeRefundData();
                            opData.setOrderId(order.getData().getOrderId());
                            opData.setMockImOrderId(order.getId());
                            var opDataService = new MockImGenericOfficeRefundStartService.Service();
                            opDataService.setMockImOrderItemId(item.getId());
                            opDataService.setBlankIdsWithAmounts(
                                    itemInfo.getOrderItemBlanks().stream()
                                            .collect(Collectors.toMap(
                                                    OrderItemBlank::getOrderItemBlankId,
                                                    imBlank -> officeRefundAmount
                                            ))
                            );
                            opData.setServices(List.of(opDataService));
                            singleOperationService.scheduleOperation(
                                    "MockImOfficeRefund" + Instant.now().toEpochMilli(),
                                    OperationTypes.MOCK_IM_START_OFFICE_REFUND.getValue(),
                                    opData,
                                    officeRefundAt
                            );
                        }
                        TTrainOfficeAction officeAcquireCase = item.getTestContext().getOfficeActions().getAcquire();
                        if (officeAcquireCase.getDelayInSeconds() > 0) {
                            item.getItemData().setOfficeAcquireAt(Instant.now().plus(officeAcquireCase.getDelayInSeconds(),
                                    ChronoUnit.SECONDS));
                        }
                        var timeoutAfter = item.getTestContext().getAlwaysTimeoutAfterConfirmDelayInSeconds();
                        if (timeoutAfter > 0) {
                            order.getData().setAlwaysTimeoutAt(Instant.now().plus(timeoutAfter, ChronoUnit.SECONDS));
                        }
                    } else if (item.getType() == MockTrainOrderItem.MockTrainOrderItemType.INSURANCE) {
                        if (item.getTestContext().getInsuranceCheckoutConfirmOutcome() == ETrainInsuranceCheckoutConfirmOutcome.ICCO_FAILURE) {
                            item.getItemData().getItemInfo().setSimpleOperationStatus(ImOperationStatus.FAILED);
                        }
                    }
                } else if (item.getTestContext().getReservationConfirmOutcome() == ETrainReservationConfirmOutcome.RCOO_FAILURE) {
                    itemInfo.setSimpleOperationStatus(ImOperationStatus.FAILED);
                    itemInfo.getOrderItemBlanks().forEach(b -> b.setBlankStatus(ImBlankStatus.CANCELLED));
                } else {
                    itemInfo.setSimpleOperationStatus(ImOperationStatus.IN_PROCESS);
                }
            }
            if (resp.getConfirmResults().size() == 0) {
                return null;
            }
            return resp;
        });

        if (response != null) {
            return response;
        } else {
            throw new ImClientException(3, "Mocked fail scenario");
        }
    }

    @Override
    public ReservationCreateResponse reservationCreate(ReservationCreateRequest request, @NotNull Object data) {
        var trainOrderData = (TrainOrderData) data;
        ETrainReservationCreateOutcome outcome = trainOrderData.items.get(0).testContext.getReservationCreateOutcome();
        if (outcome == ETrainReservationCreateOutcome.RCRO_SUCCESS) {
            if (request.getReservationItems().stream().flatMap(i -> i.getPassengers().stream())
                    .anyMatch(p -> p.getContactEmailOrPhone() != null && INVALID_EMAIL.equalsIgnoreCase(p.getContactEmailOrPhone()))) {
                throw new ImClientValidationException(1386, "Invalid passenger's email");
            }
            return withTx(trainOrderData, (orderData) -> {
                var orderId = mockTrainOrderRepository.getNextOrderId();
                var mockOrder = new MockTrainOrder();
                mockOrder.setId(orderId);
                mockOrder.getData().setOrderId(orderData.orderId);
                mockOrder = mockTrainOrderRepository.save(mockOrder);
                var response = new ReservationCreateResponse();
                response.setOrderId(orderId);
                response.setCustomers(mockImResponseGenerator.createReservationCustomers(request.getCustomers()));
                response.setReservationResults(new ArrayList<>());
                response.setAmount(BigDecimal.ZERO);
                for (RailwayReservationRequest reserveRequestItem : request.getReservationItems()) {
                    TrainOrderItemData orderItemData = trainOrderData.items.stream()
                            .filter(x -> x.reservation.getPassengers().stream().map(TrainPassenger::getPartnerItemIndex)
                                    .collect(Collectors.toSet()).contains(reserveRequestItem.getIndex()))
                            .findFirst().orElseThrow();
                    TTrainTestContext testContext = orderItemData.testContext;
                    TrainReservation reservation = orderItemData.reservation;
                    Integer orderItemId = mockTrainOrderRepository.getNextOrderItemId();
                    RailwayReservationResponse reservationResponse = mockImResponseGenerator.createReservationResponse(
                            reserveRequestItem, reservation, orderItemId, response.getCustomers());
                    response.getReservationResults().add(reservationResponse);

                    OrderItemResponse buyRailwayItemResponse =
                            mockImResponseGenerator.toOrderItemResponse(reservationResponse);
                    MockTrainOrderItem orderItem = new MockTrainOrderItem();
                    orderItem.setId(orderItemId);
                    orderItem.setTestContext(testContext);
                    orderItem.setItemData(new MockTrainOrderItem.MockTrainOrderItemData(buyRailwayItemResponse));
                    orderItem.setType(MockTrainOrderItem.MockTrainOrderItemType.PURCHASE);
                    orderItem.setOrder(mockOrder);
                    mockTrainOrderItemRepository.save(orderItem);
                    response.setAmount(response.getAmount().add(reservationResponse.getAmount()));
                    response.setConfirmTill(reservationResponse.getConfirmTill());
                }
                response.setReservationNumber(null);
                mockOrder.getData().setReservationCreateResponse(response);
                return response;
            });
        } else if (outcome == ETrainReservationCreateOutcome.RCRO_INVALID) {
            return new ReservationCreateResponse();
        } else {
            throw new ImClientException(3, "Mocked fail scenario");
        }
    }

    @Override
    public UpdateBlanksResponse updateBlanks(int orderItemId, Duration timeout) {
        UpdateBlanksResponse result = withTx(orderItemId, (itemId) -> {
            MockTrainOrderItem item = mockTrainOrderItemRepository.getOne(itemId);
            var now = Instant.now();
            var order = item.getOrder();
            if (order.getData().getAlwaysTimeoutAt() != null &&
                    Instant.now().isAfter(order.getData().getAlwaysTimeoutAt())) {
                return null;
            }
            UpdateBlanksResponse response = new UpdateBlanksResponse();
            response.setBlanks(new ArrayList<>());
            Instant officeAcquireAt = item.getItemData().getOfficeAcquireAt();
            if (officeAcquireAt != null && officeAcquireAt.isBefore(now)) {
                for (OrderItemBlank blank : item.getItemData().getItemInfo().getOrderItemBlanks()) {
                    if (blank.getBlankStatus() != ImBlankStatus.STRICT_BOARDING_PASS) {
                        blank.setBlankStatus(ImBlankStatus.STRICT_BOARDING_PASS);
                        response.setModified(true);
                    }
                }
            }
            return response;
        });
        if (result == null) {
            throw new ImClientIOException("Generated timeout due to TestContext");
        }
        return result;
    }

    @Override
    public CompletableFuture<byte[]> orderReservationBlankAsync(OrderReservationBlankRequest request,
                                                                Duration timeout) {
        return CompletableFuture.completedFuture(orderReservationBlank(request, timeout));
    }

    @Override
    public CompletableFuture<OrderInfoResponse> orderInfoAsync(int orderId, Duration timeout) {
        OrderInfoResponse orderInfoResponse = createOrderInfoResponse(orderId);
        return CompletableFuture.completedFuture(orderInfoResponse);
    }

    @Override
    public CompletableFuture<OrderReservationTicketBarcodeResponse> orderReservationTicketBarcodeAsync(OrderReservationTicketBarcodeRequest request,
                                                                                                       Duration timeout) {
        throw new NotImplementedException("orderReservationTicketBarcodeAsync not implemented");
    }

    public OrderInfoResponse createOrderInfoResponse(int orderId) {
        return withTx(orderId, (id) -> {
            var order = mockTrainOrderRepository.getOne(id);
            List<OrderItemResponse> orderItems = order.getItems().stream()
                    .map(x -> x.getItemData().getItemInfo()).collect(Collectors.toList());
            OrderInfoResponse resp = new OrderInfoResponse();
            resp.setOrderItems(orderItems);
            return resp;
        });
    }

    private <ReqT, RspT> RspT withTx(ReqT request, Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(request));
    }
}
