package ru.yandex.travel.orders.services.mock;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.repository.mock.MockTrainOrderRepository;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.ReservationPlaceType;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.OrderCreateReservationCustomerResponse;
import ru.yandex.travel.train.partners.im.model.OrderFullCustomerRequest;
import ru.yandex.travel.train.partners.im.model.PendingElectronicRegistration;
import ru.yandex.travel.train.partners.im.model.PlaceWithType;
import ru.yandex.travel.train.partners.im.model.RailwayFareInfo;
import ru.yandex.travel.train.partners.im.model.RailwayPassengerRequest;
import ru.yandex.travel.train.partners.im.model.RailwayPassengerResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReservationBlankResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReservationRequest;
import ru.yandex.travel.train.partners.im.model.RailwayReservationResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.RateValue;
import ru.yandex.travel.train.partners.im.model.TariffInfo;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingResult;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.BoardingSystemType;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationType;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOrderItemType;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemCustomer;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;


@Service
@RequiredArgsConstructor
public class MockImResponseGenerator {
    private static final int placeStartNumber = 10;
    private static final ReservationPlaceType placeType = ReservationPlaceType.NEAR_TABLE_FORWARD;
    private static final String carrierInn = "7705445700";

    private static final BigDecimal tariffAmount = BigDecimal.valueOf(3000.0);
    private static final BigDecimal tariffVatAmount = BigDecimal.ZERO;
    private static final Double tariffVatRate = 0.0;
    private static final BigDecimal serviceAmount = BigDecimal.valueOf(156.0);
    private static final BigDecimal serviceVatAmount = BigDecimal.valueOf(26.0);
    private static final Double serviceVatRate = 20.0;
    private static final BoardingSystemType boardingSystemType = BoardingSystemType.ELECTRONIC_TICKET_CONTROL;
    private static final BigDecimal insuranceAmount = BigDecimal.valueOf(100);

    private static final int tripDuration = 60 * 24;  // TODO: calculate from input data

    // TODO: get from input data
    private static final String carDescription = "";
    private static final String carrierCode = "42";
    private static final String countryCode = "RU";
    private static final int destinationTimeZoneDifference = 0;
    private static final int originTimeZoneDifference = 0;
    private static final boolean onlyFullReturnPossible = false;
    private static final String timeDescription = "ПРЕДВАРИТЕЛЬНЫЙ ДОСМОТР НА ВОКЗАЛЕ.";
    private static final String trainDescription = "СКОРЫЙ";

    private static final Map<String, String> MAP_TARIFF_CODE_TO_IM_RESPONSE_CODE = Map.of(
            "full", "Full",
            "senior", "Senior",
            "junior", "Junior",
            "pupil", "Pupil",
            "baby", "FreeChild",
            "child", "Child",
            "child_17", "Child17"
    );
    private static final String KARTA_TARIFF_CARD_TYPE = "UniversalRzhdCard";
    private static final String KARTA_TARIFF_CARD_NUMBER = "9001900090068";
    private static final String KARTA_TARIFF_TYPE = "UniversalCard";
    private static final String KARTA_TARIFF_NAME = "КАРТА";

    private final MockTrainOrderRepository mockTrainOrderRepository;

    public OrderItemResponse createBuyInsuranceItemResponse(int orderItemId) {
        OrderItemResponse item = new OrderItemResponse();
        item.setOperationType(ImOperationType.BUY);
        item.setType(ImOrderItemType.INSURANCE);
        item.setSimpleOperationStatus(ImOperationStatus.OK);
        item.setOrderItemId(orderItemId);
        OrderItemBlank blank = new OrderItemBlank();
        blank.setOrderItemBlankId(mockTrainOrderRepository.getNextBlankId());
        blank.setAmount(insuranceAmount);
        item.setAmount(blank.getAmount());
        item.setOrderItemBlanks(List.of(blank));
        return item;
    }

    public OrderItemResponse createRefundInsuranceItemResponse(int orderItemId, OrderItemResponse buyItem) {
        OrderItemResponse item = new OrderItemResponse();
        item.setOperationType(ImOperationType.REFUND);
        item.setType(ImOrderItemType.INSURANCE);
        item.setSimpleOperationStatus(ImOperationStatus.OK);
        item.setOrderItemId(orderItemId);
        item.setPreviousOrderItemId(buyItem.getOrderItemId());
        OrderItemBlank blank = new OrderItemBlank();
        blank.setOrderItemBlankId(mockTrainOrderRepository.getNextBlankId());
        blank.setPreviousOrderItemBlankId(buyItem.getOrderItemBlanks().get(0).getOrderItemBlankId());
        blank.setAmount(buyItem.getOrderItemBlanks().get(0).getAmount());
        item.setAmount(blank.getAmount());
        item.setOrderItemBlanks(List.of(blank));
        return item;
    }

    public OrderItemResponse createRefundRailwayItemResponse(int orderItemId, OrderItemResponse buyItem,
                                                             String agentReferenceId, boolean external,
                                                             Map<Integer, Optional<BigDecimal>> blankIdsWithAmounts,
                                                             boolean successRefund) {
        OrderItemResponse item = new OrderItemResponse();
        item.setOperationType(ImOperationType.REFUND);
        item.setType(ImOrderItemType.RAILWAY);
        item.setSimpleOperationStatus(successRefund ? ImOperationStatus.OK : ImOperationStatus.FAILED);
        item.setOrderItemId(orderItemId);
        item.setPreviousOrderItemId(buyItem.getOrderItemId());
        item.setAgentReferenceId(agentReferenceId);
        item.setIsExternallyLoaded(external);

        List<OrderItemBlank> blanks = new ArrayList<>();
        for (var prevBlank : buyItem.getOrderItemBlanks()) {
            int prevBlankId = prevBlank.getOrderItemBlankId();
            if (!blankIdsWithAmounts.containsKey(prevBlankId)) {
                continue;
            }
            Optional<BigDecimal> amount = blankIdsWithAmounts.get(prevBlankId);
            OrderItemBlank blank = new OrderItemBlank();
            blank.setOrderItemBlankId(mockTrainOrderRepository.getNextBlankId());
            blank.setPreviousOrderItemBlankId(prevBlankId);
            blank.setAmount(amount.orElse(prevBlank.getAmount()));
            blank.setBlankStatus(ImBlankStatus.NO_REMOTE_CHECK_IN);
            blanks.add(blank);
        }
        item.setOrderItemBlanks(blanks);
        item.setAmount(item.getOrderItemBlanks().stream().map(OrderItemBlank::getAmount)
                .reduce(BigDecimal::add).orElse(BigDecimal.ZERO));
        return item;
    }

    public RailwayReservationResponse createReservationResponse(
            RailwayReservationRequest reserveRequestItem, TrainReservation reservation, int orderItemId,
            List<OrderCreateReservationCustomerResponse> customerList) {
        Map<Integer, Integer> adultBlankIds = reserveRequestItem.getPassengers().stream()
                .filter(x -> x.getCategory() != PassengerCategory.BABY)
                .collect(Collectors.toMap(RailwayPassengerRequest::getOrderCustomerIndex,
                        x -> mockTrainOrderRepository.getNextBlankId()));
        Map<Integer, OrderCreateReservationCustomerResponse> customers = customerList.stream()
                .collect(Collectors.toMap(OrderCreateReservationCustomerResponse::getIndex, x -> x));
        Map<String, TrainPassenger> passengersByDocNumber = reservation.getPassengers().stream()
                .collect(Collectors.toMap(TrainPassenger::getDocumentNumber, x -> x));
        List<Integer> babyBlankIds = new ArrayList<>(adultBlankIds.values());
        int placeNumberCounter = this.placeStartNumber;
        var orderAmount = BigDecimal.ZERO;

        var response = new RailwayReservationResponse();
        response.setBlanks(new ArrayList<>());
        response.setPassengers(new ArrayList<>());
        for (RailwayPassengerRequest requestPassenger : reserveRequestItem.getPassengers()) {
            OrderCreateReservationCustomerResponse c = customers.get(requestPassenger.getOrderCustomerIndex());
            TrainPassenger p = passengersByDocNumber.get(c.getDocumentNumber());
            var passenger = new RailwayPassengerResponse();
            response.getPassengers().add(passenger);
            passenger.setCategory(requestPassenger.getCategory());
            passenger.setOrderCustomerId(c.getOrderCustomerId());
            passenger.setPlacesWithType(new ArrayList<>());
            if (requestPassenger.getCategory() != PassengerCategory.BABY) {
                var price = tariffAmount.add(serviceAmount);
                passenger.setAmount(price);
                orderAmount = orderAmount.add(price);
                var place = new PlaceWithType();
                passenger.getPlacesWithType().add(place);
                var placeNumber = p.getRequestedPlaces() != null && p.getRequestedPlaces().size() > 0 ?
                        p.getRequestedPlaces().get(0) : placeNumberCounter++;
                place.setNumber(String.format("%03d", placeNumber));
                place.setType(placeType);
                Integer blankId = adultBlankIds.get(requestPassenger.getOrderCustomerIndex());
                passenger.setOrderItemBlankId(blankId);
                RailwayReservationBlankResponse blank = createReservationBlankResponse(blankId, requestPassenger, p);
                response.getBlanks().add(blank);
            } else {
                Integer blankId = babyBlankIds.remove(0);
                passenger.setOrderItemBlankId(blankId);
                passenger.setAmount(BigDecimal.ZERO);
            }
        }
        var reservationRequestData = reservation.getReservationRequestData();
        response.setAmount(orderAmount);
        response.setCarDescription(carDescription);
        response.setCarNumber(reserveRequestItem.getCarNumber());
        response.setCarrier(reservation.getCarrier());
        response.setCarrierCode(carrierCode);
        response.setCarrierTin(carrierInn);
        response.setCarType(reserveRequestItem.getCarType());
        var now = LocalDateTime.now(Clock.system(ZoneId.of("UTC+3")));
        response.setConfirmTill(now.plusMinutes(15));
        response.setCountryCode(countryCode);
        var origin = reserveRequestItem.getOriginCode();
        response.setOriginStation(origin);
        response.setOriginStationCode(origin);
        response.setOriginTimeZoneDifference(originTimeZoneDifference);
        var destination = reserveRequestItem.getDestinationCode();
        response.setDestinationStation(destination);
        response.setDestinationStationCode(destination);
        response.setDestinationTimeZoneDifference(destinationTimeZoneDifference);
        response.setIndex(reserveRequestItem.getIndex());
        response.setTripDuration(tripDuration);
        var departureTime = toLocalDateTime(reservationRequestData.getDepartureTime(),
                reservation.getStationFromTimezone());
        var arrivalTime = departureTime.plusMinutes(tripDuration);
        if (reservationRequestData.getArrivalTime() != null) {
            arrivalTime = toLocalDateTime(reservationRequestData.getArrivalTime(), reservation.getStationToTimezone());
        }
        response.setDepartureDateTime(departureTime);
        response.setArrivalDateTime(arrivalTime);
        response.setLocalDepartureDateTime(departureTime);
        response.setLocalArrivalDateTime(arrivalTime);
        response.setOnlyFullReturnPossible(onlyFullReturnPossible);
        response.setOrderItemId(orderItemId);
        response.setServiceClass(reserveRequestItem.getServiceClass());
        response.setSuburban(false);
        response.setTimeDescription(timeDescription);
        response.setTrainDescription(trainDescription);
        response.setTrainNumber(reserveRequestItem.getTrainNumber());
        return response;
    }

    private static LocalDateTime toLocalDateTime(Instant dateTime, String zone) {
        if (dateTime == null) {
            return null;
        }
        ZoneId zoneId = ZoneId.of(zone);
        return LocalDateTime.ofInstant(dateTime, zoneId);
    }

    public List<OrderCreateReservationCustomerResponse> createReservationCustomers(
            List<OrderFullCustomerRequest> requestCustomers) {
        var customers = new ArrayList<OrderCreateReservationCustomerResponse>();
        for (OrderFullCustomerRequest requestCustomer : requestCustomers) {
            var customer = new OrderCreateReservationCustomerResponse();
            customer.setBirthday(requestCustomer.getBirthday());
            customer.setCitizenshipCode(requestCustomer.getCitizenshipCode());
            customer.setDocumentNumber(requestCustomer.getDocumentNumber());
            customer.setDocumentType(requestCustomer.getDocumentType());
            customer.setFirstName(requestCustomer.getFirstName());
            customer.setLastName(requestCustomer.getLastName());
            customer.setMiddleName(requestCustomer.getMiddleName());
            customer.setSex(requestCustomer.getSex());
            customer.setIndex(requestCustomer.getIndex());
            customer.setOrderCustomerId(mockTrainOrderRepository.getNextCustomerId());
            customers.add(customer);
        }
        return customers;
    }

    private RailwayReservationBlankResponse createReservationBlankResponse(
            int blankId, RailwayPassengerRequest requestPassenger, TrainPassenger passenger) {
        var blank = new RailwayReservationBlankResponse();
        blank.setOrderItemBlankId(blankId);

        var price = tariffAmount.add(serviceAmount);
        blank.setAmount(price);
        blank.setAdditionalPrice(price);
        blank.setServicePrice(serviceAmount);

        blank.setBaseFare(BigDecimal.ZERO);
        var fareInfo = new RailwayFareInfo();
        fareInfo.setCarrierTin(carrierInn);
        fareInfo.setReservNumber(String.valueOf(77000000000L + blankId));
        blank.setFareInfo(fareInfo);

        boolean isKartaTariff = isKartaTariff(requestPassenger);
        if (isKartaTariff) {
            blank.setTariffType(KARTA_TARIFF_TYPE);
            var tariffInfo = new TariffInfo();
            tariffInfo.setTariffType(KARTA_TARIFF_TYPE);
            tariffInfo.setTariffName(KARTA_TARIFF_NAME);
            tariffInfo.setDisplayName(KARTA_TARIFF_NAME);
            blank.setTariffInfo(tariffInfo);
        } else {
            var tariffCode = passenger.getTariffCodeWithFallback();
            var tariffType = MAP_TARIFF_CODE_TO_IM_RESPONSE_CODE.get(tariffCode);
            blank.setTariffType(tariffType);
            var tariffInfo = new TariffInfo();
            tariffInfo.setTariffType(tariffType);
            blank.setTariffInfo(tariffInfo);
        }

        var tariffVat = new RateValue();
        tariffVat.setRate(tariffVatRate);
        tariffVat.setValue(tariffVatAmount);
        var serviceVat = new RateValue();
        serviceVat.setRate(serviceVatRate);
        serviceVat.setValue(serviceVatAmount);
        blank.setVatRateValues(Arrays.asList(tariffVat, serviceVat));

        return blank;
    }

    public InsurancePricingResponse createInsurancePricingResponse(List<OrderCreateReservationCustomerResponse> customers) {
        var pricingInfoList = new ArrayList<RailwayTravelProductPricingInfo>();
        for (OrderCreateReservationCustomerResponse c : customers) {
            var product = new RailwayTravelProductPricingInfo();
            product.setCompensation(BigDecimal.valueOf(100500));
            product.setAmount(insuranceAmount);
            product.setOrderCustomerId(c.getOrderCustomerId());
            product.setCompany("CCC");
            product.setProductPackage("random");
            product.setProvider("PPP");
            pricingInfoList.add(product);
        }
        var result = new RailwayTravelPricingResult();
        result.setProductPricingInfoList(pricingInfoList);
        var response = new InsurancePricingResponse();
        response.setPricingResult(result);
        return response;
    }

    public RailwayReturnBlankResponse createRailwayReturnBlankResponse(int blankId) {
        RailwayReturnBlankResponse blankResponse = new RailwayReturnBlankResponse();
        blankResponse.setPurchaseOrderItemBlankId(blankId);
        blankResponse.setAmount(tariffAmount.add(serviceAmount));
        return blankResponse;
    }

    private boolean isKartaTariff(RailwayPassengerRequest passenger) {
        if (passenger.getRailwayBonusCards() == null) {
            return false;
        }
        return passenger.getRailwayBonusCards().stream().anyMatch(card ->
                KARTA_TARIFF_CARD_TYPE.equals(card.getCardType()) &&
                        KARTA_TARIFF_CARD_NUMBER.equals(card.getCardNumber()));
    }


    public OrderItemResponse toOrderItemResponse(RailwayReservationResponse reservationResponse) {
        var now = LocalDateTime.now(Clock.system(ZoneId.of("UTC+3")));
        var res = new OrderItemResponse();
        res.setType(ImOrderItemType.RAILWAY);
        res.setOperationType(ImOperationType.BUY);
        res.setSimpleOperationStatus(ImOperationStatus.IN_PROCESS);
        res.setIsExternallyLoaded(false);
        res.setAmount(reservationResponse.getAmount());
        res.setOrderItemId(reservationResponse.getOrderItemId());
        res.setReservationNumber(reservationResponse.getBlanks().get(0).getFareInfo().getReservNumber());
        res.setBoardingSystemType(boardingSystemType);
        res.setMskCreatedAt(now);
        res.setMskReservedTo(reservationResponse.getConfirmTill());
        res.setElectronicRegistrationExpirationDateTime(reservationResponse.getDepartureDateTime().minusHours(2));
        res.setOrderItemBlanks(reservationResponse.getBlanks().stream().map(x -> {
            var b = new OrderItemBlank();
            b.setAmount(x.getAmount());
            b.setOrderItemBlankId(x.getOrderItemBlankId());
            b.setBlankStatus(ImBlankStatus.NO_REMOTE_CHECK_IN);
            b.setElectronicRegistrationExpirationDateTime(res.getElectronicRegistrationExpirationDateTime());
            b.setOnlineTicketReturnExpirationDateTime(res.getElectronicRegistrationExpirationDateTime());
            b.setPendingElectronicRegistration(PendingElectronicRegistration.NO_VALUE);
            return b;
        }).collect(Collectors.toList()));
        res.setOrderItemCustomers(reservationResponse.getPassengers().stream().map(x -> {
            var c = new OrderItemCustomer();
            c.setOrderCustomerId(x.getOrderCustomerId());
            c.setOrderItemBlankId(x.getOrderItemBlankId());
            c.setOrderItemCustomerId(x.getOrderCustomerId() * 10 + x.getOrderItemBlankId()); // no matter
            return c;
        }).collect(Collectors.toList()));
        return res;
    }
}
