package ru.yandex.travel.orders.services.mock;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.orders.configurations.stress.MockTrustProperties;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponseOrder;
import ru.yandex.travel.orders.services.payments.model.TrustClearResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketOrder;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentReceiptResponse;
import ru.yandex.travel.orders.services.payments.model.TrustRefundState;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.services.payments.model.TrustResizeResponse;
import ru.yandex.travel.orders.services.payments.model.TrustResponseStatus;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartRefundResponse;
import ru.yandex.travel.orders.services.payments.model.TrustUnholdResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStartResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupStatusResponse;

@Slf4j
public class MockTrustClient extends AbstractMockClient implements TrustClient {

    public final static BigDecimal AUTO_SUCCESS_SUM = BigDecimal.valueOf(7777L);
    public final static BigDecimal AUTO_FAILURE_SUM = BigDecimal.valueOf(9999L);

    private int orderCounter = 0;
    private int basketCounter = 0;
    private int refundCounter = 0;

    private final Object sync = new Object();

    private final Map<String, InternalPayment> payments;
    private final Map<String, InternalRefund> refunds;

    private final Map<String, List<InternalRefund>> paymentRefunds;

    private final long paymentDelayMinMillis;
    private final long paymentDelayMaxMillis;

    private final boolean scenarioSumsEnabled;

    private final String instanceId;

    private final ScheduledExecutorService scheduledExecutorService;

    /**
     * Use default properties defined in {@link MockTrustProperties}
     */
    public MockTrustClient() {
        this(new MockTrustProperties(),false);
    }

    public MockTrustClient(MockTrustProperties mockTrustProperties, boolean scenarioSumsEnabled) {
        this(mockTrustProperties.getMaxPayments(), mockTrustProperties.getMaxRefunds(),
                mockTrustProperties.getCallDelayMin(), mockTrustProperties.getCallDelayMax(),
                mockTrustProperties.getPaymentDelayMin(), mockTrustProperties.getPaymentDelayMax(),
                scenarioSumsEnabled);
    }

    public MockTrustClient(int maxPayments, int maxRefunds, Duration callDelayMin, Duration callDelayMax,
                           Duration paymentDelayMin, Duration paymentDelayMax, boolean scenarioSumsEnabled) {
        super(callDelayMin.toMillis(), callDelayMax.toMillis());
        Preconditions.checkArgument(paymentDelayMin.toMillis() <= paymentDelayMax.toMillis(),
                "Payment delay min must be less or equal to max");
        payments = new LinkedHashMap<>(maxPayments + 1, 0.75f, true) {
            @Override
            protected boolean removeEldestEntry(Map.Entry<String, InternalPayment> eldest) {
                return maxPayments < size();
            }
        };
        paymentRefunds = new LinkedHashMap<>(maxPayments + 1, 0.75f, true) {
            @Override
            protected boolean removeEldestEntry(Map.Entry<String, List<InternalRefund>> eldest) {
                return maxPayments < size();
            }
        };
        refunds = new LinkedHashMap<>(maxRefunds + 1, 0.75f, true) {
            @Override
            protected boolean removeEldestEntry(Map.Entry<String, InternalRefund> eldest) {
                return maxRefunds < size();
            }
        };
        this.paymentDelayMinMillis = paymentDelayMin.toMillis();
        this.paymentDelayMaxMillis = paymentDelayMax.toMillis();
        this.scenarioSumsEnabled = scenarioSumsEnabled;
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder().setDaemon(true).setNameFormat("MockTrustClientPayer").build()
        );
        this.instanceId = DateTimeFormatter.ofPattern("yyyy_MM_dd___HH_mm_ss").format(LocalDateTime.now());
    }

    @Override
    public TrustCreateOrderResponse createOrder(String productId, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            TrustCreateOrderResponse response = new TrustCreateOrderResponse();
            response.setProductId(productId);
            response.setOrderId("order" + (++orderCounter));
            return response;
        });
    }

    @Override
    public TrustCreateBasketResponse createBasket(TrustCreateBasketRequest request,
                                                  TrustUserInfo userInfo,
                                                  Object testContext) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = new InternalPayment();
            payment.setPurchaseToken("payment___" + instanceId + "___c" + (++basketCounter));
            log.info("Creating a new mock payment: purchase token {}", payment.getPurchaseToken());
            payment.setOrders(new HashMap<>());
            payment.setStatusEnum(PaymentStatusEnum.NOT_STARTED);
            for (TrustCreateBasketOrder order : request.getOrders()) {
                InternalOrder intOrder = new InternalOrder();
                intOrder.setOrderId(order.getOrderId());
                intOrder.setPrice(order.getPrice());
                intOrder.setOriginalPrice(order.getPrice());
                payment.getOrders().put(intOrder.getOrderId(), intOrder);
            }
            payments.put(payment.getPurchaseToken(), payment);

            TrustCreateBasketResponse response = new TrustCreateBasketResponse();
            response.setPurchaseToken(payment.getPurchaseToken());
            return response;

        });
    }

    @Override
    public TrustStartPaymentResponse startPayment(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not" +
                            " found",
                    purchaseToken);
            TrustStartPaymentResponse response = new TrustStartPaymentResponse();
            // TODO (mbobrov): add other params
            response.setPaymentUrl("http://test.payments/payments/?token=" + purchaseToken);
            payment.setStatusEnum(PaymentStatusEnum.STARTED);
            if (scenarioSumsEnabled) {
                if (AUTO_SUCCESS_SUM.compareTo(payment.getTotal()) == 0) {
                    delayOrExecutePaymentAction(() -> paymentAuthorized(purchaseToken));
                } else if (AUTO_FAILURE_SUM.compareTo(payment.getTotal()) == 0) {
                    delayOrExecutePaymentAction(() -> paymentNotAuthorized(purchaseToken));
                }
            }
            return response;
        });
    }

    private void delayOrExecutePaymentAction(Runnable runnable) {

        long sleepTime = paymentDelayMinMillis +
                (long) (Math.random() * (paymentDelayMaxMillis - paymentDelayMinMillis));
        if (sleepTime > 0) {
            scheduledExecutorService.schedule(runnable, sleepTime, TimeUnit.MILLISECONDS);
        } else {
            runnable.run();
        }
    }

    @Override
    public TrustClearResponse clear(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            payment.setStatusEnum(PaymentStatusEnum.CLEARED);
            return new TrustClearResponse();
        });
    }

    @Override
    public TrustUnholdResponse unhold(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            payment.getOrders().forEach((k, v) -> v.setPrice(BigDecimal.ZERO));
            return new TrustUnholdResponse();
        });
    }

    @Override
    public TrustResizeResponse resize(String purchaseToken, String orderId,
                                      TrustResizeRequest resizeRequest, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            InternalOrder order = Preconditions.checkNotNull(payment.getOrders().get(orderId), "Order %s not found in" +
                    " " +
                    "payment", orderId);
            order.setPrice(resizeRequest.getAmount());
            return new TrustResizeResponse();
        });
    }

    @Override
    public TrustCreateRefundResponse createRefund(TrustCreateRefundRequest request,
                                                  TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalPayment payment = Preconditions.checkNotNull(payments.get(request.getPurchaseToken()), "Payment " +
                    "%s not found", request.getPurchaseToken());

            InternalRefund refund = new InternalRefund();
            refund.setPurchaseToken(request.getPurchaseToken());
            refund.setStatus(TrustRefundState.WAIT_FOR_NOTIFICATION);
            refund.setOrderDeltas(request.getOrders().stream().collect(Collectors.toMap(TrustCreateRefundRequest.Order::getOrderId, TrustCreateRefundRequest.Order::getDeltaAmount)));
            refund.setRefundId("refund" + (++refundCounter));
            refunds.put(refund.getRefundId(), refund);
            List<InternalRefund> refs = paymentRefunds.computeIfAbsent(request.getPurchaseToken(),
                    (key) -> new ArrayList<>());
            refs.add(refund);

            TrustCreateRefundResponse response = new TrustCreateRefundResponse();
            response.setTrustRefundId(refund.getRefundId());
            return response;
        });
    }

    @Override
    public TrustStartRefundResponse startRefund(String refundId, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalRefund refund = Preconditions.checkNotNull(refunds.get(refundId), "Refund %s not found", refundId);
            InternalPayment payment = Preconditions.checkNotNull(payments.get(refund.getPurchaseToken()),
                    "Payment %s not found", refund.getPurchaseToken());
            for (Map.Entry<String, BigDecimal> entry : refund.getOrderDeltas().entrySet()) {
                // we don't care about sum integrity here, as the exception will stop the test
                InternalOrder order = Preconditions.checkNotNull(payment.getOrders().get(entry.getKey()),
                        "Order %s not found", entry.getKey());
                order.setPrice(order.getPrice().subtract(entry.getValue()));
            }
            refund.setStatus(TrustRefundState.SUCCESS);
            return new TrustStartRefundResponse();
        });
    }

    @Override
    public TrustRefundStatusResponse getRefundStatus(String refundId, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            InternalRefund refund = Preconditions.checkNotNull(refunds.get(refundId), "Refund %s not found", refundId);
            TrustRefundStatusResponse response = new TrustRefundStatusResponse();
            if (refund.getStatus() == TrustRefundState.SUCCESS) {
                response.setFiscalReceiptUrl("http://refund.fiscal.receipt/" + refundId);
            }
            response.setStatus(refund.getStatus());
            return response;
        });
    }

    @Override
    public TrustBasketStatusResponse getBasketStatus(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            log.info("Getting payment info for purchase token {}", purchaseToken);
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            TrustBasketStatusResponse response = new TrustBasketStatusResponse();
            response.setPaymentStatus(payment.getStatusEnum());
            response.setOrders(
                    payment.getOrders().values().stream().map(internalOrder -> {
                        TrustBasketStatusResponseOrder order = new TrustBasketStatusResponseOrder();
                        order.setPaidAmount(internalOrder.getPrice());
                        order.setOrderId(internalOrder.getOrderId());
                        order.setOrigAmount(internalOrder.getOriginalPrice());
                        return order;
                    }).collect(Collectors.toList()));
            switch (payment.getStatusEnum()) {
                case AUTHORIZED:
                    response.setFiscalReceiptUrl("http://fiscal.receipt/" + payment.getPurchaseToken());
                    response.setPaymentTs(payment.paidAt);
                    break;
                case CLEARED:
                    response.setFiscalReceiptUrl("http://fiscal.receipt/" + payment.getPurchaseToken());
                    response.setFiscalReceiptClearingUrl("http://fiscal.clearing.receipt/" + payment.getPurchaseToken());
                    response.setClearTs(LocalDateTime.now().minusMinutes(2).toInstant(ZoneOffset.UTC));
                    break;
            }
            return response;
        });
    }

    @Override
    public TrustPaymentMethodsResponse getPaymentMethods(TrustUserInfo userInfo) {
        return syncWithRandomSleep(() ->
                TrustPaymentMethodsResponse.builder()
                        .boundPaymentMethods(Collections.emptyList())
                        .build());
    }

    @Override
    public TrustCreateAccountResponse createAccount(TrustCreateAccountRequest request, TrustUserInfo userInfo) {
        throw new UnsupportedOperationException("Mock accounts aren't supported yet");
    }

    @Override
    public TrustCreateTopupResponse createTopup(TrustTopupRequest request, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() ->
                TrustCreateTopupResponse.builder()
                        .status(TrustResponseStatus.SUCCESS)
                        .purchaseToken(UUID.randomUUID().toString())
                        .build());
    }


    @Override
    public TrustTopupStatusResponse getTopupStatus(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            var response = new TrustTopupStatusResponse();
            response.setStatus(TrustResponseStatus.SUCCESS);
            response.setPurchaseToken(UUID.randomUUID().toString());
            return response;
        });
    }

    @Override
    public TrustTopupStartResponse startTopup(String purchaseToken, TrustUserInfo userInfo) {
        return syncWithRandomSleep(() -> {
            var response = new TrustTopupStartResponse();
            response.setStatus(TrustResponseStatus.SUCCESS);
            response.setPurchaseToken(UUID.randomUUID().toString());
            return response;
        });
    }

    @Override
    public TrustPaymentReceiptResponse getReceipt(String purchaseToken, String receiptId, TrustUserInfo userInfo) {
        throw new UnsupportedOperationException("Mock getReceipt is not supported");
    }

    @Override
    public CompletableFuture<TrustPaymentMethodsResponse> getPaymentMethodsAsync(TrustUserInfo userInfo) {
        return CompletableFuture.completedFuture(getPaymentMethods(userInfo));
    }

    public void paymentAuthorized(String purchaseToken) {
        synchronized (sync) {
            log.info("init mock: payment authorized for purchase token {}", purchaseToken);
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            Preconditions.checkState(payment.getStatusEnum() == PaymentStatusEnum.STARTED,
                    "Only payment in STARTED state can be authorized");
            payment.setStatusEnum(PaymentStatusEnum.AUTHORIZED);
            payment.paidAt = Instant.now();
        }
    }

    public void paymentNotAuthorized(String purchaseToken) {
        synchronized (sync) {
            log.info("init mock: payment is not authorized for purchase token {}", purchaseToken);
            InternalPayment payment = Preconditions.checkNotNull(payments.get(purchaseToken), "Payment %s not found",
                    purchaseToken);
            Preconditions.checkState(payment.getStatusEnum() == PaymentStatusEnum.STARTED,
                    "Only payment in STARTED state can be not authorized");
            payment.setStatusEnum(PaymentStatusEnum.NOT_AUTHORIZED);
        }
    }

    private <T> T syncWithRandomSleep(Callable<T> callable) {
        sleepRandomly();
        Stopwatch sw = Stopwatch.createStarted();
        synchronized (sync) {
            Duration lockAcquisition = sw.elapsed();
            if (lockAcquisition.toMillis() > 100) {
                log.warn("long mock lock acquisition: {}", lockAcquisition);
            }
            try {
                return callable.call();
            } catch (Exception e) {
                throw ExceptionUtils.throwException(e);
            } finally {
                Duration duration = sw.elapsed();
                if (duration.toSeconds() >= 1) {
                    log.warn("long mock operation: {}; caller 1: {}; caller 2 {}",
                            duration, getCallingOperationName(0), getCallingOperationName(1));
                }
            }
        }
    }

    // some illegal mambo-jambo for integration tests debug only
    private static String getCallingOperationName(int callerIdx) {
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        // 0th - Thread.getStackTrace
        // 1st - MockTrustClient.getCallingOperationName
        // 2nd - MockTrustClient.syncWithRandomSleep
        // 3rd - MockTrustClient.<caller 0>
        // ...
        String name = stackTraceElements[3 + callerIdx].toString();
        // getting only the last part the output: some.pkg.MockTrustClient.main(MockTrustClient.java:354)
        return name.substring(name.lastIndexOf(".", name.lastIndexOf(".") - 1) + 1);
    }

    @Data
    private static class InternalOrder {
        private String orderId;
        private BigDecimal price;
        private BigDecimal originalPrice;
    }

    @Data
    private static class InternalPayment {
        private PaymentStatusEnum statusEnum;
        private String purchaseToken;
        private Map<String, InternalOrder> orders;
        private Instant paidAt;

        BigDecimal getTotal() {
            return orders.values().stream().map(o -> o.price).reduce(BigDecimal.ZERO, (v, acc) -> acc.add(v));
        }
    }

    @Data
    private static class InternalRefund {
        private TrustRefundState status;
        private String purchaseToken;
        private String refundId;
        private Map<String, BigDecimal> orderDeltas;
    }
}
