package ru.yandex.travel.orders.services.mock.jobs;

import java.time.Clock;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustAccountRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustBasketRepository;
import ru.yandex.travel.orders.repository.mock.MockTrustRefundRepository;
import ru.yandex.travel.orders.services.mock.MockDBTrustClient;
import ru.yandex.travel.orders.services.mock.TrustDBMockProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Configuration
@RequiredArgsConstructor
@ConditionalOnProperty(value = "trust-db-mock.enabled", havingValue = "true")
@EnableConfigurationProperties(TrustDBMockProperties.class)
public class MockTrustDbConfiguration {
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final Environment environment;

    private final Clock clock;

    private final TrustDBMockProperties trustDBMockProperties;

    @Bean
    public MockDBTrustClient mockDBTrustClient(MockTrustBasketRepository mockTrustBasketRepository,
                                               MockTrustRefundRepository mockTrustRefundRepository,
                                               MockTrustAccountRepository mockTrustAccountRepository,
                                               TrustInvoiceRepository trustInvoiceRepository,
                                               WorkflowMessageSender workflowMessageSender) {
        if (environment.acceptsProfiles("prod")) {
            throw new RuntimeException("\n\nATTENTION: MOCK TRUST MUST BE DISABLED FOR PROD ENVIRONMENT\n\n");
        }
        return new MockDBTrustClient(
                mockTrustBasketRepository, mockTrustRefundRepository, mockTrustAccountRepository,
                trustInvoiceRepository,
                workflowMessageSender, clock, trustDBMockProperties
        );
    }

    @Bean
    public TaskProcessor<UUID> mockTrustBasketUserActionProcessor(MockDBTrustClient mockDBTrustClient) {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return mockDBTrustClient.fetchBasketsWaitingUserAction(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return mockDBTrustClient.countBasketsWaitingUserAction(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName(trustDBMockProperties.getUserActionTaskProcessor().getName());
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, mockDBTrustClient::performUserAction,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name(trustDBMockProperties.getUserActionTaskProcessor().getName())
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trustDBMockProperties.getUserActionTaskProcessor().getPoolSize())
                        .initialStartDelay(trustDBMockProperties.getUserActionTaskProcessor().getInitialStartDelay())
                        .scheduleRate(trustDBMockProperties.getUserActionTaskProcessor().getScheduleRate())
                        .build());
    }

    @Bean
    public TaskProcessor<UUID> mockTrustBasketCallbackProcessor(MockDBTrustClient mockDBTrustClient) {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return mockDBTrustClient.fetchBasketsWaitingCallback(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return mockDBTrustClient.countBasketsWaitingForCallback(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName(trustDBMockProperties.getCallbackTaskProcessor().getName());
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, mockDBTrustClient::performCallbackEmulation,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name(trustDBMockProperties.getCallbackTaskProcessor().getName())
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trustDBMockProperties.getCallbackTaskProcessor().getPoolSize())
                        .initialStartDelay(trustDBMockProperties.getCallbackTaskProcessor().getInitialStartDelay())
                        .scheduleRate(trustDBMockProperties.getCallbackTaskProcessor().getScheduleRate())
                        .build());
    }
}
