package ru.yandex.travel.orders.services.notifications;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.orders.entities.notifications.MailSenderAttachment;
import ru.yandex.travel.orders.entities.notifications.MailSenderRetryableException;
import ru.yandex.travel.orders.services.MailSenderConfigurationProperties;

@Service
@Slf4j
@EnableConfigurationProperties(MailSenderConfigurationProperties.class)
public class TemplatedMailSenderServiceImpl implements TemplatedMailSenderService {
    private final MailSenderConfigurationProperties config;
    private final AsyncHttpClientWrapper ahcClient;
    private final ObjectMapper mapper;

    public TemplatedMailSenderServiceImpl(MailSenderConfigurationProperties configurationProperties,
                                          @Qualifier("templatedMailerAhcClientWrapper") AsyncHttpClientWrapper ahcClient) {
        this.config = configurationProperties;
        this.ahcClient = ahcClient;

        this.mapper = new ObjectMapper()
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
                .setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_NULL);
    }

    @Override
    public void sendEmailSync(String targetEmail, String template, JsonNode data, List<MailSenderAttachment> attachments) {
        sync(sendEmailAsync(targetEmail, template, data, attachments));
    }

    @Override
    public CompletableFuture<Response> sendEmailAsync(String targetEmail, String template, JsonNode data,
                                                      List<MailSenderAttachment> attachments) {
        if (config.getDebug().isEnabled() && config.getDebug().getFailEmail().equals(targetEmail)) {
            return CompletableFuture.failedFuture(new RuntimeException("Exception for testing purposes"));
        }
        String url = config.getTemplatedMailerUrlBase();
        try {
            RequestBuilder requestBuilder = new RequestBuilder()
                    .setMethod("POST")
                    .setUrl(url)
                    .addFormParam("to", targetEmail)
                    .addFormParam("type", template)
                    .addFormParam("data", mapper.writeValueAsString(data))
                    .addFormParam("attachments", mapper.writeValueAsString(attachments));
            return ahcClient.executeRequest(requestBuilder);
        } catch (JsonProcessingException e) {
            log.error("Error occurred while serializing email info", e);
            throw new RuntimeException("Error while serializing email info", e);
        }

    }

    private Response sync(CompletableFuture<Response> future) {
        try {
            Response response = future.get();
            int statusCode = response.getStatusCode();
            if (statusCode >= 200 && statusCode < 300) {
                return response;
            } else if (statusCode >= 500) {
                throw new MailSenderRetryableException(
                        String.format("Exception happened while executing http call: %s. Response body: %s",
                                response.getStatusText(), response.getResponseBody()));
            } else {
                throw new RuntimeException(
                        String.format("Exception happened while executing http call: %s. Response body: %s",
                                response.getStatusText(), response.getResponseBody()));
            }
        } catch (InterruptedException e) {
            log.error("Mailer call interrupted", e);
            Thread.currentThread().interrupt(); // preserved interruption status
            throw new MailSenderRetryableException(e);
        } catch (ExecutionException e) {
            if (e.getCause() == null) {
                throw new RuntimeException("No root cause for ExecutionException found", e);
            }
            Throwable cause = e.getCause();
            if (cause instanceof TimeoutException) {
                throw new MailSenderRetryableException(e);
            } else if (cause instanceof IOException) {
                throw new MailSenderRetryableException(e);
            } else if (cause instanceof RuntimeException) {
                throw (RuntimeException) cause;
            } else {
                throw new RuntimeException(e);
            }
        }
    }
}
