package ru.yandex.travel.orders.services.notifier;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.repository.NotifierStateChangeRepository;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorHelper;

@Configuration
@EnableConfigurationProperties(NotifierStateRefreshConfigurationProperties.class)
@RequiredArgsConstructor
public class NotifierStateRefreshConfiguration {

    private final NotifierStateChangeRepository notifierStateChangeRepository;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final NotifierStateRefreshConfigurationProperties notifierStateRefreshConfigurationProperties;

    @Bean
    public NotifierClient notifierClient(NotifierClientFactory notifierClientFactory) {
        if (notifierStateRefreshConfigurationProperties.isMock()) {
            return new MockNotifierClient();
        } else {
            return new NotifierClientImpl(notifierClientFactory);
        }
    }

    @Bean
    public NotifierStateRefresher notifierStateRefresher(NotifierClient notifierClient) {
        return new NotifierStateRefresher(notifierClient, notifierStateChangeRepository);
    }

    @Bean
    @ConditionalOnProperty("notifier-state-refresh.task-processor.enabled")
    public TaskProcessor<UUID> notifierRefreshTaskProcessor(NotifierStateRefresher notifierStateRefresher) {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                Set<UUID> excludeIds = getLockedTaskKeys();
                excludeIds = excludeIds == null || excludeIds.isEmpty() ?
                        NotifierStateChangeRepository.NO_EXCLUDE_IDS : excludeIds;
                List<String> orderIds =
                        notifierStateChangeRepository.getOrdersWaitingStateRefresh(
                                excludeIds,
                                Instant.now().minus(notifierStateRefreshConfigurationProperties.getDebounce()),
                                maxResultSize
                        );
                return orderIds.stream().map(UUID::fromString)
                        .collect(Collectors.toUnmodifiableList());

            }

            @Override
            public long getPendingTasksCount() {
                Set<UUID> excludeIds = getLockedTaskKeys();
                excludeIds = excludeIds == null || excludeIds.isEmpty() ?
                        NotifierStateChangeRepository.NO_EXCLUDE_IDS : excludeIds;
                return notifierStateChangeRepository.countOrdersWaitingStateRefresh(
                        excludeIds,
                        Instant.now().minus(notifierStateRefreshConfigurationProperties.getDebounce()));
            }
        };
        return new TaskProcessor<>(
                taskKeyProvider,
                notifierStateRefresher::refreshState,
                forcedRollbackTxManagerWrapper,
                TaskProcessorHelper.createDefaultTxDefinition("refreshNotifierState"),
                notifierStateRefreshConfigurationProperties.getTaskProcessor()
        );
    }
}
