package ru.yandex.travel.orders.services.notifier;

import java.util.UUID;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.repository.NotifierStateChangeRepository;
import ru.yandex.travel.tx.utils.TransactionMandatory;

@Slf4j
@RequiredArgsConstructor
public class NotifierStateRefresher {

    private final NotifierClient notifierClient;

    private final NotifierStateChangeRepository notifierStateChangeRepository;

    private final Counter successCounter = Counter.builder("notifier_state_refresh").tag("result", "success")
            .register(Metrics.globalRegistry);

    private final Counter errorCounter = Counter.builder("notifier_state_refresh").tag("result", "error")
            .register(Metrics.globalRegistry);

    @TransactionMandatory
    public void refreshState(UUID orderId) {
        Long maxChangeId = null;
        try (NestedMdc ignored = NestedMdc.forEntity(orderId, null)) {
            try {
                maxChangeId = notifierStateChangeRepository.findMaxIdForOrder(orderId);
                log.info("Refreshing notifier state, max change id: {}", maxChangeId);
                notifierClient.orderChanged(orderId);
                successCounter.increment();
            } catch (Exception e) {
                log.error("Error refreshing notifier state", e);
                errorCounter.increment();
                maxChangeId = null;
            } finally {
                if (maxChangeId == null) {
                    // there are no existing changes, the method is called for the first time, nothing to delete yet
                    //noinspection ReturnInsideFinallyBlock
                    return;
                }
                try {
                    log.debug("Removing all notifier state changes before {}", maxChangeId);
                    notifierStateChangeRepository.cleanupByOrderIdAndIdLessThanEqual(orderId, maxChangeId);
                } catch (Exception ex) {
                    log.error("Error deleting notifier state changes", ex);
                }

            }
        }
    }
}
