package ru.yandex.travel.orders.services.orders;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;

@Service
@Slf4j
@RequiredArgsConstructor
public class CheckMoneyRefundsService {
    private final static Set<EOrderState> MONEY_REFUND_ALLOWED_GENERIC_STATES = Set.of(
            EOrderState.OS_REFUNDED,
            EOrderState.OS_CONFIRMED
    );

    private final static Set<MoneyRefundState> ACTIVE_REFUND_STATES = Set.of(
            MoneyRefundState.PENDING,
            MoneyRefundState.IN_PROGRESS);

    private boolean checkTrainManualRefundAllowed(TrainOrder order) {
        boolean hasPendingOrActiveMoneyRefunds = order.getMoneyRefunds().stream()
                .anyMatch(refund -> ACTIVE_REFUND_STATES.contains(refund.getState()));
        if (hasPendingOrActiveMoneyRefunds) {
            return false;
        } else {
            ETrainOrderState state = order.getState();
            switch (state) {
                case OS_MANUAL_PROCESSING:
                case OS_REFUNDED:
                    return true;
                case OS_CONFIRMED:
                    Preconditions.checkState(order.getOrderItems().size() == 1, "Only 1 order item is supported");
                    TrainOrderItem orderItem = (TrainOrderItem) order.getOrderItems().get(0);
                    return Instant.now().isAfter(orderItem.getReservation().getDepartureTime()); // is order done?
                default:
                    return false;
            }
        }
    }

    private boolean checkGenericManualRefundAllowed(GenericOrder order) {
        if (order.isUserActionScheduled()) {
            return false;
        } else {
            return MONEY_REFUND_ALLOWED_GENERIC_STATES.contains(order.getState());
        }
    }

    private boolean checkHotelManualRefundAllowed(HotelOrder order) {
        EHotelOrderState state = order.getState();
        switch (state) {
            case OS_MANUAL_PROCESSING:
            case OS_REFUNDED:
                return true;
            default:
                return false;
        }
    }

    public boolean checkManualRefundAllowed(Order order) {
        switch (order.getPublicType()) {
            case OT_TRAIN:
                return checkTrainManualRefundAllowed((TrainOrder) order);
            case OT_HOTEL_EXPEDIA:
                return checkHotelManualRefundAllowed((HotelOrder) order);
            case OT_GENERIC:
                return checkGenericManualRefundAllowed((GenericOrder) order);
            default:
                return false;
        }
    }

    public boolean checkAllMoneyRefundAllowed(Order order) {
        return checkByFiscalItemManualRefundAllowed(order);
    }

    public boolean checkFeeRefundAllowed(Order order) {
        Invoice invoice = order.getCurrentInvoice();
        if (invoice == null) {
            return false;
        }
        boolean isAnythingToRefund = invoice.getInvoiceItems().stream()
                .anyMatch(item -> item.getFiscalItemType().isYandexFee()
                        && item.getPrice().compareTo(BigDecimal.ZERO) > 0);
        return checkManualRefundAllowed(order) && isAnythingToRefund;
    }

    public boolean checkByFiscalItemManualRefundAllowed(Order order) {
        return checkManualRefundAllowed(order) && checkByFiscalItemRefundAllowed(order);
    }

    public boolean checkByFiscalItemRefundAllowed(Order order) {
        return canRefundInvoice(order.getCurrentInvoice());
    }

    public boolean checkByFiscalItemManualHotelRefundAllowed(Order order) {
        var invoices = order.getInvoices();
        if (invoices.isEmpty()) {
            return false;
        }

        return invoices.stream().anyMatch(this::canRefundInvoice);
    }

    private boolean canRefundInvoice(Invoice invoice) {
        if (invoice == null) {
            return false;
        }

        return invoice.getInvoiceItems().stream().anyMatch(this::canRefundInvoiceItem);
    }

    private boolean canRefundInvoiceItem(InvoiceItem item) {
        return item.getPrice().compareTo(BigDecimal.ZERO) > 0;
    }
}
