package ru.yandex.travel.orders.services.orders;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.EInvoiceRefundType;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;

@Service
@Slf4j
public class GenericOrderMoneyRefundService {
    public void startInvoiceRefund(MoneyRefund refund, StateContext<EOrderState, GenericOrder> ctx) {
        Preconditions.checkArgument(refund.getState() == MoneyRefundState.PENDING,
                "Expected PENDING MoneyRefund but got %s", refund.getState());
        refund.setState(MoneyRefundState.IN_PROGRESS);

        GenericOrder order = ctx.getWorkflowEntity();
        Invoice invoice = order.getCurrentInvoice();
        Preconditions.checkState(invoice != null, "Active invoice must be present in order");
        if (refund.getTargetFiscalItems().size() == 0) {
            log.info("Skip payment refund cause zero items");
            ctx.scheduleEvent(TInvoiceRefunded.newBuilder()
                    .setOrderRefundId(ProtoUtils.toStringOrEmpty(refund.getOrderRefundId()))
                    .setInvoiceId(invoice.getId().toString())
                    .setRefundType(EInvoiceRefundType.EIR_UNKNOWN)
                    .build());
        } else {
            ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), RefundingUtils.createRefundEvent(refund));
        }
    }

    public MoneyRefund getPendingMoneyRefund(GenericOrder order) {
        return order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.PENDING)
                .findFirst()
                .orElse(null);
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    public boolean hasActiveMoneyRefunds(Order order) {
        return order.getMoneyRefunds().stream().anyMatch(r -> r.getState() == MoneyRefundState.IN_PROGRESS ||
                r.getState() == MoneyRefundState.WAITING_CLEARING);
    }
}
