package ru.yandex.travel.orders.services.orders;

import java.time.Clock;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.proto.EGenericOrderAggregateState;
import ru.yandex.travel.orders.proto.EHotelOrderAggregateState;
import ru.yandex.travel.orders.proto.EPaymentErrorDisplayState;
import ru.yandex.travel.orders.proto.ETrainOrderAggregateErrorMessageCode;
import ru.yandex.travel.orders.proto.ETrainOrderAggregateErrorType;
import ru.yandex.travel.orders.proto.ETrainOrderAggregateState;
import ru.yandex.travel.orders.proto.ETrainOrderInsuranceAggregateState;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateErrorInfo;
import ru.yandex.travel.orders.proto.TTrainOrderAggregateStateExtInfo;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.train.model.ErrorCode;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.workflow.EWorkflowState;

@Component
@RequiredArgsConstructor
public class OrderAggregateStateMapper {
    private static final Set<ETrustInvoiceState> PAID_INVOICE_STATES =
            ImmutableSet.of(ETrustInvoiceState.IS_CANCELLED, ETrustInvoiceState.IS_REFUNDED);
    private static final Map<String, EPaymentErrorDisplayState> PAYMENT_ERROR_STR_TO_ENUM =
            ImmutableMap.<String, EPaymentErrorDisplayState>builder()
                    .put("authorization_reject", EPaymentErrorDisplayState.EPE_AUTHORIZATION_REJECT)
                    .put("expired_card", EPaymentErrorDisplayState.EPE_EXPIRED_CARD)
                    .put("fail_3ds", EPaymentErrorDisplayState.EPE_FAIL_3DS)
                    .put("limit_exceeded", EPaymentErrorDisplayState.EPE_LIMIT_EXCEEDED)
                    .put("not_enough_funds", EPaymentErrorDisplayState.EPE_NOT_ENOUGH_FUNDS)
                    .put("transaction_not_permitted", EPaymentErrorDisplayState.EPE_TRANSACTION_NOT_PERMITTED)
                    .put("user_cancelled", EPaymentErrorDisplayState.EPE_USER_CANCELLED)
                    .put("restricted_card", EPaymentErrorDisplayState.EPE_RESTRICTED_CARD)
                    .put("blacklisted", EPaymentErrorDisplayState.EPE_BLACKLISTED)
                    .put("payment_timeout", EPaymentErrorDisplayState.EPE_PAYMENT_TIMEOUT)
                    .build();
    private static final Map<InsuranceStatus, ETrainOrderInsuranceAggregateState> INSURANCE_TO_PROTO_ENUM =
            ImmutableMap.<InsuranceStatus, ETrainOrderInsuranceAggregateState>builder()
                    .put(InsuranceStatus.DISABLED, ETrainOrderInsuranceAggregateState.EIAG_DISABLED)
                    .put(InsuranceStatus.NEW, ETrainOrderInsuranceAggregateState.EIAG_NEW)
                    .put(InsuranceStatus.PRICED, ETrainOrderInsuranceAggregateState.EIAG_PRICED)
                    .put(InsuranceStatus.PRICING_FAILED, ETrainOrderInsuranceAggregateState.EIAG_PRICING_FAILED)
                    .put(InsuranceStatus.CHECKED_OUT, ETrainOrderInsuranceAggregateState.EIAG_CHECKED_OUT)
                    .put(InsuranceStatus.CHECKOUT_FAILED, ETrainOrderInsuranceAggregateState.EIAG_CHECKOUT_FAILED)
                    .put(InsuranceStatus.AUTO_RETURN, ETrainOrderInsuranceAggregateState.EIAG_AUTO_RETURN)
                    .build();
    private static final Map<ErrorCode, ETrainOrderAggregateErrorMessageCode> ERROR_MESSAGE_CODE_TO_PROTO_ENUM =
            ImmutableMap.<ErrorCode, ETrainOrderAggregateErrorMessageCode>builder()
                    .put(ErrorCode.REBOOKING_FAILED, ETrainOrderAggregateErrorMessageCode.EMC_REBOOKING_FAILED)
                    .put(ErrorCode.TRY_LATER, ETrainOrderAggregateErrorMessageCode.EMC_TRY_LATER)
                    .put(ErrorCode.INVALID_BONUS_CARD, ETrainOrderAggregateErrorMessageCode.EMC_INVALID_BONUS_CARD)
                    .put(ErrorCode.TARIFF_ERROR, ETrainOrderAggregateErrorMessageCode.EMC_TARIFF_ERROR)
                    .put(ErrorCode.CITIZENSHIP_NOT_MATCH_DOCUMENT_TYPE, ETrainOrderAggregateErrorMessageCode.EMC_CITIZENSHIP_NOT_MATCH_DOCUMENT_TYPE)
                    .put(ErrorCode.INVALID_DOCUMENT_NUMBER, ETrainOrderAggregateErrorMessageCode.EMC_INVALID_DOCUMENT_NUMBER)
                    .put(ErrorCode.NAME_REQUIRED_LATIN_LETTERS, ETrainOrderAggregateErrorMessageCode.EMC_NAME_REQUIRED_LATIN_LETTERS)
                    .put(ErrorCode.NO_PLACES, ETrainOrderAggregateErrorMessageCode.EMC_NO_PLACES)
                    .put(ErrorCode.TOO_LATE, ETrainOrderAggregateErrorMessageCode.EMC_TOO_LATE)
                    .put(ErrorCode.TOO_LATE_FOR_ORDER, ETrainOrderAggregateErrorMessageCode.EMC_TOO_LATE_FOR_ORDER)
                    .put(ErrorCode.UNKNOWN_PARTNER_ERROR, ETrainOrderAggregateErrorMessageCode.EMC_UNKNOWN_PARTNER_ERROR)
                    .build();
    private static final Set<EPaymentState> ACTIVE_PAYMENT_STATES = Set.of(
            EPaymentState.PS_INVOICE_PENDING,
            EPaymentState.PS_PAYMENT_IN_PROGRESS,
            EPaymentState.PS_PARTIALLY_PAID);


    private final ObjectMapper objectMapper =
            new ObjectMapper().setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY);
    private final Clock clock;

    public TOrderAggregateState mapAggregateStateFromOrder(Order order) {
        switch (order.getPublicType()) {
            case OT_HOTEL_EXPEDIA:
                return mapHotelOrderAggregateState((HotelOrder) order);
            case OT_TRAIN:
                return mapTrainOrderAggregateState((TrainOrder) order);
            case OT_GENERIC:
                return mapGenericOrderAggregateState((GenericOrder) order);
            case OT_AVIA_AEROFLOT:
                return mapAeroflotOrderAggregateState((AeroflotOrder) order);
            default:
                throw new RuntimeException(
                        String.format("Don't know how to handle order of type: %s", order.getPublicType())
                );
        }
    }

    private void setDisplayState(Order order, TOrderAggregateState.Builder aggregateState) {
        aggregateState.setDisplayOrderState(EDisplayOrderStateMapper.fromOrder(order));
    }

    private void setReservedTo(Order order, TOrderAggregateState.Builder aggregateState) {
        if (order.getExpiresAt() != null) {
            aggregateState.setReservedTo(ProtoUtils.fromInstant(order.getExpiresAt()));
        }
    }

    private TOrderAggregateState mapHotelOrderAggregateState(HotelOrder order) {
        TOrderAggregateState.Builder result = TOrderAggregateState.newBuilder();
        setPaymentFields(order, result);
        setDisplayState(order, result);
        setReservedTo(order, result);
        result.setHotelOrderAggregateState(stateFromOrder(order));
        return result.build();
    }

    private void setPaymentFields(Order order, TOrderAggregateState.Builder aggregateState) {
        TrustInvoice currentInvoice = (TrustInvoice) order.getCurrentInvoice();
        if (currentInvoice != null) {
            aggregateState.setPaymentUrl(Strings.nullToEmpty(currentInvoice.getPaymentUrl()));
            if (!Strings.isNullOrEmpty(currentInvoice.getAuthorizationErrorCode())) {
                EPaymentErrorDisplayState paymentErrorDisplayState = PAYMENT_ERROR_STR_TO_ENUM.getOrDefault(
                        currentInvoice.getAuthorizationErrorCode(),
                        EPaymentErrorDisplayState.EPE_OTHER
                );
                aggregateState.setPaymentError(paymentErrorDisplayState);
            }
        }
    }

    private TOrderAggregateState mapGenericOrderAggregateState(GenericOrder order) {
        TOrderAggregateState.Builder result = TOrderAggregateState.newBuilder();
        setDisplayState(order, result);
        result.setGenericOrderAggregateState(getStatus(order.getState(), order.getWorkflow().getState(),
                order.getCurrentInvoice()));
        if (order.getStateContext() != null) {
            result.setVersionHash(String.valueOf(order.getStateContext().hashCode()));
        }

        // backward compatibility for happy page (simple orders only)
        if (order.getDisplayType() == EDisplayOrderType.DT_TRAIN && order.getOrderItems().size() == 1) {
            setPaymentFields(order, result);
            setReservedTo(order, result);
            TrainOrderItem orderItem = OrderCompatibilityUtils.getOnlyTrainOrderItem(order);
            result.setTrainAggregateExtInfo(getTrainExtInfo(order.getWorkflow().getState(), orderItem));
        }
        return result.build();
    }

    private TOrderAggregateState mapAeroflotOrderAggregateState(AeroflotOrder order) {
        TOrderAggregateState.Builder result = TOrderAggregateState.newBuilder();
        setDisplayState(order, result);
        return result.build();
    }

    private EGenericOrderAggregateState getStatus(EOrderState orderState, EWorkflowState orderWorkflowState,
                                                  Invoice currentInvoice) {
        switch (orderState) {
            case OS_NEW:
                return EGenericOrderAggregateState.GOAG_NEW;
            case OS_WAITING_RESERVATION:
                return EGenericOrderAggregateState.GOAG_WAITING_RESERVATION;
            case OS_RESERVED:
                return EGenericOrderAggregateState.GOAG_RESERVED;
            case OS_WAITING_PAYMENT:
                if (currentInvoice == null) {
                    return EGenericOrderAggregateState.GOAG_STARTING_PAYMENT;
                } else if (currentInvoice instanceof TrustInvoice) {
                    ETrustInvoiceState currentInvoiceState = ((TrustInvoice) currentInvoice).getInvoiceState();
                    if (currentInvoiceState == ETrustInvoiceState.IS_NEW) {
                        return EGenericOrderAggregateState.GOAG_STARTING_PAYMENT;
                    } else if (currentInvoiceState == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                        return EGenericOrderAggregateState.GOAG_PAYMENT_FAILED;
                    } else {
                        return EGenericOrderAggregateState.GOAG_WAITING_PAYMENT;
                    }
                } else {
                    return EGenericOrderAggregateState.GOAG_WAITING_PAYMENT;
                }
            case OS_WAITING_CONFIRMATION:
                return EGenericOrderAggregateState.GOAG_WAITING_CONFIRMATION;
            case OS_CONFIRMED:
                return EGenericOrderAggregateState.GOAG_CONFIRMED;
            case OS_CANCELLED:
                return EGenericOrderAggregateState.GOAG_CANCELLED;
            case OS_WAITING_CANCELLATION:
                return EGenericOrderAggregateState.GOAG_WAITING_CANCELLATION;
            case OS_REFUNDING:
                return EGenericOrderAggregateState.GOAG_WAITING_REFUND;
            case OS_REFUNDED:
                return EGenericOrderAggregateState.GOAG_REFUNDED;
        }
        return EGenericOrderAggregateState.GOAG_CANCELLED;
    }

    private TOrderAggregateState mapTrainOrderAggregateState(TrainOrder order) {

        TOrderAggregateState.Builder result = TOrderAggregateState.newBuilder();

        ETrainOrderState orderState = order.getState();
        TrainOrderItem orderItem = (TrainOrderItem) order.getOrderItems().get(0);
        EOrderItemState orderItemState = orderItem.getItemState();
        EWorkflowState workflowState = order.getWorkflow().getState();
        ETrustInvoiceState currentInvoiceState = ETrustInvoiceState.IS_UNKNOWN;
        if (order.getCurrentInvoice() != null) {
            currentInvoiceState = ((TrustInvoice) order.getCurrentInvoice()).getInvoiceState();
        }
        setPaymentFields(order, result);
        setDisplayState(order, result);
        setReservedTo(order, result);

        var isOrderExpired = false;
        if (order.getExpiresAt() != null) {
            isOrderExpired = order.getExpiresAt().isBefore(Instant.now(clock));
        }
        result.setTrainOrderAggregateState(
                getStatus(
                        orderState, orderItemState, workflowState, currentInvoiceState,
                        order.isUserActionScheduled(), isOrderExpired
                )
        );

        result.setTrainAggregateExtInfo(getTrainExtInfo(order.getWorkflow().getState(), orderItem));

        return result.build();

    }

    private ETrainOrderAggregateState getStatus(ETrainOrderState orderState, EOrderItemState orderItemState,
                                                EWorkflowState orderWorkflowState,
                                                ETrustInvoiceState currentInvoiceState,
                                                boolean userActionScheduled, boolean isOrderExpired) {
        if (orderWorkflowState == EWorkflowState.WS_CRASHED) {
            if (orderState == ETrainOrderState.OS_WAITING_CONFIRMATION) {
                return ETrainOrderAggregateState.TOAG_CONFIRMATION_FAILED;
            }
            return ETrainOrderAggregateState.TOAG_CANCELLED;
        }
        switch (orderState) {
            case OS_CANCELLED:
            case OS_WAITING_CANCELLATION:
                return ETrainOrderAggregateState.TOAG_CANCELLED;
            case OS_WAITING_PAYMENT:
                if (currentInvoiceState == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                    if (isOrderExpired) {
                        return ETrainOrderAggregateState.TOAG_CANCELLED;
                    }
                    return ETrainOrderAggregateState.TOAG_PAYMENT_FAILED;
                } else {
                    return ETrainOrderAggregateState.TOAG_WAITING_PAYMENT;
                }
            case OS_WAITING_CONFIRMATION:
                return ETrainOrderAggregateState.TOAG_WAITING_CONFIRMATION;
            case OS_CONFIRMED:
                if (orderItemState == EOrderItemState.IS_REFUNDING) {
                    return ETrainOrderAggregateState.TOAG_WAITING_REFUND;
                } else if (userActionScheduled) {
                    return ETrainOrderAggregateState.TOAG_IN_PROGRESS;
                }
            case OS_REFUNDED:
                return ETrainOrderAggregateState.TOAG_CONFIRMED;
            case OS_MANUAL_PROCESSING:
                return ETrainOrderAggregateState.TOAG_CONFIRMED;
            case OS_NEW:
            case OS_WAITING_RESERVATION:
                return ETrainOrderAggregateState.TOAG_WAITING_RESERVATION;
        }
        return ETrainOrderAggregateState.TOAG_CANCELLED;
    }

    private TTrainOrderAggregateStateExtInfo getTrainExtInfo(EWorkflowState orderWorkflowState, TrainOrderItem orderItem) {
        TTrainOrderAggregateStateExtInfo.Builder extInfo = TTrainOrderAggregateStateExtInfo.newBuilder();

        if (orderItem.getPayload().getMaxPendingTill() != null) {
            extInfo.setMaxPendingTill(ProtoUtils.fromInstant(orderItem.getPayload().getMaxPendingTill()));
        }
        if (orderItem.getPayload().getInsuranceStatus() != null) {
            extInfo.setInsuranceState(
                    INSURANCE_TO_PROTO_ENUM.getOrDefault(orderItem.getPayload().getInsuranceStatus(),
                            ETrainOrderInsuranceAggregateState.EIAG_UNKNOWN)
            );
        }

        TTrainOrderAggregateErrorInfo errorInfo = setTrainOrderErrorInfo(orderWorkflowState, orderItem.getPayload());
        if (errorInfo != null) {
            extInfo.setErrorInfo(errorInfo);
        }

        return extInfo.build();
    }

    private TTrainOrderAggregateErrorInfo setTrainOrderErrorInfo(EWorkflowState orderWorkflowState, TrainReservation payload) {
        if (orderWorkflowState == EWorkflowState.WS_CRASHED) {
            return TTrainOrderAggregateErrorInfo.newBuilder()
                    .setErrorType(ETrainOrderAggregateErrorType.ET_PROCESS_EXCEPTION_STATE)
                    .build();
        }
        if (payload.getErrorInfo() != null) {
            TTrainOrderAggregateErrorInfo.Builder builder = TTrainOrderAggregateErrorInfo.newBuilder();
            if (payload.getErrorInfo().getMessageParams() != null) {
                builder.addAllMessageParam(payload.getErrorInfo().getMessageParams());
            }
            builder.setMessage(Strings.nullToEmpty(payload.getErrorInfo().getMessage()));
            ErrorCode messageCode = payload.getErrorInfo().getCode();
            builder.setMessageCode(
                    ERROR_MESSAGE_CODE_TO_PROTO_ENUM.getOrDefault(
                            messageCode,
                            ETrainOrderAggregateErrorMessageCode.EMC_UNKNOWN
                    )
            );
            if (messageCode == ErrorCode.REBOOKING_FAILED) {
                builder.setErrorType(ETrainOrderAggregateErrorType.ET_REBOOKING_FAILED);
            } else {
                builder.setErrorType(ETrainOrderAggregateErrorType.ET_PARTNER_ERROR);
            }
            return builder.build();
        }
        return null;
    }

    private EHotelOrderAggregateState mapWaitingPayment(Collection<ETrustInvoiceState> invoiceStates,
                                                        boolean promoCodesEmptyOrSuccess, boolean retryPaymentAvailable,
                                                        ETrustInvoiceState currentInvoiceState) {
        if (retryPaymentAvailable) {
            if (currentInvoiceState == ETrustInvoiceState.IS_UNKNOWN) {
                if (promoCodesEmptyOrSuccess) {
                    return EHotelOrderAggregateState.HOAG_RESERVED;
                } else {
                    return EHotelOrderAggregateState.HOAG_RESERVED_WITH_RESTRICTIONS;
                }
            } else if (currentInvoiceState == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT) {
                return EHotelOrderAggregateState.HOAG_AWAITS_PAYMENT;
            } else if (currentInvoiceState == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                return EHotelOrderAggregateState.HOAG_PAYMENT_FAILED;
            } else {
                return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
            }
        } else {
            // we don't touch this logic to preserve existing behaviour
            if (invoiceStates.isEmpty()) {
                if (promoCodesEmptyOrSuccess) {
                    return EHotelOrderAggregateState.HOAG_RESERVED;
                } else {
                    return EHotelOrderAggregateState.HOAG_RESERVED_WITH_RESTRICTIONS;
                }
            } else if (invoiceStates.stream().anyMatch(is -> is == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT)) {
                return EHotelOrderAggregateState.HOAG_AWAITS_PAYMENT;
            } else {
                return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
            }
        }
    }

    private boolean isPromoCodesEmptyOrSuccess(Order order) {
        return order.getPromoCodeApplications().size() == 0 ||
                order.getPromoCodeApplications().stream()
                        .allMatch(a -> a.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS);
    }

    private EHotelOrderAggregateState stateFromOrder(HotelOrder order) {
        EWorkflowState workflowState = order.getWorkflow().getState();
        EHotelOrderState orderState = order.getState();

        boolean actionScheduled = order.isUserActionScheduled();
        ETrustInvoiceState currentInvoiceState = ETrustInvoiceState.IS_UNKNOWN;
        List<ETrustInvoiceState> invoiceStates = order.getInvoices().stream().map(
                i -> ((TrustInvoice) i).getState()
        ).collect(Collectors.toUnmodifiableList());
        if (order.getCurrentInvoice() != null) {
            currentInvoiceState = ((TrustInvoice) order.getCurrentInvoice()).getState();
        }
        boolean promoCodesEmptyOrSuccess = isPromoCodesEmptyOrSuccess(order);
        boolean retryPaymentAvailable = order.getPaymentRetryEnabled();

        //noinspection SwitchStatementWithTooFewBranches
        switch (workflowState) {
            case WS_CRASHED:
                switch (orderState) {
                    case OS_WAITING_INVOICE_REFUND:
                    case OS_WAITING_SERVICE_REFUND:
                    case OS_REFUNDED:
                        return EHotelOrderAggregateState.HOAG_REFUND_FAILED;
                    case OS_WAITING_PAYMENT:
                        return EHotelOrderAggregateState.HOAG_PAYMENT_FAILED;
                    default:
                        return EHotelOrderAggregateState.HOAG_FAILED;
                }
            default:
                if (actionScheduled) {
                    return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
                } else {
                    switch (orderState) {
                        case OS_CANCELLED:
                            if (invoiceStates.stream().anyMatch(PAID_INVOICE_STATES::contains)) {
                                return EHotelOrderAggregateState.HOAG_CANCELLED_WITH_REFUND;
                            } else {
                                if (invoiceStates.stream().anyMatch(s -> s == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)) {
                                    return EHotelOrderAggregateState.HOAG_PAYMENT_FAILED;
                                } else {
                                    return EHotelOrderAggregateState.HOAG_CANCELLED;
                                }
                            }
                        case OS_WAITING_PAYMENT:
                            return mapWaitingPayment(invoiceStates, promoCodesEmptyOrSuccess,
                                    retryPaymentAvailable, currentInvoiceState);
                        case OS_CONFIRMED:
                        case OS_WAITING_EXTRA_PAYMENT:
                            return mapConfirmedHotelOrderState(order);
                        case OS_REFUNDED:
                            return EHotelOrderAggregateState.HOAG_REFUNDED;
                        case OS_MANUAL_PROCESSING:
                            return EHotelOrderAggregateState.HOAG_FAILED;
                        default:
                            return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
                    }
                }
        }
    }


    private EHotelOrderAggregateState mapCurrentPaymentStateToConfirmedOrderState(Payment payment) {
        switch (payment.getState()) {
            case PS_INVOICE_PENDING:
                // We have a pending, but not started invoice
                if (payment.getLastAttempt() == null || Strings.isNullOrEmpty(payment.getLastAttempt().getAuthorizationErrorCode())) {
                    // No active errors - the invoice was never started
                    return EHotelOrderAggregateState.HOAG_CONFIRMED;
                } else {
                    // There is an active error - the invoice was started and failed on payment
                    return EHotelOrderAggregateState.HOAG_PAYMENT_FAILED;
                }
            case PS_PAYMENT_IN_PROGRESS:
                // we have a pending and started invoice:
                if (Strings.isNullOrEmpty(payment.getLastAttempt().getPaymentUrl()) && Strings.isNullOrEmpty(payment.getLastAttempt().getAuthorizationErrorCode())) {
                    //  no errors and no active payment url: we are in progress
                    return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
                } else if (!Strings.isNullOrEmpty(payment.getLastAttempt().getPaymentUrl())) {
                    // Started invoice has a trust basket waiting for payment
                    return EHotelOrderAggregateState.HOAG_AWAITS_PAYMENT;
                } else {
                    // errors on last payment
                    return EHotelOrderAggregateState.HOAG_PAYMENT_FAILED;
                }
            case PS_PARTIALLY_PAID:
                // we have a partially paid payment schedule, so we need to look at its "next" payments
                Optional<Payment> nextPayment = payment.getNextPayments().stream()
                        .filter(pi -> ACTIVE_PAYMENT_STATES.contains(pi.getState()))
                        .findFirst();
                if (nextPayment.isEmpty()) {
                    // next payment is not started yet - we are in progress
                    return EHotelOrderAggregateState.HOAG_IN_PROGRESS;
                } else {
                    // recursively call mapper for the next payment
                    return mapCurrentPaymentStateToConfirmedOrderState(nextPayment.get());
                }
            default:
                // we should not be here, as this switch covers all possible values of CURRENT_PAYMENT_STATES
                throw new IllegalStateException();
        }
    }

    private EHotelOrderAggregateState mapConfirmedHotelOrderState(HotelOrder order) {
        var currentPayment = order.getPayments().stream()
                .filter(p -> ACTIVE_PAYMENT_STATES.contains(p.getState()))
                .findFirst();
        if (currentPayment.isEmpty()) {
            // no current pending invoice: we have a fully paid confirmed order
            return EHotelOrderAggregateState.HOAG_CONFIRMED;
        } else {
            // We have a current pending invoice, started or not
            Payment payment = currentPayment.get();
            return mapCurrentPaymentStateToConfirmedOrderState(payment);
        }
    }
}
